from typing import List, Union, Tuple, Optional, Iterable

from PySide2.QtCore import QPointF
from PySide2.QtGui import QTransform
from axipy.cpp_core_geometry import (
    ShadowGeometry, ShadowPoint, ShadowLine, ShadowLineString, ShadowPointsLS, ShadowPolygon,
    ShadowCollection, ShadowMultiPoint, ShadowMultiLineString, ShadowMultiPolygon
)
# TODO: rename cpp name
from axipy.cpp_core_geometry import Type as GeometryType

from axipy.cs import CoordSystem, LinearUnit, AreaUnit, LinearUnits
from axipy.utl import Pnt, Rect
from .._internal._decorator import _deprecated_by

# backwards compatibility
Type = GeometryType

__all__ = [
    "Geometry",
    "Point",
    "Line",
    "ListPoint",
    "LineString",
    "ListHoles",
    "Polygon",
    "GeometryCollection",
    "MultiPoint",
    "MultiLineString",
    "MultiPolygon",
    "GeometryType",
    # backwards compatibility
    "Type"
]


class Geometry:
    """
    Абстрактный класс геометрического объекта (геометрии).
    
    Note:
        Для получения краткого текстового представления геометрии можно воспользоваться функцией :class:`str`.
        Для более полного - :func:`repr`.
    """

    def __init__(self):
        raise NotImplementedError

    def _set_shadow(self, shadow: ShadowGeometry):
        self._shadow = shadow

    @classmethod
    def _wrap_typed(cls, shadow):
        obj = cls.__new__(cls)
        obj._shadow = shadow
        return obj

    def get_length(self, u: LinearUnit = None) -> float:
        """Рассчитывает длину геометрии. Метод применим только для линейных объектов. В противном случае возвращает 0.
        В случае, если СК задана как Широта/Долгота, то расчет производится на сфере в метрах.

        .. literalinclude:: /../../tests/doc_examples/da/test_example_geometry.py
            :caption: Пример.
            :pyobject: test_run_example_geometry_length
            :lines: 2-
            :dedent: 4

        Args:
            u: Единица измерения, в которой необходимо получить результат.
                Если не задана, то используется единица измерения для СК.
                Если и она не задана, то производится расчет на плоскости.
        """
        if u is not None:
            if not isinstance(u, LinearUnit):
                raise ValueError("Unit must be of LinearUnit type.")
            if self.coordsystem is None:
                raise RuntimeWarning('CoordSystem is not defined')
        return self._shadow.length(u._shadow if u is not None else None)

    def get_perimeter(self, u: LinearUnit = None) -> float:
        """
        Рассчитывает периметр геометрии. Метод применим только для площадных объектов.
        В противном случае возвращает 0.
        В случае, если СК задана как Широта/Долгота, то расчет производится на сфере в метрах.

        Пример см. :meth:`get_area`

        Args:
            u: Единица измерения, в которой необходимо получить результат.
                Если не задана, то используется единица измерения для СК.
                Если и она не задана, то производится расчет на плоскости.
        """
        if u is not None:
            if not isinstance(u, LinearUnit):
                raise ValueError("Unit must be of LinearUnit type.")
            if self.coordsystem is None:
                raise RuntimeWarning('CoordSystem is not defined')
        return self._shadow.perimeter(u._shadow if u is not None else None)

    def get_area(self, area_unit: AreaUnit = None) -> float:
        """
        Рассчитывает площадь, если объект площадной. В противном случае возвращает 0.
        В случае, если СК задана как Широта/Долгота, то расчет производится на сфере в квадратных метрах.

        .. literalinclude:: /../../tests/doc_examples/da/test_example_geometry.py
            :caption: Пример.
            :pyobject: test_run_example_geometry_area
            :lines: 2-
            :dedent: 4

        Args:
            area_unit: Единица измерения, в которой необходимо получить результат.
                Если не задана, то используется единица измерения для СК.
                Если и она не задана, то производится расчет на плоскости.
        """

        if area_unit is None:
            if self.coordsystem is None:
                return self._shadow.area()
            else:
                linear_unit = self.coordsystem.unit
                if linear_unit == LinearUnits.degree:
                    return self._shadow.area()
                else:
                    area_unit = AreaUnit.from_linear_unit(linear_unit)

        if not isinstance(area_unit, AreaUnit):
            raise TypeError("Unit type mismatch, needs AreaUnit")
        elif self.coordsystem is None:
            raise RuntimeWarning('CoordSystem is not defined')
        else:
            return self._shadow.area(area_unit._shadow)

    @property
    def bounds(self) -> Rect:
        """Возвращает минимальный ограничивающий прямоугольник."""
        return Rect.from_qt(self._shadow.bounds())

    @property
    def type(self) -> GeometryType:
        """Возвращает тип геометрического элемента.

        .. csv-table:: Возможные значения
            :header: Значение, Наименование
            :align: left

            `Unknown`, Не определен
            `Point`, Точка
            `Line`, Линия
            `LineString`, Полилиния
            `Polygon`, Полигон
            `MultiPoint`, Коллекция точек
            `MultiLineString`, Коллекция полилиний
            `MultiPolygon`, Коллекция полигонов
            `GeometryCollection`, Смешанная коллекция
            `Arc`, Дуга
            `Ellipse`, Эллипс
            `Rectangle`, Прямоугольник
            `RoundedRectangle`, Скругленный прямоугольник
            `Text`, Текст

        .. literalinclude:: /../../tests/doc_examples/da/test_example_geometry.py
            :caption: Пример.
            :pyobject: test_run_example_geometry_type
            :lines: 2-
            :dedent: 4
        """
        return self._shadow.type()

    @property
    def name(self) -> str:
        """Возвращает наименование геометрического объекта."""
        return self._shadow.name()

    def get_distance(self, other: 'Geometry', u: LinearUnit = None) -> float:
        """Производит расчет расстояния до объекта `other`. Результат возвращает в СК текущего объекта.

        Args:
            other: Анализируемый объект.
            u: Единицы измерения, в которых требуется получить результат.

        .. literalinclude:: /../../tests/doc_examples/da/test_example_geometry.py
            :caption: Пример.
            :pyobject: test_run_example_geometry_distance
            :lines: 2-
            :dedent: 4

        """
        if u is not None:
            if not isinstance(u, LinearUnit):
                raise ValueError("Unit must be of LinearUnit type.")
            if self.coordsystem is None:
                raise RuntimeWarning('CoordSystem is not defined')
        return self._shadow.distance(other._shadow, u._shadow if u is not None else None)

    @staticmethod
    def from_wkt(wkt: str, coordsystem: CoordSystem = None) -> 'Geometry':
        """
        Создает геометрический объект из строки формата
        `WKT <https://en.wikipedia.org/wiki/Well-known_text_representation_of_geometry>`_.

        Args:
            wkt: Строка WKT. Допустимо задание строки в формате с указанием SRID (EWKT).
                В данном случае система координат
                для создаваемой геометрии будет установлено исходя их этого значения.
            coordsystem: Система координат, которая будет установлена для геометрии.
                Если строка задана в виде EWKT и указано значение SRID, игнорируется.

        .. literalinclude:: /../../tests/doc_examples/da/test_example_geometry.py
            :caption: Пример.
            :pyobject: test_run_example_geometry_wkt
            :lines: 2-
            :dedent: 4
        """
        return Geometry._wrap(ShadowGeometry.from_wkt(wkt, coordsystem._shadow if coordsystem is not None else None))

    @property
    @_deprecated_by("axipy.Geometry.to_wkt")
    def wkt(self) -> str:
        return self._shadow.to_wkt()
    
    def to_wkt(self) -> str:
        """Возвращает WKT строку для геометрии."""
        return self._shadow.to_wkt()

    @staticmethod
    def from_wkb(wkb: bytes, coordsystem: CoordSystem = None) -> 'Geometry':
        """
        Создает геометрический объект из строки формата
        `WKB <https://en.wikipedia.org/wiki/Well-known_text_representation_of_geometry>`_.

        Args:
            wkb: Строка WKB
            coordsystem: Система координат, которая будет установлена для геометрии. 

        .. literalinclude:: /../../tests/doc_examples/da/test_example_geometry.py
            :caption: Пример.
            :pyobject: test_run_example_geometry_wkb
            :lines: 2-
            :dedent: 4

        """
        return Geometry._wrap(ShadowGeometry.from_wkb(wkb, coordsystem._shadow if coordsystem is not None else None))

    @property
    @_deprecated_by("axipy.Geometry.to_wkb")
    def wkb(self) -> bytes:
        return bytes(self._shadow.to_wkb())
    
    def to_wkb(self) -> bytes:
        """Возвращает WKB строку для геометрии."""
        return bytes(self._shadow.to_wkb())

    @property
    def coordsystem(self) -> CoordSystem:
        """Система Координат (СК) геометрии."""
        return CoordSystem._wrap(self._shadow.cs())

    @coordsystem.setter
    def coordsystem(self, cs: CoordSystem):
        self._shadow.set_cs(cs._shadow if cs is not None else None)

    # Relations
    def equals(self, other: 'Geometry') -> bool:
        """Производит сравнение с другой геометрией.

        Args:
            other: Сравниваемая геометрия.

        Returns:
            Возвращает True, если геометрии равны.
        """
        return self._shadow.equals(other._shadow)

    def __eq__(self, other) -> bool:
        return other is not None and self._shadow.equals(other._shadow)

    def almost_equals(self, other: 'Geometry', tolerance: float) -> bool:
        """Производит примерное сравнения с другой геометрией в пределах заданной точности.

        Args:
            other: Сравниваемый объект.
            tolerance: Точность сравнения.

        Returns:
            Возвращает True, если геометрии равны в пределах заданного отклонения.
        """
        return self._shadow.almost_equals(other._shadow, tolerance)

    def contains(self, other: 'Geometry') -> bool:
        """Возвращает True, если геометрия полностью содержит передаваемую в качестве параметра геометрию."""
        return self._shadow.contains(other._shadow)

    def crosses(self, other: 'Geometry') -> bool:
        """Возвращает True, если при пересечении геометрий объекты частично пересекаются."""
        return self._shadow.crosses(other._shadow)

    def disjoint(self, other: 'Geometry') -> bool:
        """Возвращает True, если геометрии не пересекаются и не соприкасаются."""
        return self._shadow.disjoint(other._shadow)

    def intersects(self, other: 'Geometry') -> bool:
        """Возвращает True, если геометрии пересекаются."""
        return self._shadow.intersects(other._shadow)

    def overlaps(self, other: 'Geometry') -> bool:
        """Возвращает True, если пересечение геометрий отличается от обеих геометрий."""
        return self._shadow.overlaps(other._shadow)

    def touches(self, other: 'Geometry') -> bool:
        """Возвращает True, если геометрии соприкасаются."""
        return self._shadow.touches(other._shadow)

    def within(self, other: 'Geometry') -> bool:
        """Возвращает True, если геометрия находится полностью внутри геометрии `other`."""
        return self._shadow.within(other._shadow)

    def covers(self, other: 'Geometry') -> bool:
        """Возвращает True, если геометрия охватывает геометрию `other`."""
        return self._shadow.covers(other._shadow)

    # de-9im
    def relate(self, other: 'Geometry') -> str:
        """Проверяет отношения между объектами. Подробнее см. `DE-9IM <https://en.wikipedia.org/wiki/DE-9IM>`_
        """
        return self._shadow.relate(other._shadow)

    # Analysis
    def boundary(self) -> 'Geometry':
        """Возвращает границы геометрии в виде полилинии."""
        return Geometry._wrap(ShadowGeometry.boundary(self._shadow))

    def centroid(self) -> 'Geometry':
        """Возвращает центроид геометрии."""
        return Geometry._wrap(ShadowGeometry.centroid(self._shadow))

    def difference(self, other: 'Geometry') -> 'Geometry':
        """Возвращает область первой геометрии, которая не пересечена второй геометрией."""
        return Geometry._wrap(ShadowGeometry.difference(self._shadow, other._shadow))

    def intersection(self, other: 'Geometry') -> 'Geometry':
        """Возвращает область пересечения с другой геометрией."""
        return Geometry._wrap(ShadowGeometry.intersection(self._shadow, other._shadow))

    def symmetric_difference(self, other: 'Geometry') -> 'Geometry':
        """Возвращает логический XOR областей геометрий (объединение разниц).

        Args:
            other: Геометрия для анализа.
        """
        return Geometry._wrap(ShadowGeometry.symmetric_difference(self._shadow, other._shadow))

    def union(self, other: 'Geometry') -> 'Geometry':
        """Возвращает результат объединения двух геометрий."""
        return Geometry._wrap(ShadowGeometry._union(self._shadow, other._shadow))

    def buffer(self, distance: float, resolution: int = 16, capStyle: int = 1, joinStyle: int = 1,
               mitreLimit: float = 5.0) -> 'Geometry':
        """Производит построение буфера.

        Args:
            distance: Ширина буфера.
            resolution: Количество сегментов на квадрант.
            capStyle: Стиль окончания.
            joinStyle: Стиль соединения.
            mitreLimit: Предел среза.
        """
        return Geometry._wrap(
            ShadowGeometry.buffer(self._shadow, distance, resolution, capStyle, joinStyle, mitreLimit))

    def convex_hull(self) -> 'Geometry':
        """Возвращает минимальный окаймляющий полигон со всеми выпуклыми углами."""
        return Geometry._wrap(ShadowGeometry.convex_hull(self._shadow))

    def envelope(self) -> 'Geometry':
        """Возвращает полигон, описывающий заданную геометрию."""
        return Geometry._wrap(ShadowGeometry.envelope(self._shadow))

    # Transform
    def affine_transform(self, trans: QTransform) -> 'Geometry':
        """Трансформирует объект исходя из заданных параметров трансформации.

        See also:
            Для простых операций типа сдвиг, масштабирование  и поворот,
            рекомендуется использовать :meth:`shift`, :meth:`scale`, :meth:`rotate` соответственно.

        Args:
            trans: Матрица трансформации.
        """
        return Geometry._wrap(ShadowGeometry.affine_transform(self._shadow, trans))

    def shift(self, dx: float, dy: float) -> 'Geometry':
        """Смещает объект на заданную величину. Значения задаются в текущей проекции.

        Args:
            dx: Сдвиг по координате X.
            dy: Сдвиг по координате Y.
        """
        return self.affine_transform(QTransform.fromTranslate(dx, dy))

    def scale(self, kx: float, ky: float) -> 'Geometry':
        """Масштабирует объект по заданным коэффициентам масштабирования. После выполнения операции центр
        результирующего объекта остается прежним.

        Args:
            kx: Масштабирование по координате X.
            ky: Масштабирование по координате Y.
        """
        c = self.bounds.center
        trShift = QTransform.fromTranslate(-c.x, -c.y)
        trScale = QTransform.fromScale(kx, ky)
        tr = trShift * trScale * trShift.inverted()[0]
        return self.affine_transform(tr)

    @classmethod
    def _wrap(cls, _shadow):
        if _shadow is None:
            return None
        from axipy.mi import Rectangle, RoundRectangle, Ellipse, Arc, Text
        typesGeometry = {
            'Point': Point,
            'Line': Line,
            'LineString': LineString,
            'Polygon': Polygon,
            'GeometryCollection': GeometryCollection,
            'MultiPoint': MultiPoint,
            'MultiLineString': MultiLineString,
            'MultiPolygon': MultiPolygon,
            'Rectangle': Rectangle,
            'RoundRectangle': RoundRectangle,
            'Ellipse': Ellipse,
            'Arc': Arc,
            'Text': Text,
        }
        T = typesGeometry.get(_shadow.id(), Geometry)
        return T._wrap_typed(_shadow)

    def reproject(self, cs: CoordSystem) -> 'Geometry':
        """Перепроецирует геометрию в другую систему координат.

        Args:
            cs: СК, в которой требуется получить объект.
        """
        return Geometry._wrap(ShadowGeometry.reproject(self._shadow, cs._shadow))

    def rotate(self, point: Union[Pnt, Tuple[float, float]], angle: float) -> 'Geometry':
        """Поворот геометрии относительно заданной точки. Поворот производится против часовой стрелки.

        Args:
            point: Точка, вокруг которой необходимо произвести поворот.
            angle: Угол поворота в градусах.
        """
        return Geometry._wrap(ShadowGeometry.rotate(self._shadow, Pnt._fixPnt(point).to_qt(), angle))

    @property
    def is_valid(self) -> bool:
        """Проверяет геометрию на валидность."""
        return self._shadow.is_valid()

    @property
    def is_valid_reason(self) -> str:
        """Если геометрия неправильная, возвращает краткую аннотацию причины."""
        return self._shadow.is_valid_reason()

    def normalize(self) -> 'Geometry':
        return Geometry._wrap(ShadowGeometry.normalize(self._shadow))

    def clone(self) -> 'Geometry':
        """Создает копию объекта."""
        return Geometry._wrap(ShadowGeometry.clone(self._shadow))

    @staticmethod
    def point_by_azimuth(point: Union[Pnt, Tuple[float, float]], azimuth: float, distance: float,
                         cs: CoordSystem = None) -> Pnt:
        """
        Производит расчет координат точки относительно заданной,
        и находящейся на расстоянии `distance` по направлению `azimuth`.

        .. seealso::
            Обратная функция :meth:`distance_by_points`

        Args:
            point: Точка, относительно которой производится расчет. Если задана СК, то в ней.
            azimuth: Азимут в градусах, указывающий направление.
            distance: Расстояние по азимуту. Задается в метрах.
            cs: СК, на базе эллипсоида которой производится расчет. Если не задана, то расчет производится на плоскости.

        Returns:
            Возвращает результирующую точку.
        """
        return Pnt.from_qt(ShadowGeometry.point_by_azimuth(Pnt._fixPnt(point).to_qt(), azimuth, distance,
                                                           None if cs is None else cs._shadow))

    @staticmethod
    def distance_by_points(start: Union[Pnt, Tuple[float, float]], end: Union[Pnt, Tuple[float, float]],
                           cs: CoordSystem = None) -> Tuple:
        """Производит расчет расстояния между двумя точками и азимут от первой до второй точки.

        .. seealso::
            Обратная функция :meth:`point_by_azimuth`

        Args:
            start: Начальная точка. Если задана СК, то координаты в ней.
            end: Конечная точка. Если задана СК, то координаты в ней.
            cs: СК, на базе эллипсоида которой производится расчет. Если не задана, то расчет производится на плоскости.

        Returns:
            Возвращается пара значений (расстояние в метрах, азимут в градусах)
        """
        return ShadowGeometry.distance_by_points(Pnt._fixPnt(start).to_qt(), Pnt._fixPnt(end).to_qt(),
                                                 None if cs is None else cs._shadow)

    def __repr__(self):
        """Представление геометрии в виде строки."""
        return self.to_wkt() + self._cs_str

    @property
    def _cs_str(self) -> str:
        return '; ' + str(self.coordsystem) if self.coordsystem else ''

    @property
    def _class_name(self):
        return self.__class__.__name__

    def to_linestring(self) -> Optional['Geometry']:
        """Пробует геометрию преобразовать в линейный объект. В случае неудачи возвращает None."""
        return Geometry._wrap(ShadowGeometry.convertToMultiLineString(self._shadow))

    def to_polygon(self) -> Optional['Geometry']:
        """Пробует геометрию преобразовать в площадной объект. В случае неудачи возвращает None."""
        return Geometry._wrap(ShadowGeometry.convertFromLineStringsToPolygons(self._shadow))

    def to_geojson(self) -> str:
        """Преобразует геометрию в формат 'GeoJSON'"""
        return ShadowGeometry.convertGeometryToJson(self._shadow)

    @staticmethod
    def from_geojson(json: str, cs: Optional[CoordSystem] = None) -> 'Geometry':
        """Возвращает геометрию из ее 'GeoJSON' представления.
        
        Args:
            json: json представление в виде строки.
            cs: Система координат.
        """
        return Geometry._wrap(ShadowGeometry.geometryFromJson(json, None if cs is None else cs._shadow))
    
    def to_mif(self) -> str:
        """
        Преобразует геометрию в формат `MIF`.
        """
        return ShadowGeometry.geometryToMif(self._shadow)
    
    @staticmethod
    def from_mif(mif: str) -> 'Geometry':
        """
        Возвращает геометрию из ее 'MIF' представления.
        
        Args:
            mif: mif представление в виде строки.
        """
        return Geometry._wrap(ShadowGeometry.geometryFromMif(mif))


class Point(Geometry):
    """
    Геометрический объект типа точка.

    Args:
        x: X координата
        y: Y координата
        cs: Система Координат, в которой создается геометрия.

    .. literalinclude:: /../../tests/doc_examples/da/test_example_geometry.py
        :caption: Пример.
        :pyobject: test_run_example_geometry_point
        :lines: 2-
        :dedent: 4
   """

    def __init__(self, x: float, y: float, cs: CoordSystem = None):
        self._set_shadow(ShadowPoint(x, y, None if cs is None else cs._shadow))

    @property
    def x(self) -> float:
        """X Координата."""
        return self._shadow.get_pos().x()

    @x.setter
    def x(self, _x):
        self._shadow.set_pos(QPointF(_x, self.y))

    @property
    def y(self) -> float:
        """Y Координата."""
        return self._shadow.get_pos().y()

    @y.setter
    def y(self, _y):
        self._shadow.set_pos(QPointF(self.x, _y))

    def __eq__(self, other) -> bool:
        if other is None:
            return False
        ob = other
        if isinstance(other, tuple) and len(other) == 2:
            ob = Point(other[0], other[1])  # Без СК!!!
        return self._shadow.equals(ob._shadow)

    def __str__(self):
        return '{} pos={}'.format(self._class_name, Pnt.from_qt(self._shadow.get_pos())) + self._cs_str


class Line(Geometry):
    """
    Геометрический объект типа линия.

    Args:
        begin: Начальная точка линии.
        end: Конечная точка линии.
        cs: Система Координат, в которой создается геометрия.

    .. literalinclude:: /../../tests/doc_examples/da/test_example_geometry.py
        :caption: Пример.
        :pyobject: test_run_example_geometry_line
        :lines: 2-
        :dedent: 4
    """

    def __init__(self, begin: Pnt, end: Pnt, cs: CoordSystem = None):
        self._set_shadow(ShadowLine(Pnt._fixPnt(begin).to_qt(), Pnt._fixPnt(
            end).to_qt(), None if cs is None else cs._shadow))

    @property
    def begin(self) -> Pnt:
        """
        Начальная точка линии.
        Точки допустимо создавать как экземпляр :class:`Pnt` либо в виде пары `float` значений tuple
        """
        return Pnt.from_qt(self._shadow.get_begin())

    @begin.setter
    def begin(self, p):
        self._shadow.set_begin(Pnt._fixPnt(p).to_qt())

    @property
    def end(self) -> Pnt:
        """Конечная точка линии."""
        return Pnt.from_qt(self._shadow.get_end())

    @end.setter
    def end(self, p):
        self._shadow.set_end(Pnt._fixPnt(p).to_qt())

    def __repr__(self):
        return '{} begin={} end={}'.format(self._class_name, self.begin, self.end) + self._cs_str


class ListPoint:

    def __init__(self, ls):
        self.m = ShadowPointsLS(ls)

    def __len__(self):
        return self.m.count()

    def __getitem__(self, idx):
        if idx >= self.m.count() or idx < 0:
            raise IndexError()
        return Pnt.from_qt(self.m.at(idx))

    def __setitem__(self, idx, p):
        self.m.set(idx, Pnt._fixPnt(p).to_qt())

    def insert(self, idx, p):
        self.m.insert(idx, Pnt._fixPnt(p).to_qt())

    def append(self, p):
        self.m.append(Pnt._fixPnt(p).to_qt())

    def remove(self, idx):
        self.m.remove(idx)

    def pop(self, idx):
        self.m.remove(idx)

    def __str__(self):
        return ''.join(str(e) for e in self)


# Different kind of input


def _fix_points_as_param(points) -> list:
    # list of tuples
    if len(points) > 1 and isinstance(points[0], (tuple, Pnt)):
        return [a for a in points]
    # simple list of points
    elif len(points) == 1 and isinstance(points[0], list):
        return points[0]
    # iterator
    elif len(points) == 1 and hasattr(points[0], '__iter__') and hasattr(points[0], '__next__'):
        return list(points[0])
    # one point
    elif len(points) == 2 and isinstance(points[0], (int, float)) and isinstance(points[0], (int, float)):
        return [(points[0], points[1])]
    elif isinstance(points, list):
        return points
    # ListPoint as first element of tuple
    elif len(points) == 1 and isinstance(*points, ListPoint):
        return [p for p in points[0]]
    else:
        raise ValueError('Unexected parameter:{}'.format(type(points)))


def _fix_points_and_cs(points, cs) -> tuple:
    points_ = None
    cs_ = None
    if cs is None and isinstance(points, tuple):
        if len(points) == 2 and isinstance(points[1], CoordSystem):
            points_ = points[0]
            cs_ = points[1]
        elif len(points) > 0 and isinstance(points[-1], CoordSystem):
            points_ = points[:-1]
            cs_ = points[-1]
    if points_ is None:
        points_ = points
        cs_ = cs
    return points_, cs_


class LineString(Geometry):
    """
    Геометрический объект типа полилиния.

    Args:
        points: Список точек.
            Может задаваться следующим образом:

            * В виде списка :class:`list` из пар :class:`tuple`.
            * В виде перечня точек. В данном случае, если необходимо задать СК,
              то требуется явно указать наименование параметра.
            * В виде итератора по элементам, состоящих из пар :class:`tuple`.

        cs: Система Координат, в которой создается геометрия.

    .. literalinclude:: /../../tests/doc_examples/da/test_example_geometry.py
        :caption: Пример.
        :pyobject: test_run_example_geometry_ls
        :lines: 2-
        :dedent: 4
    """

    def __init__(
            self,
            *points: Union[
                Pnt,
                Tuple[float, float],
                Iterable[Pnt],
                Iterable[Tuple[float, float]],
            ],
            cs: CoordSystem = None,
    ):
        pars = _fix_points_and_cs(points, cs)
        self._set_shadow(ShadowLineString(
            [Pnt._fixPnt(p).to_qt() for p in _fix_points_as_param(pars[0])],
            None if pars[1] is None else pars[1]._shadow,
        ))
        self._points = None

    @classmethod
    def _wrap_typed(cls, _shadow_: ShadowLineString):
        obj = cls.__new__(cls)
        obj._shadow = _shadow_
        obj._points = None
        return obj

    @property
    def points(self) -> List[Pnt]:
        """
        Точки полилинии. Реализован как список python :class:`list` точек :class:`Pnt`.
        Также поддерживаются список пар :class:`tuple`.

        Note:
            При обновление значения точки допустимо только изменение ее заменой.

        """
        if self._points is None:
            self._points = ListPoint(self._shadow)
        return self._points

    def __len__(self):
        return len(self.points)

    def __str__(self):
        return '{} (points={})'.format(self._class_name, self.__len__()) + self._cs_str


class ListHoles:

    def __init__(self, poly: ShadowPolygon):
        self.m = poly

    def __len__(self):
        return self.m.interiorCount()

    def __getitem__(self, idx):
        if idx >= self.m.interiorCount() or idx < 0:
            raise IndexError()
        return ListPoint(self.m.interior(idx))

    def append(
            self,
            *points: Union[
                Pnt,
                Tuple[float, float],
                Iterable[Pnt],
                Iterable[Tuple[float, float]],
            ],
    ):
        self.m.addInterior([Pnt._fixPnt(p).to_qt()
                            for p in _fix_points_as_param(points)])

    def remove(self, idx):
        self.m.removeInterior(idx)


class Polygon(Geometry):
    """
    Геометрический объект типа полигон. Представляет собой часть плоскости, ограниченной замкнутой
    полилинией. Кроме внешней границы, полигон может иметь одну или несколько внутренних (дырок).

    Args:
        points: Список точек внешнего контура. Может задаваться следующим образом:

            * В виде списка :class:`list` из пар :class:`tuple`.
            * В виде перечня точек. В данном случае, если необходимо задать СК,
              то требуется явно указать наименование параметра.
            * В виде итератора по элементам, состоящих из пар :class:`tuple`.

        cs: Система Координат, в которой создается геометрия.

    .. literalinclude:: /../../tests/doc_examples/da/test_example_geometry.py
        :caption: Пример.
        :pyobject: test_run_example_geometry_poly
        :lines: 2-
        :dedent: 4
    """

    def __init__(
            self,
            *points: Union[
                Pnt,
                Tuple[float, float],
                Iterable[Pnt],
                Iterable[Tuple[float, float]],
            ],
            cs: CoordSystem = None
    ):
        pars = _fix_points_and_cs(points, cs)
        self._set_shadow(ShadowPolygon([Pnt._point_value_to_qt(p) for p in _fix_points_as_param(
            pars[0])], None if pars[1] is None else pars[1]._shadow))
        self._points = None
        self._holes = None

    @classmethod
    def _wrap_typed(cls, _shadow_: ShadowPolygon):
        obj = cls.__new__(cls)
        obj._shadow = _shadow_
        obj._points = None
        obj._holes = None
        return obj

    @staticmethod
    def from_rect(rect: Rect, cs: CoordSystem = None) -> 'Polygon':
        """Создает полигон на базе прямоугольника.

        Args:
            rect: Прямоугольник, на основе которого формируются координаты.
            cs: Система Координат, в которой создается геометрия.
        """

        points = [(rect.xmin, rect.ymin), (rect.xmin, rect.ymax), (rect.xmax, rect.ymax), (rect.xmax, rect.ymin)]
        return Polygon(points, cs=cs)

    @property
    def points(self) -> List[Pnt]:
        """
        Точки полигона.
        Реализован как список python :class:`list` точек :class:`Pnt`.
        В каждом контуре, количество точек на 1 больше, чем количество узлов, так как контур - это замкнутая линия,
        и первая и последняя точка совпадают.
        """
        if self._points is None:
            self._points = ListPoint(self._shadow.exterior())
        return self._points

    @property
    def holes(self) -> List[Pnt]:
        """Дырки полигона. Реализован в виде списка :class:`list`.

        .. literalinclude:: /../../tests/doc_examples/da/test_example_geometry.py
            :caption: Пример.
            :pyobject: test_run_example_geometry_poly_holes
            :lines: 2-
            :dedent: 4
        """
        if self._holes is None:
            self._holes = ListHoles(self._shadow)
        return self._holes

    def __len__(self):
        return len(self.points)

    def __str__(self):
        return '{} (points={}, holes={})'.format(self._class_name, self.__len__(),
                                                 len(self._holes) if self._holes else 0) + self._cs_str


class GeometryCollection(Geometry):
    """
    Коллекция разнотипных геометрических объектов. Допустимо хранение геометрических объектов различного типа,
    за исключением коллекций.
    Доступ к элементам производится по аналогии работы со списком :class:`list`.

    Args:
        cs: Система Координат, в которой создается геометрия.


    Для получения размера коллекции используйте функцию len::

        cnt = len(coll)

    Доступ к элементам производится по индексу. Нумерация начинается с 0.

    В качестве примера получим первый элемент::

        coll[1]

    Обновление геометрии в коллекции так же производится по ее индексу.
    Также допустимо изменение некоторых свойств геометрии в зависимости от ее типа.

    .. literalinclude:: /../../tests/doc_examples/da/test_example_geometry.py
        :pyobject: test_run_example_geometry_coll
        :lines: 9-
        :dedent: 4
    """

    def __init__(self, cs: CoordSystem = None):
        self._set_shadow(ShadowCollection(None if cs is None else cs._shadow))

    def __len__(self) -> int:
        return self._shadow.count()

    def _fix_tuple(self, t: List) -> Geometry:  # virtual
        if len(t) == 1:
            return Point(t[0][0], t[0][1], cs=self.coordsystem)
        return LineString(t, cs=self.coordsystem)

    def __fix_geometry(self, g) -> Geometry:
        if len(g) == 1:
            gg = g[0]
        else:
            gg = self._fix_tuple(_fix_points_as_param(g))

        if isinstance(gg, GeometryCollection):
            raise TypeError('Allow only simple Geometry type')
        if isinstance(gg, Geometry):
            return gg
        if isinstance(gg, tuple) and len(gg) == 2:
            return Point(gg[0], gg[1], self.coordsystem)
        if isinstance(gg, list) and len(gg) >= 2:
            return self._fix_tuple(gg)
        if isinstance(gg, Pnt):
            return Point(gg.x, gg.y, self.coordsystem)
        raise TypeError('Allow only Geometry type')

    def append(self, *value: Union[Geometry, Pnt, Tuple[float, float]]):
        """
        Добавление геометрии в коллекцию. Задание параметров аналогично указанию их
        при создании объектов конкретного типа.
        Если опустить указание класса, то для одной точки или пары значений float
        будет создан точечный объект :class:`Point`,
        если точек больше, - объект класса :class:`LineString`.

        .. literalinclude:: /../../tests/doc_examples/da/test_example_geometry.py
            :caption: Пример.
            :pyobject: test_run_example_geometry_coll
            :lines: 2-8
            :dedent: 4
        """
        g = self.__fix_geometry(value)
        self._shadow.append(g._shadow)

    def remove(self, idx: int):
        """" Удаление геометрии из коллекции.

        Args:
            idx: Индекс геометрии в коллекции.
        """
        self._shadow.remove(idx)

    def __getitem__(self, idx) -> Geometry:
        if idx >= len(self) or idx < 0:
            raise IndexError()
        return Geometry._wrap(ShadowCollection.at(self._shadow, idx))

    def __setitem__(self, idx, *value: Union[Geometry, Pnt, Tuple[float, float]]):
        if idx >= len(self) or idx < 0:
            raise IndexError()
        g = self.__fix_geometry(value)
        self._shadow.set(idx, g._shadow)

    def __repr__(self):
        try:
            return self.to_wkt()
        except (Exception,):
            return ', '.join(g.__repr__ for g in self)

    def __str__(self):
        return '{} (objects={})'.format(self._class_name, self.__len__()) + self._cs_str

    def try_to_simplified(self) -> Union['MultiPoint', 'MultiLineString', 'MultiPolygon', 'GeometryCollection']:
        """Создает копию коллекции при этом пробует упростить ее тип.
        К примеру, если в коллекции только полигоны, то вернет объект типа :class:`MultiPolygon`.
        Если исходная коллекция разнородна, возвращается копия исходной."""
        polygons = 0
        points = 0
        lines = 0
        for o in self:
            polygons += isinstance(o, Polygon)
            points += isinstance(o, Point)
            lines += isinstance(o, (Line, LineString))

        if polygons > 0 and points == 0 and lines == 0:
            res = MultiPolygon(cs=self.coordsystem)
        elif polygons == 0 and points > 0 and lines == 0:
            res = MultiPoint(cs=self.coordsystem)
        elif polygons == 0 and points == 0 and lines > 0:
            res = MultiLineString(cs=self.coordsystem)
        else:
            return self.clone()
        for o in self:
            res.append(o)
        return res


class MultiPoint(GeometryCollection):
    """Коллекция точечных объектов. Может содержать только объекты типа точка.

    Args:
        cs: Система Координат, в которой создается геометрия.

    .. literalinclude:: /../../tests/doc_examples/da/test_example_geometry.py
        :caption: Пример.
        :pyobject: test_run_example_geometry_coll_point
        :lines: 2-
        :dedent: 4
    """

    def __init__(self, cs: CoordSystem = None):
        self._set_shadow(ShadowMultiPoint(None if cs is None else cs._shadow))


class MultiLineString(GeometryCollection):
    """Коллекция полилиний. Может содержать только объекты типа полилиния.

    Args:
        cs: Система Координат, в которой создается геометрия.

    .. literalinclude:: /../../tests/doc_examples/da/test_example_geometry.py
        :caption: Пример.
        :pyobject: test_run_example_geometry_coll_ls
        :lines: 2-
        :dedent: 4
    """

    def __init__(self, cs: CoordSystem = None):
        self._set_shadow(ShadowMultiLineString(
            None if cs is None else cs._shadow))


class MultiPolygon(GeometryCollection):
    """Коллекция полигонов. Может содержать только объекты типа полигон.

    Args:
        cs: Система Координат, в которой создается геометрия.

    .. literalinclude:: /../../tests/doc_examples/da/test_example_geometry.py
        :caption: Пример.
        :pyobject: test_run_example_geometry_coll_poly
        :lines: 2-
        :dedent: 4
    """

    def __init__(self, cs: CoordSystem = None):
        self._set_shadow(ShadowMultiPolygon(None if cs is None else cs._shadow))

    def _fix_tuple(self, g) -> Geometry:
        return Polygon(g, cs=self.coordsystem)
