from enum import Flag

from axipy.cpp_gui import ShadowProgressSpecification

__all__ = ["ProgressGuiFlags", "ProgressSpecification"]


class ProgressGuiFlags(Flag):
    """
    Флаги для настройки диалога, отображающего прогресс.
    """
    IDLE = 0
    """Стандартный диалог с прогрессом."""
    CANCELABLE = 1 << 1
    """У диалога с прогрессом будет кнопка отмены."""
    NO_DELAY = 1 << 2
    """Диалог с прогрессом появляется сразу без задержки. По умолчанию это 2 - 3 секунды."""


class ProgressSpecification:
    """
    Содержит параметры для настройки отображения диалога с прогрессом.

    .. literalinclude:: /../../tests/doc_examples/concurrent/test_example_concurrent.py
        :caption: Пример использования.
        :pyobject: test_progress_specification
        :lines: 2-
        :dedent: 4

    .. attribute:: window_title

        Задаёт заголовок окна для диалога, отображающего прогресс.

        :type: :class:`str`

    .. attribute:: flags

        С помощью флагов можно выбрать тип диалога который будет отображаться
        пользователю. Флаги можно комбинировать с помощью побитовых операций.

        :type: :class:`~ProgressGuiFlags`

    .. attribute:: description

        Описание выполняемой задачи. 

        :type: :class:`str`

    .. attribute:: with_handler

        По умолчанию в пользовательскую функцию будет передаваться обработчик 
        прогресса выполнения задачи :class:`~AxipyProgressHandler`. Но иногда 
        это не нужно, тогда можно этот параметр установить в False.

        :type: :class:`bool`
    """

    def __init__(
            self,
            description: str = "",
            window_title="",
            flags: ProgressGuiFlags = ProgressGuiFlags.IDLE,
            with_handler=True
    ) -> None:
        self.description: str = description
        self.window_title: str = window_title
        self.flags: ProgressGuiFlags = flags
        self.with_handler = with_handler


def _to_cpp_progress_spec(
        spec: ProgressSpecification
) -> ShadowProgressSpecification:
    cpp_spec = ShadowProgressSpecification()
    cpp_spec.flags = spec.flags.value
    cpp_spec.description = spec.description
    cpp_spec.window_title = spec.window_title
    return cpp_spec
