from axipy.da.data_object import Table
from .data_provider import DataProvider
from .source import Source, Destination, Schema

__all__ = [
    "ExcelSource",
    "ExcelDestination",
    "ExcelDataProvider",
]


class ExcelSource(Source):
    pass


class ExcelDestination(Destination):
    pass


class ExcelDataProvider(DataProvider):
    """Провайдер чтения файлов Excel.
    
    Note:
        Ссылку на провайдер можно получить через глобальную переменную :attr:`axipy.da.provider_manager.excel`.
    """
    _identifier = 'XlsDataProvider'

    def get_source(self, filepath: str, page: str = None, with_header: bool = False,
                   encoding: str = 'utf8', alias: str = None) -> Source:
        """Создает источник данных.

        Args:
            filepath: Путь к файлу.
            page: Имя страницы. Если не указана, то берется первая.
            with_header: Признак того, что в первой строке содержатся имена атрибутов таблицы.
            encoding: Кодировка.
        """
        return ExcelSource(
            Source._provider(self.id),
            Source._table_file(filepath),
            Source._alias(alias),
            {
                'SheetName': page,
                'FirstRowAsFieldName': with_header,
                'charset': encoding,
            }
        )

    def open(self, filepath: str, page: str = None, with_header: bool = False,
             encoding: str = 'utf8', alias: str = None) -> Table:
        """Открывает объект данных.

        Args:
            filepath: Путь к файлу.
            page: Имя страницы.
            with_header: Признак того, что в первой строке содержатся имена атрибутов таблицы.
            encoding: Кодировка.
            alias: Псевдоним для открываемой таблицы.
        """
        return self.get_source(filepath, page, with_header, encoding, alias).open()

    def get_destination(self, filepath: str, schema: Schema) -> Destination:
        """Создает назначение объекта данных.

        Args:
            filepath: Путь к файлу.
            schema: Схема таблицы.
        """
        return ExcelDestination(schema,
                                Source._provider(self.id),
                                Source._table_file(filepath),
                                )

    def create_open(self, filepath: str, schema: Schema) -> Table:
        """Создает и открывает объект данных.

        Args:
            filepath: Путь к файлу.
            schema: Схема таблицы.
        """
        return self.get_destination(filepath, schema).create_open()
