import os
import sys
from pathlib import Path

__all__ = []


def _install_excepthook():
    # install custom exception hook

    def axioma_excepthook(extype, value, traceback):
        print("Unhandled Python Exception:", file=sys.stderr)
        sys.__excepthook__(extype, value, traceback)

    sys.excepthook = axioma_excepthook


# function uses paths, relative to file location
def _setup_environment():
    if "AXIOMA_HOME" in os.environ:
        return None

    if os.name == "nt":
        installpath = Path(__file__).parents[5]
    elif sys.platform.startswith("darwin"):
        installpath = Path(__file__).parents[9] / "Frameworks"
    else:
        installpath = Path(__file__).parents[6] / "bin"

    devpath = Path(__file__).parents[3]

    def contains_core_binary(path: Path):
        """
        Possible values:
            core.dll libcore.so.* libcore.*.dylib
        """
        if not path.is_dir():
            raise Exception("Not a dir")

        for arg in path.iterdir():
            name = arg.name
            if name.startswith("core.") or name.startswith("libcore."):
                return True
        return False

    if contains_core_binary(installpath):
        home = installpath
        dev = False
    elif contains_core_binary(devpath):
        home = devpath
        dev = True
    else:
        raise RuntimeError("Can't find Axioma installation")

    def insert_first_environ_path(key: str, environ_path: str):
        pathsep = os.pathsep
        env_dict = os.environ

        value = env_dict.get(key, None)
        if value is None:
            env_dict[key] = environ_path
        elif not isinstance(value, str):
            raise TypeError("environ value not a string")
        elif value.startswith(pathsep):
            env_dict[key] = environ_path + env_dict[key]
        else:
            env_dict[key] = environ_path + pathsep + env_dict[key]

    if dev:
        insert_first_environ_path("PATH", str(home.parents[2] / "third-party" / "out" / "bin"))

    insert_first_environ_path("PATH", str(home))
    os.environ["AXIOMA_HOME"] = str(home)

    if not dev:
        if sys.platform.startswith("darwin"):
            os.environ["QT_PLUGIN_PATH"] = str(home.parents[1] / "PlugIns")
        else:
            os.environ["QT_PLUGIN_PATH"] = str(home / "qtplugins")


def _setup_all():
    _install_excepthook()
    _setup_environment()
