from typing import Union, Optional

from PySide2.QtCore import QRectF, QPointF, QRect, QLocale
from axipy.cpp_common import CommonUtils

from .float_formatter import FloatFormatter
from .pnt import Pnt

__all__ = ["Rect"]


class Rect:
    """
    Прямоугольник, который не обладает геопривязкой. Используется для различного вида запросов.

    .. literalinclude:: /../../tests/doc_examples/utl/test_example_rect.py
        :caption: Создание прямоугольника.
        :pyobject: test_run_example_rect
        :start-after: # start init
        :end-before: # finish init
        :dedent: 4

    """

    def __init__(self, xmin: float, ymin: float, xmax: float, ymax: float) -> None:
        self.__rect = QRectF(xmin, ymin, xmax - xmin, ymax - ymin).normalized()

    @classmethod
    def from_qt(cls, r: Union[QRectF, QRect]) -> Optional['Rect']:
        """
        Преобразует из формата Qt. Если класс не соответствует, возвращает None.

        Args:
            r: Преобразуемый прямоугольник.
        """
        if not isinstance(r, (QRectF, QRect)):
            return None
        obj = cls.__new__(cls)
        obj.__rect = r if isinstance(r, QRectF) else QRectF(r)
        return obj

    def to_qt(self) -> QRectF:
        """Преобразование в формат Qt."""
        return self.__rect

    @classmethod
    def _rect_value_to_qt(cls, v: Union[QRectF, tuple, 'Rect', QRect]) -> QRectF:
        if isinstance(v, tuple) and len(v) == 4:
            return QRectF(QPointF(v[0], v[1]), QPointF(v[2], v[3]))
        elif isinstance(v, Rect):
            return v.to_qt()
        elif isinstance(v, QRect):
            return QRectF(v)
        elif isinstance(v, QRectF):
            return v
        elif isinstance(v, Rect):
            return v.to_qt()
        raise ValueError(f'Bad value {v}')

    @classmethod
    def _rect_value_from_qt(cls, v: Union[QRectF, 'Rect', QRect]) -> 'Rect':
        if isinstance(v, QRectF):
            return Rect.from_qt(v)
        elif isinstance(v, QRect):
            return Rect.from_qt(QRectF(v))
        elif isinstance(v, Rect):
            return v
        raise ValueError(f'Bad value {v}')

    @property
    def xmin(self) -> float:
        """
        Устанавливает или возвращает минимальное значение X.
        """
        return self.__rect.left()

    @xmin.setter
    def xmin(self, v: float):
        self.__rect.setLeft(v)

    @property
    def ymin(self) -> float:
        """
        Устанавливает или возвращает минимальное значение Y.
        """
        return self.__rect.top()

    @ymin.setter
    def ymin(self, v: float):
        self.__rect.setTop(v)

    @property
    def xmax(self) -> float:
        """
        Устанавливает или возвращает максимальное значение X.
        """
        return self.__rect.right()

    @xmax.setter
    def xmax(self, v: float):
        self.__rect.setRight(v)

    @property
    def ymax(self) -> float:
        """
        Устанавливает или возвращает максимальное значение Y.
        """
        return self.__rect.bottom()

    @ymax.setter
    def ymax(self, v: float):
        self.__rect.setBottom(v)

    @property
    def center(self) -> Pnt:
        """
        Устанавливает или возвращает центр прямоугольника.
        """
        return Pnt.from_qt(self.__rect.center())

    @center.setter
    def center(self, p: Pnt):
        self.__rect.moveCenter(Pnt._fixPnt(p).to_qt())

    def __eq__(self, other) -> bool:
        if isinstance(other, Rect):
            return CommonUtils.approx_compare_rects(self.__rect, other.__rect)
        return False
    
    @classmethod
    def eq_approx(cls, rect1: 'Rect', rect2: 'Rect', precision: float = 1e-12) -> bool:
        """
        Сравнивает два прямоугольника с заданной точностью.

        Args:
            rect1: Первый прямоугольник сравнения
            rect2: Второй прямоугольник сравнения
            precision: Точность сравнения

        Return:
            True если прямоугольники равны
        """
        if isinstance(rect1, Rect) and isinstance(rect2, Rect) :
            return CommonUtils.approx_compare_rects(rect1.__rect, rect2.__rect, precision)
        return False


    def __repr__(self) -> str:
        xmin = FloatFormatter.to_localized_string_round(self.xmin, 15, QLocale.English)
        ymin = FloatFormatter.to_localized_string_round(self.ymin, 15, QLocale.English)
        xmax = FloatFormatter.to_localized_string_round(self.xmax, 15, QLocale.English)
        ymax = FloatFormatter.to_localized_string_round(self.ymax, 15, QLocale.English)
        return f"{__class__.__name__}({xmin}, {ymin}, {xmax}, {ymax})"

    def __str__(self) -> str:
        return "({} {}) ({} {})".format(self.xmin, self.ymin, self.xmax, self.ymax)

    @property
    def is_empty(self) -> bool:
        """Если один или оба размера равны нулю."""
        return self.__rect.isEmpty()

    @property
    def is_valid(self) -> bool:
        """Является ли прямоугольник правильным."""
        return self.__rect.isValid()

    def normalize(self):
        """
        Исправляет прямоугольник, если его ширина или высота отрицательны.

        .. literalinclude:: /../../tests/doc_examples/utl/test_example_rect.py
            :caption: Пример.
            :pyobject: test_run_example_rect
            :start-after: # start normalize
            :end-before: # finish normalize
            :dedent: 4

        """
        self.__rect = self.__rect.normalized()

    @property
    def width(self) -> float:
        """Ширина прямоугольника."""
        return self.__rect.width()

    @property
    def height(self) -> float:
        """Высота прямоугольника."""
        return self.__rect.height()

    def merge(self, other: 'Rect') -> 'Rect':
        """
        Возвращает прямоугольник, занимаемый обоими прямоугольниками.

        Args:
            other: Прямоугольник, с которым производится операция.

        .. literalinclude:: /../../tests/doc_examples/utl/test_example_rect.py
            :caption: Пример.
            :pyobject: test_run_example_rect
            :start-after: # start merge
            :end-before: # finish merge
            :dedent: 4

        """
        r2 = Rect._rect_value_to_qt(other)
        return Rect.from_qt(self.__rect.united(r2))

    def intersected(self, other: 'Rect') -> 'Rect':
        """
        Возвращает общий для обоих прямоугольник.

        Args:
            other: Прямоугольник, с которым производится операция.

        .. literalinclude:: /../../tests/doc_examples/utl/test_example_rect.py
            :caption: Пример.
            :pyobject: test_run_example_rect
            :start-after: # start intersected
            :end-before: # finish intersected
            :dedent: 4

        """
        r2 = Rect._rect_value_to_qt(other)
        return Rect.from_qt(self.__rect.intersected(r2))

    def translated(self, dx: float, dy: float) -> 'Rect':
        """
        Возвращает прямоугольник, смещенный на заданную величину.

        Args:
            dx: смещение по X
            dy: смещение по Y

        .. literalinclude:: /../../tests/doc_examples/utl/test_example_rect.py
            :caption: Пример.
            :pyobject: test_run_example_rect
            :start-after: # start translated
            :end-before: # finish translated
            :dedent: 4

        """
        return Rect.from_qt(self.__rect.translated(dx, dy))

    def expanded(self, dx: float, dy: float) -> 'Rect':
        """
        Возвращает прямоугольник, увеличенный на заданные величины.
        Увеличение размеров производится по отношению к центру,
        который не меняется в результате операции.

        Args:
            dx: расширение по X
            dy: расширение по Y

        .. literalinclude:: /../../tests/doc_examples/utl/test_example_rect.py
            :caption: Пример.
            :pyobject: test_run_example_rect
            :start-after: # start expanded
            :end-before: # finish expanded
            :dedent: 4
        """
        dx2 = dx / 2
        dy2 = dy / 2
        return Rect.from_qt(self.__rect.adjusted(-dx2, -dy2, dx2, dy2))

    def contains(self, other: Union[Pnt, QPointF, 'Rect', QRectF]) -> bool:
        """
        Содержит ли полностью в своих границах переданный объект.
        
        Args:
            other: Переданный объект - точка или прямоугольник.

        .. literalinclude:: /../../tests/doc_examples/utl/test_example_rect.py
            :caption: Пример.
            :pyobject: test_run_example_rect
            :start-after: # start contains
            :end-before: # finish contains
            :dedent: 4
        """
        try:
            r = Rect._rect_value_to_qt(other)
            return self.__rect.contains(r)
        except ValueError:
            try:
                p = Pnt._point_value_to_qt(other)
                return self.__rect.contains(p)
            except ValueError:
                pass
        return False
