from typing import List

from PySide2.QtCore import QObject, Signal, Slot
from PySide2.QtWidgets import QWidget, QListView
from axipy.cpp_gui import (
    ShadowLayerControlWidget, ShadowDataCatalogWidget, ShadowNotificationWidget, ShadowListWindowWidget,
    ShadowPythonConsoleWidget
)

from axipy.da import DataObject
from .gui_class import gui_instance
from .view import View, MapView
from .view_manager_ import view_manager

__all__ = [
    "DataManagerWidget",
    "ViewManagerWidget",
    "LayerControlWidget",
    "NotificationWidget",
    "PythonConsoleWidget",
]


class DataManagerWidget:
    """Виджет управления списком открытых данных."""

    def __init__(self) -> None:
        self._shadow = ShadowDataCatalogWidget.create(gui_instance._shadow)

    @property
    def widget(self) -> QWidget:
        """Виджет, соответствующий содержимому окна.

        Return:
            Qt5 виджет содержимого.
        """
        return self._shadow.widget()

    @property
    def list_widget(self) -> QListView:
        """Виджет, соответствующий содержимому списка таблиц.

        Return:
            Qt5 виджет списка.
        """
        return self._shadow.listWidget()

    @property
    def selection_changed(self) -> Signal:
        """
        Выделение в списке было изменено.

        :rtype: Signal[]
        """
        return self._shadow.selectionChanged

    @property
    def objects(self) -> List[DataObject]:
        """Список выделенных объектов."""
        return [DataObject._wrap(obj) for obj in self._shadow.dataObjectsSelected()]


class ViewManagerWidget:
    """Виджет списка окон."""

    class _ListWindowSignals(QObject):
        view_changed = Signal(View)

    def __init__(self) -> None:
        self._signals = ViewManagerWidget._ListWindowSignals()
        self._signal = self._signals.view_changed

        self._shadow = ShadowListWindowWidget(gui_instance._shadow)
        self._shadow.selectionChanged.connect(self._slot_view_changed)

    @property
    def view_changed(self) -> Signal:
        """
        Сигнал об изменении выделения в списке.

        :rtype: Signal[View]
        """
        return self._signal

    @property
    def widget(self) -> QWidget:
        """
        Виджет, соответствующий содержимому окна.

        Return:
            Qt5 виджет содержимого.
        """
        return self._shadow

    @Slot(View)
    def _slot_view_changed(self, w):
        for v in view_manager.views:
            if v.widget == w:
                self._signal.emit(v)


class LayerControlWidget:
    """Виджет управления слоями карты."""

    class _LayerControlSignals(QObject):
        mapview_activated = Signal(MapView)

    def __init__(self) -> None:
        self._signals = LayerControlWidget._LayerControlSignals()
        self._signal = self._signals.mapview_activated

        self._shadow = ShadowLayerControlWidget.create(gui_instance._shadow)
        self._shadow.mapViewActivated.connect(self._slot_mapview_activated)

    @property
    def mapview_activated(self) -> Signal:
        """
        Сигнал об изменении активной карты.

        :rtype: Signal[MapView]
        """
        return self._signal

    @property
    def widget(self) -> QWidget:
        """Виджет, соответствующий содержимому окна.

        Return:
            Qt5 виджет содержимого.
        """
        return self._shadow.widget()

    @Slot(MapView)
    def _slot_mapview_activated(self, w):
        for map_view in view_manager.mapviews:
            if map_view.widget == w:
                self._signal.emit(map_view)


class NotificationWidget:
    """Виджет уведомлений."""

    def __init__(self) -> None:
        self._shadow = ShadowNotificationWidget()

    @property
    def widget(self) -> QWidget:
        """Виджет, соответствующий содержимому окна.

        Return:
            Qt5 виджет содержимого.
        """
        return self._shadow


class PythonConsoleWidget:
    """Виджет ввода команд python."""

    def __init__(self) -> None:
        self._shadow = ShadowPythonConsoleWidget.create(gui_instance._shadow)

    @property
    def widget(self) -> QWidget:
        """Виджет, соответствующий содержимому окна.

        Return:
            Qt5 виджет содержимого.
        """
        return self._shadow.widget()
