import sys
import traceback
from abc import abstractmethod

from axipy.cpp_core_core import ShadowTask

from .axipy_progress_handler import AxipyProgressHandler

__all__ = ["AxipyTask", "AxipyAnyCallableTask"]


class AxipyTask(ShadowTask):

    """
    Базовый класс пользовательской задачи. 
    От него можно наследоваться, создавая собственный задачи. Для этого 
    нужно переопределить метод :meth:`run`.
    """

    def __init__(self):
        super().__init__()
        self.set_progress_handler(AxipyProgressHandler())

    def _init_progress_handler(self, ph: AxipyProgressHandler):
        if not issubclass(type(ph), AxipyProgressHandler):
            raise TypeError(
                f"ph should has AxipyProgressHandler type or one of subclass, but was gotten {type(ph)}")
        super().init_progress_handler(ph._wrapper())

    def progress_handler(self) -> AxipyProgressHandler:
        """
        Возвращает обработчик для текущей задачи.
        """
        return self.__ph

    def set_progress_handler(self, ph: AxipyProgressHandler):
        """
        Устанавливает обработчик для текущей задачи. Поддерживаются
        любые наследники :class:`AxipyProgressHandler`.
        """
        self.__ph = ph
        self._init_progress_handler(self.__ph)

    def run_internally(self):
        try:
            self.__ph.result = self.run()
        except (Exception,):
            traceback.print_exc()
            exctype, value = sys.exc_info()[:2]
            self.__ph.error.emit(
                (exctype, value, traceback.format_exc()))
        # finally:
            # run_in_gui_thread(self.on_finished, self.__ph.result)

    @abstractmethod
    def run(self):
        """
        Функция, выполняющая код пользовательской задачи. Переопределяется в
        дочерних классах.
        """
        pass

    def on_finished(self, result):
        """
        Функция вызывается при завершении пользовательской задачи в потоке
        интерфейса.
        """
        pass


class AxipyAnyCallableTask(AxipyTask):
    """
    Объекты этого класса оборачивают пользовательские функции, превращая
    их в задачу, которая будет выполнена в фоновом потоке.

    :param fn: Пользовательская функция, которая будет выполнятся. В нее
     будут переданы сохраненные параметры: список `args` и словарь `kwargs`.
    :param args: Список аргументов, передаваемый в функцию при запуске.
    :param kwargs: Словарь, передаваемый в функцию при запуске.


    .. literalinclude:: /../../tests/doc_examples/concurrent/test_example_concurrent.py
        :caption: Пример использования.
        :pyobject: test_callable_task
        :lines: 2-
        :dedent: 4
    """

    def __init__(self, fn, *args, **kwargs):
        super().__init__()
        self.__fn = fn
        self.__args = args
        self.__kwargs = kwargs
        self.__with_handler = True

    def with_handler(self, value: bool):
        """
        По умолчанию в пользовательскую функцию первым аргументом передаётся 
        обработчик для управления задачей, установки прогресса и обработки 
        отмены. Однако он не будет передаваться если вызвать эту функцию с False.
        """
        self.__with_handler = value

    def run(self):
        """
        Метод запускает выполнение задачи.

        Warning:
            Вызывается автоматически при выполнении задачи. Вручную вызывать
            не следует.
        """
        if self.__with_handler:
            return self.__fn(
                self.progress_handler(),
                *self.__args,
                **self.__kwargs)
        else:
            return self.__fn(*self.__args, **self.__kwargs)
