import logging
import os
import platform
import subprocess
from pathlib import Path

from typing import Optional

from .dependency_installer import DependencyInstaller, InstallError, Dependencies

__all__ = [
    'PipDependencyInstaller',
]


def python_interpreter() -> Optional[Path]:
    result = Path(os.environ['AXIOMA_PYTHON_INTERPRETER'])
    return result if result.is_file() else None


class PipDependencyInstaller(DependencyInstaller):

    def _to_list(self, data):
        list_filter = filter(None, data.split("\n"))
        return list(list_filter)

    def install(self, dependency: Dependencies, dependencies_destination: Path):
        p = dependencies_destination.absolute().as_posix()
        logging.info(f'Dependency catalog: {p}')
        executable = python_interpreter()
        if not executable:
            raise InstallError('Python interpreter is not found')

        # split process by packages
        deps_file = open(str(dependency.requirements), 'r')
        data = deps_file.read()
        deps_file.close()

        if platform.system() == 'Windows':
            flags = {'creationflags': subprocess.CREATE_NO_WINDOW}
        else:
            flags = {}

        for package in self._to_list(data):
            logging.info(f'--- Package deps: {package} ---')
            try:
                exit_data = subprocess.run(
                    [str(executable), '-m', 'pip', 'install', f'--target={p}', package],
                    stdout=subprocess.PIPE,
                    stderr=subprocess.PIPE,
                    **flags
                )
                exit_string = exit_data.stdout if exit_data.returncode == 0 else exit_data.stderr
                for m in self._to_list(exit_string.decode()):
                    logging.info(m)
                if exit_data.returncode != 0:
                    raise InstallError('Error while installing dependent packages')
            except subprocess.CalledProcessError as e:
                logging.error(f"Execution error:\"{e.cmd}\"")
                raise e
            except Exception as e:
                raise e
