from abc import abstractmethod
from collections.abc import KeysView, ItemsView, ValuesView
from typing import Tuple, List, Any, Iterator


class _NoCallMeta(type):
    def __call__(cls, *args, **kwargs):
        raise TypeError("Class doesn't support instance creation")


class _NoSetAttrMeta(type):
    def __setattr__(cls, key, value):
        raise TypeError("Class is read-only")


class _MappingMetaBase(type):

    @abstractmethod
    def __iter__(cls) -> Iterator:
        raise NotImplementedError

    @abstractmethod
    def __len__(cls) -> int:
        raise NotImplementedError

    @abstractmethod
    def __getitem__(cls, key: Any) -> Any:
        raise NotImplementedError

    # Default
    def __contains__(cls, key):
        try:
            cls[key]
        except KeyError:
            return False
        else:
            return True

    def keys(cls):
        return list(KeysView(cls))

    def values(cls):
        return list(ValuesView(cls))

    def items(cls):
        return list(ItemsView(cls))

    def get(cls, key, default=None):
        try:
            return cls[key]
        except KeyError:
            return default

    def __eq__(cls, other):
        if not isinstance(other, _MappingMetaBase):
            return NotImplemented
        return dict(cls.items()) == dict(other.items())


class _MappingMetaReprStr(type):

    def __repr__(cls):
        return repr(cls.items())

    def __str__(cls):
        return str(cls.items())


class _MappingMetaExtended(_NoCallMeta, _NoSetAttrMeta, _MappingMetaReprStr, _MappingMetaBase):
    ...


class _MappingMetaDocumentation:
    """
    Нужен для документации sphinx и для IntelliSense в IDE.
    Методы абстрактные, чтобы наследник переопределял docstring и type-hint.
    При наследовании нужен только вызов super().
    """

    @classmethod
    @abstractmethod
    def items(cls: _MappingMetaBase) -> List[Tuple[str, Any]]:
        """Возвращает список кортежей ключ-значение."""
        return type(cls).items(cls)

    @classmethod
    @abstractmethod
    def keys(cls: _MappingMetaBase) -> List[str]:
        """Возвращает список ключей."""
        return type(cls).keys(cls)

    @classmethod
    @abstractmethod
    def values(cls: _MappingMetaBase) -> List[Any]:
        """Возвращает список значений."""
        return type(cls).values(cls)

    @classmethod
    @abstractmethod
    def get(cls: _MappingMetaBase, key: str, default_value: Any = None):
        """Возвращает значение по ключу."""
        return type(cls).get(cls, key, default_value)
