import importlib
import traceback
from pathlib import Path
from typing import Tuple

from axipy import AxiomaPlugin, tr, Position, DefaultKeys
from axipy.app import Notifications
from axipy.menubar import ActionButton, ToolButton, Button

from .buttons import (
    copy_paste_geometry,
    copy_paste_style,
    outline_action,
    outline_tool,
    mirror,
    perpendicular,
    right_angle,
    redirect,
    voronoi,
    arc_by_points,
    circle_by_points
)
from .buttons.cad_polyline import main as cad_polyline_main
from .buttons.select_by_style import main as select_by_style_main

is_master = str(Path(__file__).parents[0].name).endswith("master")

new_line_n = 10

icon_path = Path(__file__).parents[0] / "images" / "24px"


class Plugin(AxiomaPlugin):

    def load(self) -> None:
        self.buttons = list()
        try:
            self.load_buttons()
        except Exception:
            traceback.print_exc()
            Notifications.push(tr("Плагин дополнительные инструменты"), tr("Не удалось загрузить плагин"),
                               Notifications.Critical)

    def unload(self) -> None:
        for button in self.buttons:
            if isinstance(button, Button):
                button.remove()
        self.buttons = None

    def load_buttons(self) -> None:

        tab_name = tr("Дополнительно")

        def add_and_save(group_arg: str, arg) -> None:
            """
            Добавление кнопок на позиции и сбор кнопок в список для последующей выгрузки.
            """

            def add_and_save_single(btn):
                if not isinstance(btn, Button):
                    return None
                self.buttons.append(btn)
                pos = Position(tab_name, group_arg)
                pos.add(btn, size=1)

            if isinstance(arg, tuple):
                for elem in arg:
                    add_and_save_single(elem)
            else:
                add_and_save_single(arg)

        group = tr("Буфер обмена")
        add_and_save(group, self.init_copy_paste_geometry())
        add_and_save(group, self.create_separator())
        add_and_save(group, self.init_copy_paste_style())
        add_and_save(group, self.init_find_by_style())

        group = tr("Действия")
        add_and_save(group, self.init_outline_action())
        add_and_save(group, self.init_mirror())
        add_and_save(group, self.init_redirect())

        group = tr("Инструменты")
        add_and_save(group, self.init_outline_tool())
        add_and_save(group, self.init_voronoi())
        add_and_save(group, self.init_cad_polyline())

        group = tr("Рисование")
        add_and_save(group, self.init_arc_by_points())
        add_and_save(group, self.init_circle_by_points())
        add_and_save(group, self.init_perpendicular())
        add_and_save(group, self.init_right_angle())

    """ Buttons """

    """ Copy/Paste """

    def init_copy_paste_geometry(self) -> Tuple[ActionButton, ToolButton]:
        storage = copy_paste_geometry.GeometryStorage()

        copy_title = tr("Копировать геометрию")

        def copy_factory():
            if is_master:
                print("\n" * new_line_n)
                importlib.reload(copy_paste_geometry)

            copy_geom = copy_paste_geometry.CopyGeometry(self, copy_title, storage)
            return copy_geom.on_triggered()

        copy_button = self.create_action(
            title=copy_title,
            on_click=copy_factory,
            enable_on=DefaultKeys.Selection,
            icon=str(icon_path / "copy_geometry.png"),
            tooltip=tr("Копирует геометрию в буфер обмена."),
            doc_file="copy_paste_geometry.html"
        )

        # Регистрируем наблюдатель
        copy_paste_geometry.register_paste_geometry_observer(storage)

        paste_title = tr("Вставить геометрию")

        def paste_factory():
            if is_master:
                print("\n" * new_line_n)
                importlib.reload(copy_paste_geometry)

            paste_geom = copy_paste_geometry.PasteGeometry(self, paste_title, storage)
            return paste_geom

        paste_button = self.create_tool(
            title=paste_title,
            on_click=paste_factory,
            enable_on=copy_paste_geometry.paste_geometry_observer_key,
            icon=str(icon_path / "paste_geometry.png"),
            tooltip=tr("Вставляет геометрию из буфера обмена в указанное место."),
            doc_file="copy_paste_geometry.html"
        )

        return copy_button, paste_button

    def init_copy_paste_style(self) -> Tuple[ActionButton, ActionButton]:
        storage = copy_paste_style.StyleStorage()

        copy_title = tr("Копировать стиль")

        def copy_factory():
            if is_master:
                print("\n" * new_line_n)
                importlib.reload(copy_paste_style)

            copy_style = copy_paste_style.CopyStyle(self, copy_title, storage)
            return copy_style.on_triggered()

        copy_button = self.create_action(
            title=copy_title,
            on_click=copy_factory,
            enable_on=DefaultKeys.Selection,
            icon=str(icon_path / "copy_style.png"),
            tooltip=tr("Копирует стиль оформления объекта в буфер обмена."),
            doc_file="copy_paste_style.html"
        )

        # Регистрируем наблюдателя
        copy_paste_style.register_paste_style_observer(storage)

        paste_title = tr("Вставить стиль")

        def paste_factory():
            if is_master:
                print("\n" * new_line_n)
                importlib.reload(copy_paste_style)

            paste_action = copy_paste_style.PasteStyle(self, paste_title, storage)
            return paste_action.on_triggered()

        paste_button = self.create_action(
            title=paste_title,
            on_click=paste_factory,
            enable_on=copy_paste_style.paste_style_observer_key,
            icon=str(icon_path / "paste_style.png"),
            tooltip=tr("Применяет скопированный ранее стиль на выделенные объекты."),
            doc_file="copy_paste_style.html"
        )

        return copy_button, paste_button

    def init_find_by_style(self) -> ToolButton:
        title = tr("Выбор по стилю")
        observer_id = DefaultKeys.MapView

        def factory():
            if is_master:
                print("\n" * new_line_n)
                importlib.reload(select_by_style_main)

            select_by_style_tool = select_by_style_main.SelectByStyle(self, title, observer_id)
            return select_by_style_tool

        button = self.create_tool(
            title=title,
            on_click=factory,
            icon=str(icon_path / "find_by_style.png"),
            enable_on=observer_id
        )

        return button

    """ Actions """

    def init_outline_action(self) -> ActionButton:
        title = tr("Оконтурить - действие")

        def factory():
            if is_master:
                print("\n" * new_line_n)
                importlib.reload(outline_action)

            outline_action_arg = outline_action.OutlineAction(self, title)
            return outline_action_arg.on_triggered()

        button = self.create_action(
            title=title,
            on_click=factory,
            enable_on=DefaultKeys.SelectionEditable,
            icon=str(icon_path / "outline_action.png"),
            tooltip=tr("Позволяет создавать контур (полигон) вокруг выбранных объектов."),
            doc_file="outline_action.html"
        )

        return button

    def init_mirror(self) -> Tuple[ActionButton, ActionButton]:
        vertical_title = tr("Отразить по вертикали")

        def vertical_factory():
            if is_master:
                print("\n" * new_line_n)
                importlib.reload(mirror)

            vertical_mirror = mirror.VerticalMirror(self, vertical_title)
            return vertical_mirror.on_triggered()

        vertical_button = self.create_action(
            title=vertical_title,
            on_click=vertical_factory,
            enable_on=DefaultKeys.SelectionEditableIsSame,
            icon=str(icon_path / "vertical_mirror.png"),
            tooltip=tr("Объекты зеркально отражаются относительно центроида описанного вокруг них прямоугольника."),
            doc_file="vertical_mirror.html"
        )

        horizontal_title = tr("Отразить по горизонтали")

        def horizontal_factory():
            if is_master:
                print("\n" * new_line_n)
                importlib.reload(mirror)

            horizontal_mirror = mirror.HorizontalMirror(self, horizontal_title)
            return horizontal_mirror.on_triggered()

        horizontal_button = self.create_action(
            title=horizontal_title,
            on_click=horizontal_factory,
            enable_on=DefaultKeys.SelectionEditableIsSame,
            icon=str(icon_path / "horizontal_mirror.png"),
            tooltip=tr("Объекты зеркально отражаются относительно центроида описанного вокруг них прямоугольника."),
            doc_file="horizontal_mirror.html"
        )

        return vertical_button, horizontal_button

    def init_redirect(self) -> ActionButton:
        title = tr("Изменить направление линии")

        def factory():
            if is_master:
                print("\n" * new_line_n)
                importlib.reload(redirect)

            redirect_line_action = redirect.RedirectLine(title)
            return redirect_line_action.on_triggered()

        button = self.create_action(
            title=title,
            on_click=factory,
            enable_on=DefaultKeys.SelectionEditableIsSame,
            icon=str(icon_path / "redirect.png"),
            tooltip=tr("Изменяет направление элементов выбранных объектов."),
            doc_file="redirect.html"
        )

        return button

    """ Tools """

    def init_outline_tool(self) -> ToolButton:
        title = tr("Оконтурить - инструмент")
        observer_id = DefaultKeys.Editable

        def factory():
            if is_master:
                print("\n" * new_line_n)
                importlib.reload(outline_tool)

            outline_tool_arg = outline_tool.OutlineTool(self, title, observer_id)
            return outline_tool_arg

        button = self.create_tool(
            title=title,
            on_click=factory,
            enable_on=observer_id,
            icon=str(icon_path / "outline_tool.png"),
            tooltip=tr("Позволяет создавать вогнутый контур вокруг выбранных объектов."),
            doc_file="outline_tool.html"
        )

        return button

    def init_voronoi(self) -> ToolButton:
        title = tr("Полигоны Вороного")
        observer_id = DefaultKeys.MapView

        def factory():
            if is_master:
                print("\n" * new_line_n)
                importlib.reload(voronoi)

            polygons_voronogo = voronoi.VoronoiMapTool(self, title, observer_id)
            return polygons_voronogo

        button = self.create_tool(
            title=title,
            on_click=factory,
            enable_on=observer_id,
            icon=str(icon_path / "voronoi.png"),
            tooltip=tr("Построение полигонов Вороного по выбранным объектам."),
            doc_file="voronoi.html"
        )

        return button

    def init_cad_polyline(self) -> ToolButton:
        title = tr("Создать по углу и расстоянию")
        observer_id = DefaultKeys.Editable

        def factory():
            if is_master:
                print("\n" * new_line_n)
                importlib.reload(cad_polyline_main)

            cad_polyline_tool = cad_polyline_main.CadPolyline(self, title, observer_id)
            return cad_polyline_tool

        button = self.create_tool(
            title=title,
            on_click=factory,
            icon=str(icon_path / "cad_polyline.png"),
            enable_on=observer_id
        )

        return button

    """ Drawing """

    def init_arc_by_points(self) -> ToolButton:
        title = tr("Дуга по 3 точкам")

        def factory():
            if is_master:
                print("\n" * new_line_n)
                importlib.reload(arc_by_points)

            arc_by_points_tool = arc_by_points.ArcByPoints(self, title)
            return arc_by_points_tool

        button = self.create_tool(
            title=title,
            on_click=factory,
            icon=str(icon_path / "arc_by_points.png"),
            enable_on=DefaultKeys.Editable
        )

        return button

    def init_circle_by_points(self) -> ToolButton:
        title = tr("Окружность по 3 точкам")

        def factory():
            if is_master:
                print("\n" * new_line_n)
                importlib.reload(circle_by_points)

            circle_by_points_tool = circle_by_points.CircleByPoints(self, title)
            return circle_by_points_tool

        button = self.create_tool(
            title=title,
            on_click=factory,
            icon=str(icon_path / "circle_by_points.png"),
            enable_on=DefaultKeys.Editable
        )

        return button

    def init_perpendicular(self) -> ToolButton:

        def factory():
            if is_master:
                print("\n" * new_line_n)
                importlib.reload(perpendicular)

            perpendicular_tool = perpendicular.PerpendicularTool()
            return perpendicular_tool

        button = self.create_tool(
            title=tr("Перпендикуляр"),
            on_click=factory,
            icon=str(icon_path / "perpendicular.png"),
            enable_on=DefaultKeys.Editable,
            tooltip=tr("Создаёт новый линейный объект перпендикулярный сегменту объекта."),
            doc_file="perpendicular.html"
        )

        return button

    def init_right_angle(self) -> ToolButton:

        def factory():
            if is_master:
                print("\n" * new_line_n)
                importlib.reload(right_angle)

            right_angle_tool = right_angle.RightAngleTool()
            return right_angle_tool

        button = self.create_tool(
            title=tr("Прямой угол"),
            on_click=factory,
            icon=str(icon_path / "right_angle.png"),
            enable_on=DefaultKeys.Editable,
            tooltip=tr("Создаёт новый линейный объект перпендикулярный сегменту объекта."),
            doc_file="right_angle.html"
        )

        return button
