from typing import List, Optional
from .data_provider import DataProvider, Table
from .source import Source, Destination, Schema
from ..attribute_schema import Schema


class SqliteSource(Source):
    pass


class SqliteDataProvider(DataProvider):
    """Векторный провайдер sqlite.
    
    Note:
        Ссылку на провайдер можно получить через глобальную переменную :attr:`axipy.da.provider_manager.sqlite`.
    """
    _identifier = 'SqliteDataProvider'

    def get_source(self, filepath: str, dataobject: Optional[str] = None, sql: Optional[str] = None, prj: str = None, alias: str = None) -> Source:
        """Создает источник данных. В качестве объекта может быть указана либо таблица, либо текст запроса.
        Если указан `sql`, то он имеет более высокий приоритет по отношению к значению `dataobject`.
        Если оба параметра опущены, будет возвращен None.

        Args:
            filepath: Путь к файлу.
            dataobject: Имя таблицы.
            sql: SQL-запрос.
            prj: Строка Системы Координат.

        Пример с таблицей::

            table = provider_manager.openfile('world.sqlite', dataobject='world')

        Пример с запросом и переопределенной СК::

            table = provider_manager.openfile('world.sqlite', sql="select * from world where Страна like 'Р%'", prj='12, 104, "m", 0')
        """
        return SqliteSource(
            Source._provider(self.id),
            Source._table_file(filepath),
            Source._prj(prj),
            Source._alias(alias),
            {'dataobject': dataobject, 
             'sql': sql
            }
        )

    def open(self, filepath: str, dataobject: Optional[str] = None, sql: Optional[str] = None, prj: str = None, alias: str = None) -> Table:
        """Открывает объект данных.
        
        В качестве объекта может быть указана либо таблица, либо текст запроса.
        Если указан `sql`, то он имеет более высокий приоритет по отношению к значению `dataobject`.
        Если оба параметра опущены, будет возвращен None.

        Args:
            filepath: Путь к файлу.
            dataobject: Имя таблицы.
            sql: SQL-запрос.
            prj: Строка Системы Координат.
            alias: Псевдоним для открываемой таблицы.
        """
        return self.get_source(filepath, dataobject, sql, prj, alias).open()

    def get_destination(self):
        """
        Attention:
            Не поддерживается.

        Raises:
            NotImplementedError
        """
        raise NotImplementedError
    
    def create_open(self):
        """
        Attention:
            Не поддерживается.

        Raises:
            NotImplementedError
        """
        raise NotImplementedError
