"""
Вспомогательные функции для построения окружности и дуги
"""
import math
from pathlib import Path
from typing import List, Tuple, Union, Optional

from PySide2.QtCore import QPoint, QPointF, QLineF, Qt
from PySide2.QtGui import QPainter, QPen, QKeyEvent, QWheelEvent, QMouseEvent, QPaintEvent
from axipy import MapView, Rect, CoordTransformer, view_manager, MapTool, Feature
from axipy.utl import Pnt

from . import helper

is_master = str(Path(__file__).parents[0].name).endswith("master")


def ensure_util_geom_visual(view: MapView, util_geom: Union[List[Pnt], Rect], inverse: bool = False
                            ) -> Union[List[Pnt], Rect]:
    """
    Поправка для долготы/широты
    """
    cs = view.coordsystem
    cs_visual = view.coordsystem_visual

    if cs == cs_visual:
        return util_geom

    if is_master:
        print("Calculating visual")

    if not inverse:
        transformer = CoordTransformer(cs, cs_visual)
    else:
        transformer = CoordTransformer(cs_visual, cs)

    if isinstance(util_geom, list):
        visual_points = [transformer.transform(pnt) for pnt in util_geom]
        return visual_points
    elif isinstance(util_geom, Rect):
        visual_rect = transformer.transform(util_geom)
        return visual_rect


def center_and_radius_of_circle(points: Union[List[Pnt], List[QPoint]]) -> Optional[Tuple[Pnt, float]]:
    """
    Уравнение 2D сферы:
    (x - x0)**2 + (y - y0)**2 = r**2
    где a и b это координаты её центра. Решая это уравнения для 3х точек
    можно найти x0, y0 и радиус r. Решено с использованием sympy.
    """
    if len(points) != 3:
        return None

    p1, p2, p3 = points[0], points[1], points[2]

    if isinstance(p1, QPoint):
        x1, y1 = p1.x(), p1.y()
        x2, y2 = p2.x(), p2.y()
        x3, y3 = p3.x(), p3.y()
    else:
        x1, y1 = p1.x, p1.y
        x2, y2 = p2.x, p2.y
        x3, y3 = p3.x, p3.y

    denominator = x1 * y2 - x1 * y3 - x2 * y1 + x2 * y3 + x3 * y1 - x3 * y2

    # Три точки лежат на одной линии
    if denominator == 0:
        return Pnt(0, 0), 0

    x0 = (1 / 2) * (
            x1 ** 2 * y2 - x1 ** 2 * y3 - x2 ** 2 * y1 + x2 ** 2 * y3 + x3 ** 2 * y1 - x3 ** 2 * y2 + y1 ** 2 * y2 - y1 ** 2 * y3 - y1 * y2 ** 2 + y1 * y3 ** 2 + y2 ** 2 * y3 - y2 * y3 ** 2) / (
             denominator)
    y0 = (-1 / 2) * (
            x1 ** 2 * x2 - x1 ** 2 * x3 - x1 * x2 ** 2 + x1 * x3 ** 2 - x1 * y2 ** 2 + x1 * y3 ** 2 + x2 ** 2 * x3 - x2 * x3 ** 2 + x2 * y1 ** 2 - x2 * y3 ** 2 - x3 * y1 ** 2 + x3 * y2 ** 2) / (
             denominator)
    r = (-1 / 2) * ((x1 ** 2 - 2 * x1 * x2 + x2 ** 2 + y1 ** 2 - 2 * y1 * y2 + y2 ** 2) * (
            x1 ** 2 - 2 * x1 * x3 + x3 ** 2 + y1 ** 2 - 2 * y1 * y3 + y3 ** 2) * (
                            x2 ** 2 - 2 * x2 * x3 + x3 ** 2 + y2 ** 2 - 2 * y2 * y3 + y3 ** 2)) ** (
            (1 / 2)) / denominator

    # радиус не может быть отрицательным
    r = abs(r)

    return Pnt(float(x0), float(y0)), float(r)


def angle_by_points(center: Pnt, p: Pnt, device=False) -> float:
    """
    Возвращает математический угол в градусах.
    """

    if isinstance(center, QPoint):
        c_x, c_y = center.x(), center.y()
    else:
        c_x, c_y = center.x, center.y

    if isinstance(p, QPoint):
        p_x, p_y = p.x(), p.y()
    else:
        p_x, p_y = p.x, p.y

    dx = p_x - c_x
    dy = p_y - c_y

    if math.isclose(dx, 0, abs_tol=1e-15):
        if dy >= 0:
            degree = 90
        else:
            degree = -90
        return degree

    tan_ = dy / dx
    radians = math.atan(tan_)
    degree = math.degrees(radians)

    if dx < 0:
        degree += 180

    if degree < 0:
        degree += 360

    if device:
        degree = 360 - degree

    assert degree <= 360

    return degree


def start_end_angles(points: List[Pnt], center: Pnt, device=False) -> Tuple[float, float]:
    """
    Рассчет начального и конечного угла для дуги, по трем точкам и центру.
    """
    p1, p2, p3 = points[0], points[1], points[2]

    if device:
        a1, a2, a3 = angle_by_points(center, p1, device=True), angle_by_points(center, p2,
                                                                               device=True), angle_by_points(center, p3,
                                                                                                             device=True)
    else:
        a1, a2, a3 = angle_by_points(center, p1), angle_by_points(center, p2), angle_by_points(center, p3)

    args = (a2 - a1, a3 - a2, a1 - a3)

    # Создание дуги изначально идет против часовой

    def is_counterclockwise(angles):
        lst = [angle > 0 for angle in angles]

        def most_common():
            return max(set(lst), key=lst.count)

        return most_common()

    if is_counterclockwise(args):
        return a1, a3
    else:
        return a3, a1


def draw_lines(q_painter: QPainter, pnts: List[QPointF]):
    """
    Нарисовать круг по точкам, каждые две точки = линия.
    Необходимо минимум две точки.
    """
    n = len(pnts)
    if n >= 2:
        lines = [QLineF(pnts[i], pnts[i + 1]) for i in range(n - 1)]

        pen = q_painter.pen()
        old_pen = QPen(pen)

        pen.setStyle(Qt.DashLine)
        q_painter.setPen(pen)
        q_painter.drawLines(lines)

        q_painter.setPen(old_pen)


def insert_feature(f: Feature):
    table = helper.ensure_editable_table()
    table_cs = table.coordsystem

    # Перепроецирование вручную, потому что иногда, при выходе за границы КС, получается некорректная геометрия
    # Лучше, если Exception произойдет на перепроецировании, чем на вставке в таблицу.
    g = f.geometry
    if g.coordsystem != table_cs:
        f.geometry = g.reproject(table_cs)

    table.insert(f)


class CircleMapTool(MapTool):

    def __init__(self, iface, title) -> None:
        super().__init__()
        self.iface = iface
        self.title = title

        self.points = list()  # type: List[Pnt]
        self.current_point = None

    def load(self):
        view_manager.active_changed.connect(self.active_changed)

    def unload(self) -> None:
        view_manager.active_changed.disconnect(self.active_changed)
        self.clear_and_redraw()

    def clear_and_redraw(self):
        self.points.clear()
        self.current_point = None

        self.redraw()

    def keyPressEvent(self, event: QKeyEvent) -> Optional[bool]:
        if event.key() == Qt.Key_Escape:
            self.reset()
        else:
            return super().keyPressEvent(event)

    def wheelEvent(self, event: QWheelEvent) -> Optional[bool]:
        self.redraw()
        return self.PassEvent

    # Signals

    def active_changed(self):
        self.clear_and_redraw()

    # Mouse

    def pnt_from_mouse_event(self, event: QMouseEvent) -> Pnt:
        event_pos = event.pos()

        if self.is_snapped():
            point = self.snap(event_pos)
        else:
            point = self.to_scene(event_pos)

        return point

    def mouseMoveEvent(self, event: QMouseEvent) -> None:
        if self.points:
            self.current_point = self.pnt_from_mouse_event(event)
            self.redraw()

    def paintEvent(self, event: QPaintEvent, painter: QPainter) -> None:
        painter.save()

        d_points = [self.to_device(point) for point in self.points]
        if self.current_point:
            d_points.append(self.to_device(self.current_point))

        n = len(d_points)

        if n == 2:
            draw_lines(painter, d_points)
        elif n == 3:
            draw_lines(painter, d_points)
            self.prepare_to_draw(painter, d_points)

        painter.restore()

    def prepare_to_draw(self, painter: QPainter, points: List[QPoint]):
        pass

    def mouseReleaseEvent(self, event: QMouseEvent) -> Optional[bool]:
        if event.button() == Qt.LeftButton:

            pnt = self.pnt_from_mouse_event(event)

            self.points.append(pnt)

            if len(self.points) == 3:
                self.prepare_to_insert()
            else:
                self.redraw()

            return self.BlockEvent
        else:
            return self.PassEvent

    def prepare_to_insert(self):
        pass
