import zipfile
from pathlib import PurePosixPath
import sys
import codecs
from .parser import (IParser, Metadata, LocationPath
        , MANIFEST_FILE_NAME, ENTRY_POINT)


ARCHIVE_EXTENSION = 'axp'
_ARCHIVE_SUFFIX = '.' + ARCHIVE_EXTENSION


def _to_text_fileobject(fileobject_from_zip):
    return codecs.getreader('utf-8')(fileobject_from_zip)


class PackageParser(IParser):
    def parse(self, location_path: LocationPath) -> Metadata:
        with zipfile.ZipFile(location_path) as archive:
            return self.read_archive(archive)

    def read_archive(self, archive) -> Metadata:
        def is_root_content(path):
            return (path.endswith('/') and path.count('/') == 1) or '/' not in path
        root_content = set(filter(is_root_content, archive.namelist()))
        if (len(root_content) != 1
                or not next(iter(root_content)).endswith('/')):
            raise RuntimeError('Archive must contain one root directory')
        plugin_dir = PurePosixPath(next(iter(root_content)))
        try:
            entry_point = plugin_dir / ENTRY_POINT
            _ = archive.getinfo(str(entry_point))
        except KeyError:
            raise RuntimeError(f'Module must contain an entry point: {ENTRY_POINT}')
        try:
            manifest_path = plugin_dir / MANIFEST_FILE_NAME
            manifest_path_info = archive.getinfo(str(manifest_path))
        except KeyError:
            raise RuntimeError('Module must contain a manifest file')
        plugin_id = plugin_dir.name
        with archive.open(manifest_path_info) as manifest_file:
            text_fileobject = _to_text_fileobject(manifest_file)
            result = self.read_config(text_fileobject)
        result['id'] = plugin_id
        return result

    @staticmethod
    def supports(location_path: LocationPath) -> bool:
        return location_path.endswith(_ARCHIVE_SUFFIX)

    @staticmethod
    def install(package_file, destination):
        with zipfile.ZipFile(package_file, 'r') as package:
            package.extractall(destination)
