"""
Класс страницы wizard для заполнения параметров открытия.
"""
from PySide2.QtCore import Qt
from PySide2.QtWidgets import QWizardPage, QVBoxLayout, QLabel, QPushButton, QLineEdit, QDialog
from axipy import CoordSystem, ChooseCoordSystemDialog, tr

from ru_axioma_gis_vector_converter import jsonloader
from ru_axioma_gis_vector_converter.wizardconverter import WizardConverter, create_hbox


class OpenSettingsPage(QWizardPage):
    """
    Страница выбора входной СК и параметров открытия входных файлов.
    """
    def __init__(self, parent: WizardConverter) -> None:
        super().__init__(parent)
        self.parent = parent

        # Вертикальная разметка страницы
        self.vbox = QVBoxLayout()
        self.setLayout(self.vbox)

        label = QLabel(tr("Формат входных файлов"))
        label.setStyleSheet("font-weight: bold")
        self.inp_format_label = QLabel()
        self.vbox.addWidget(label)
        self.vbox.addWidget(self.inp_format_label)

        # Заголовок СК
        title_label = QLabel(tr("Входная система координат"))
        title_label.setStyleSheet("font-weight: bold")
        self.vbox.addWidget(title_label)
        # Отображение и выбор СК
        self.line_edit = QLineEdit()
        self.line_edit.setReadOnly(True)
        btn = QPushButton("...", clicked=self.choose_in_coord, objectName="btn_choose_in_coord")
        btn.setMaximumWidth(20)
        hbox = create_hbox(self.line_edit, btn)
        hbox.setAlignment(Qt.AlignLeft)
        self.vbox.addLayout(hbox)

        self.form_view = None

    def initializePage(self) -> None:
        """
        Инициализация страницы. Считывание входной СК и загрузка формы для определенного входного формата gdal
        и для настроек открытия.
        """
        self.inp_format_label.setText("{}".format(self.parent.drivers[self.parent.inp_gdal_format]["name"]))
        # Синхронизация с внутр. значением при инициализации страницы
        self.line_edit.setText(self.parent.inp_cs.name)
        # Загрузка и добавление формы с параметрами открытия
        self.form_view = jsonloader.get_form_view(self, self.parent.inp_gdal_format, ["open_options"])
        self.vbox.addWidget(self.form_view)

    def cleanupPage(self) -> None:
        """
        Вызывается при нажатии кнопки назад. Удаляет добавленную форму.
        """
        self.form_view.setParent(None)

    def validatePage(self) -> True:
        """
        Считывает польз. ввод с формы при нажатии кнопки далее.
        """
        form = self.form_view.widget()
        self.parent.open_options, _, _ = self.parent.read_form(form, self.parent.inp_gdal_format)
        return True

    def dynamic_form_validation(self) -> None:
        """
        Вызов статического метода валидации формы parent.
        """
        self.parent.dynamic_form_validation(self)

    def choose_in_coord(self) -> None:
        """
        Выбор входной СК при нажатии кнопки "...".
        """
        # Обработка декартовых СК
        if self.parent.inp_cs.non_earth:
            cs = CoordSystem.from_wkt(self.parent.inp_cs.wkt)
        else:
            cs = CoordSystem.from_proj(self.parent.inp_cs.proj)

        # Диалог выбора СК
        dlg = ChooseCoordSystemDialog(cs)
        dlg.setModal(True)
        if dlg.exec_() == QDialog.Accepted:
            cs = dlg.chosenCoordSystem()
            self.parent.inp_cs = cs
            self.line_edit.setText(self.parent.inp_cs.name)
