"""
Точка входа в модуль.
"""
from pathlib import Path

from osgeo import gdal
from PySide2.QtWidgets import QMessageBox
from axipy import AxiomaPlugin, Position, tr
from axipy.app import mainwindow

from ru_axioma_gis_vector_converter import jsonloader
from ru_axioma_gis_vector_converter.wizardconverter import WizardConverter

json_name = "drivers.json"  # Имя файла json, где находяться параметры векторных драйверов


class Plugin(AxiomaPlugin):
    """
    Вспомогательный класс для корректной загрузки и выгрузки модуля в ГИС "Аксиома".
    """
    def load(self) -> None:
        """
        Метод загрузки плагина.
        """
        # Создание действия
        path = str(Path(__file__).parent / "icon" / "Translate-01.png")  # Путь к иконке действия
        self.__action = self.create_action(
            tr("Конвертер"),  # Название
            icon=path,  # Иконка
            on_click=self.show_plugin_window,  # Действие по клику
        )
        self.__action.action.setToolTip(tr("Конвертер векторных форматов"))
        # Определение позиции
        position = Position(tr("Основные"),
                            tr("Команды"))
        # Добавление действия на позицию
        position.add(self.__action)

        self.plugin_window = None  # Обнуление окна плагина

    def unload(self) -> None:
        """
        Метод выгрузки плагина.
        """
        # Удаление действия
        self.__action.remove()

        if self.plugin_window is not None:  # Если окно плагина существует
            self.plugin_window = None  # Обнулить окно плагина

    def show_plugin_window(self) -> None:
        """
        Функция показа окна плагина.
        """
        if self.plugin_window is None:  # Если окно плагина (wizard) не существует
            lang = AxiomaPlugin.language.fget(self)
            try:  # Чтение из json
                drivers = jsonloader.load(json_name, lang)  # Загрузка словаря из json с текущей локализацией
            except Exception as e:  # Если возникло исключение, показывается диалог. окно с сообщением об ошибке
                m_box = QMessageBox(
                    QMessageBox.Warning,  # Значок
                    tr("Ошибка"),  # Заголовок окна
                    "{}: {}".format(tr("Ошибка чтения json"), e),  # Сообщение
                    buttons=QMessageBox.Ok,
                    parent=mainwindow.qt_object(),
                )
                m_box.exec_()
            else:  # Если ошибок не возникло
                # Создание окна плагина (wizard)
                self.plugin_window = WizardConverter(mainwindow.qt_object(), drivers)
                self.plugin_window.load_pages()  # Загрузка страниц wizard

        # Если окно плагина инициализировалось
        if self.plugin_window:
            # Запомнить состояние режима исключений gdal
            self.plugin_window.use_exceptions = gdal.GetUseExceptions()
            # Показ окна плагина
            self.plugin_window.show()
