

from typing import Optional, List, Union

from PySide2.QtGui import QMouseEvent, QPaintEvent, QPainter, QKeyEvent
from PySide2.QtCore import QPoint, QPointF, Qt

import axipy
from axipy.gui import MapTool
from axipy.interface import AxiomaInterface, Notifications
from axipy.utl import Pnt, Rect
from axipy.mi import Ellipse
from axipy.da import Feature, Style

from .circle_utils import draw_lines, center_and_radius_of_circle
from .helper import editable_table_from_view

MAX_COUNT_BEFORE_FINILIZE_ARC=2

class CircleByPoints(MapTool):

    def __init__(self, iface: AxiomaInterface, title: str) -> None:
        super().__init__()
        self.__iface = iface
        self.__title = title
        self.__points = [] # type: List[Pnt]
        self.__current_point = None # type: Pnt

    def mousePressEvent(self, event: QMouseEvent) -> Optional[bool]:
        if event.button() != Qt.LeftButton:
            return self.PassEvent
        scene_pos = self.pnt_from_event(event)
        if len(self.__points) >= MAX_COUNT_BEFORE_FINILIZE_ARC:
            return
        self.__points.append(scene_pos)
        self.redraw()

    def pnt_from_event(self, event: QMouseEvent) -> Pnt:
        scene_pos = self.snap(event.pos()) # type: Union[Pnt, QPoint]
        if not self.is_snapped():
            scene_pos = self.to_scene(scene_pos) # type: Pnt
        return scene_pos

    def mouseDoubleClickEvent(self, event: QMouseEvent) -> Optional[bool]:
        point_size = len(self.__points)
        if point_size == 0 or point_size != MAX_COUNT_BEFORE_FINILIZE_ARC:
            return
        pnt = self.pnt_from_event(event)
        pnts = self.__points
        pnts.append(pnt)
        try:
            center, radius = center_and_radius_of_circle(pnts)
        except:
            error = axipy.tr("Не удалось вычислить коордианты центра окружности. "
                "Попробуйте изменить добавленные точки.")
            Notifications.push(self.__title, error, Notifications.Critical)
            self.__clear()
            return
        self.__insert_ellipce(center, radius)
        self.__clear()

    def __insert_ellipce(self, center: Pnt, radius: float):
        ellipse = Ellipse(rect=Rect(0,0,0,0), cs=self.view.coordsystem)
        ellipse.center = center
        ellipse.majorSemiAxis = radius
        ellipse.minorSemiAxis = radius
        f = Feature(geometry=ellipse, style=Style.for_geometry(ellipse))
        table = editable_table_from_view(self.view)
        if table is None: 
            return
        table.insert(f)
        Notifications.push(self.__title, axipy.tr("Окружность успешно создана"), \
            Notifications.Success)
        self.redraw()

    def __clear(self):
        self.__points.clear()
        self.__current_point = None
        self.redraw()

    def mouseMoveEvent(self, event: QMouseEvent) -> Optional[bool]:
        if len(self.__points) == 0:
            return
        self.__current_point = self.pnt_from_event(event)
        self.redraw()

    def paintEvent(self, event: QPaintEvent, painter: QPainter):
        painter.save()
        qt_points = [] # type: List[QPointF]
        for point in self.__points:
            qt_points.append(self.to_device(point))
        if self.__current_point is not None:
            qt_points.append(self.to_device(self.__current_point))
        draw_lines(painter, qt_points)
        painter.restore()

    def unload(self):
        self.redraw()

    def keyPressEvent(self, event: QKeyEvent) -> Optional[bool]:
        if event.key() == Qt.Key_Escape:
            self.reset()
            return
        return super().keyPressEvent(event)

