"""
Класс страницы wizard для получения прогресса и результатов конвертации.
"""
from pathlib import Path
import time
from datetime import datetime
import re

from PySide2.QtCore import Signal, Slot, QObject, QTimer
from PySide2.QtWidgets import QProgressBar, QTextEdit, QVBoxLayout, QWizard, QWizardPage
from axipy import tr
from axipy.concurrent import task_manager, AxipyAnyCallableTask
from osgeo import gdal

from ru_axioma_gis_vector_converter.wizardconverter import WizardConverter, add_dot, notify


class ResultsPage(QWizardPage):
    """
    Страница результатов конвертации. При инициализации страницы собирается пользовательский ввод из внутренних
    параметров parent(QWizard), выполняется обработка исключений. Собранные параметры передается в функцию
    gdal.VectorTranslate для конвертации. Конвертация выполняется в отдельном потоке. Информация о прогрессе
    конвертации и об ошибках и предупреждениях gdal выводится в окно вывода на странице и отображается
    в полосе прогресса.
    """

    def __init__(self, parent: WizardConverter) -> None:
        super().__init__()
        self.parent = parent

        # Вертикальная разметка
        convert_vbox = QVBoxLayout()
        self.setLayout(convert_vbox)
        # Окно вывода информации
        self.text_edit = QTextEdit()
        self.text_edit.setReadOnly(True)
        convert_vbox.addWidget(self.text_edit)
        # Полоса прогресса
        self.pbar = QProgressBar(self)
        convert_vbox.addWidget(self.pbar)

        # Словарь для обработчика ошибок gdal
        self.err_dict = {2: tr("Предупреждение"),
                         3: tr("Ошибка")}

        # Для подсчета одинаковых сообщений в обработчике gdal
        self.last_err_str = None  # Последнее сообщение
        self.Skipped = None  # Количество пропущенных сообщений

        self.err_str_dict = {}

        # Внутреннее значение progress bar для драйверов с callback
        self.internal_pbar_value = None

        # Разделительная линия консоли
        self.console_line = "_" * 40 + "\n"

        # Задача выполняемая в потоке
        self.task = None

        # Сигналы для связи со слотами из потока
        self.signals = None

    def initializePage(self) -> None:
        """
        Инициализаци страницы.
        Инициализация вспомогательных параметров для конвертации.
        Сбор польз. ввода и передача списка параметров в функцию для конвертирования.
        """

        # Отключение кнопки назад
        QTimer.singleShot(0, lambda: self.parent.button(QWizard.BackButton).setEnabled(False))
        # Сброс текста кнопки отмена
        self.setButtonText(QWizard.CancelButton, self.parent.buttonText(QWizard.CancelButton))

        # Механизм пропуска одинаковых сообщений
        self.last_err_str = ""
        self.Skipped = 0

        # Механизм режимов полоски прогресса
        self.internal_pbar_value = 100
        self.pbar.setRange(0, 0)
        self.pbar.resetFormat()
        self.pbar.show()

        # Инициализация text edit
        self.text_edit.clear()

        creation_options = {
            "format": self.parent.out_gdal_format,  # Выходной формат gdal
            "srcSRS": None,  # Входная СК
            "dstSRS": None,  # Выходная СК
            "reproject": self.parent.reproject,  # Параметр перепроецирования
            "datasetCreationOptions": self.parent.dataset,  # Параметры создания набора данных
            "layerCreationOptions": self.parent.layer,  # Параметры создания слоя
            # "skipFailures": self.parent.skip_failures,
            "callback": self.callback
        }

        # Обработка декартовых СК
        if self.parent.inp_cs.non_earth:
            creation_options["srcSRS"] = self.parent.inp_cs.wkt
        else:
            creation_options["srcSRS"] = self.parent.inp_cs.proj

        if self.parent.out_cs.non_earth:
            creation_options["dstSRS"] = self.parent.out_cs.wkt
        else:
            creation_options["dstSRS"] = self.parent.out_cs.proj

        # Драйвера-исключения из общих правил
        if "FORMAT=MIF" in creation_options["datasetCreationOptions"]:
            self.parent.out_ext = ".mif"

        # Параметры, передаваемые в функцию конвертирования
        params = [
            self.parent.input_path_list,  # Список входных файлов
            [self.parent.out_path, self.parent.out_ext],  # Выходная директория, выходное расширение
            self.parent.open_options,  # Параметры открытия входных файлов
            creation_options,  # Параметры создания
        ]

        # Отладочные данные, предназначенные для пользователя
        self.text_edit.append("Входной формат: {}\n".format(self.parent.inp_gdal_format))
        self.text_edit.append("Выходной формат: {}\n".format(self.parent.out_gdal_format))
        for k, v in creation_options.items():
            if k in ("callback", "format"):
                continue
            txt = f"{k}={v}"
            self.text_edit.append(txt)
            if k == "srcSRS":
                self.text_edit.append("\n{} = {}.".format(tr("Название входной СК"), self.parent.inp_cs.name))
            elif k == "dstSRS":
                self.text_edit.append("\n{} = {}.".format(tr("Название выходной СК"), self.parent.out_cs.name))
            self.text_edit.append("")

        self.text_edit.append(self.console_line)

        gdal.DontUseExceptions()  # Отключение исключений (Все ошибки отправляются в обработчик)

        self.signals = Signals()
        self.signals.text_edit_append.connect(self.text_edit_append)
        self.signals.change_pbar_cb.connect(self.change_pbar_cb)
        self.signals.change_pbar_range.connect(self.change_pbar_range)
        self.signals.change_pbar.connect(self.change_pbar)
        self.signals.change_pbar_no_cb.connect(self.change_pbar_no_cb)

        self.task = AxipyAnyCallableTask(self.convert, params)
        self.task.with_handler(False)
        self.task.progress_handler().finished.connect(self.finished)

        task_manager.start_task(self.task)

    def cleanupPage(self) -> None:
        """
        Очистка страницы, при нажатии кнопки назад.
        Сбрасывает кнопки навигации внизу wizard.
        """
        layout = [QWizard.Stretch, QWizard.BackButton, QWizard.NextButton, QWizard.CancelButton]
        self.parent.setButtonLayout(layout)
        self.setButtonText(QWizard.NextButton, tr("Конвертировать"))

    def callback(self, *args) -> bool:
        """
        Функция обратной связи для получения прогресса конвертации.
        Если польз-ль отменил процесс конвертации, отправляет сигнал отмены в функцию VectorTranslate.
        *args в пармаетрах функции, потому что gdal при ошибке может отправить неопределенное кол-во аргументов,
        что приводит к падению программы.

        :param arg0: Значение прогресса конвертации.
        :param arg1:
        :param arg2:
        """
        try:
            arg0 = args[0]
            value = int(arg0 * 100)
            self.signals.change_pbar_cb.emit(value)

            if not self.task.progress_handler().is_canceled():
                return True
            else:
                return False

        except Exception as e:
            self.signals.text_edit_append.emit(str(e))

    def convert(self, params: list) -> None:
        """
        Функция конвертации, выполняется в отдельном потоке.
        Посылает сигналы изменения прогресса в польз. интерфейс по ходу конвертации.

        :param params: Список параметров конвертирования.
        """

        def get_time(start: datetime, end: datetime) -> str:
            """
            Получение времени между двумя промежутками типа datetime.
            На выходе строка в формате %H:%M:%S либо в долях секунды.
            """
            time_delta = end - start
            seconds = time_delta.total_seconds()
            if seconds < 1:
                return "{:.3f}s".format(seconds)
            else:
                return time.strftime("%H:%M:%S", time.gmtime(seconds))

        def handler(*args) -> None:
            """
            Обработчик ошибок gdal.
            Отправляет строковый сигнал на поле вывода в польз. интерфейс.
            *args в пармаетрах функции, потому что gdal при ошибке может отправить неопределенное кол-во аргументов,
            что приводит к падению программы.

            :param err_level: Уровень исключения gdal.
            :param err_no: Идентификатор исключения gdal.
            :param err_msg: Описание исключения gdal.
            """
            try:
                try:
                    err_level, err_no, err_msg = args[0], args[1], args[2]
                except IndexError:  # Обработка случая, когда gdal присылает нечитаемый callback при исключении
                    return None  # Заглушение этого сообщения

                err_level = self.err_dict.get(err_level, "Unexpected")

                # Если gdal выдает сообщение о том что драйвер не поддерживает callback
                if str(err_msg).startswith("Progress turned off"):
                    creation_options.pop("callback")  # Отключение callback
                    return None

                # Удаление новой строки и пробелов в конце строки
                p = re.compile(r"(\n|\s)*$")
                err_msg = p.sub("", err_msg)

                err_str = "{} {}: {}".format(err_level, err_no, err_msg)
                full_string = ""

                # Обработка частного сообщения gdal (MapInfo File to ESRI Shapefile)
                p = re.compile(r"(?P<err_level>{}) (?P<err_no>{}): {}".format(
                    self.err_dict[2], 1,
                    r"(?P<txt1>Value) (?P<value>.*) (?P<txt2>of field) (?P<field>.*) (?P<txt3>of feature) "
                    r"(?P<feature>.*) (?P<txt4>not successfully written\."
                    r"\sPossibly due to too larger number with respect to field width)"))
                if p.match(err_str):
                    full_string = err_str
                    err_str = p.sub("(?P=txt1)(?P=txt2)(?P=field)(?P=txt3)(?P=txt4)", err_str)

                if err_str not in self.err_str_dict.keys():  # Если строки ошибки нет в ключах
                    # Добавить ошибку в ключ со значением 1
                    self.err_str_dict[err_str] = {"full_string": full_string, "count": 1}
                else:  # Если строка ошибки есть в ключах
                    self.err_str_dict[err_str]["count"] += 1  # Значение по ключу увеличивается на 1
            except Exception as e:
                self.signals.text_edit_append.emit(str(e))

        def print_err_str_dict() -> None:
            """
            Вывод ошибок и предупреждений после каждого файла.
            """
            try:
                for k, v in self.err_str_dict.items():
                    self.signals.text_edit_append.emit("")

                    if v["full_string"]:
                        err_str = v["full_string"]
                    else:
                        err_str = k
                    # Точка в конце строки если строка не заканчивается на точку
                    err_str = add_dot(err_str)
                    if v["count"] > 1:
                        self.signals.text_edit_append.emit("{}".format(err_str))
                        self.signals.text_edit_append.emit("{}: {}.".format(
                            tr("Пропущено похожих сообщений"), v["count"]))
                    else:
                        self.signals.text_edit_append.emit("{}".format(err_str))
            except Exception as e:
                self.signals.text_edit_append.emit(str(e))

        def change_cs_for_mif(out_path_arg: Path) -> None:
            """
            Замена proj на prj для MapInfo File формата MIF,т.к. строка СК, формируемая gdal не вполне
            корректная (например, 9999 для долготы широты).
            """
            p = re.compile(r'^(CoordSys\s+)Earth\s+Projection\s+(?:\s*(?:\S+|"[^"]*")\s*,)*\s*(?:\S+|"[^"]*")(.*)')

            if (self.parent.out_gdal_format == "MapInfo File" and
                    "FORMAT=MIF" in creation_options["datasetCreationOptions"]):
                # Чтение из сконвертированного файла
                with open(str(out_path_arg), "r", encoding="cp1251") as f:
                    lines = f.readlines()
                # Поиск и замена строчки системы координат
                for line_indx, line in enumerate(lines):
                    # Проверка отмены конвертации
                    if self.task.progress_handler().is_canceled():
                        return None
                    if p.match(line):
                        prj = self.parent.out_cs.prj
                        lines[line_indx] = p.sub(r"\1{}\2".format(prj), line)
                        break
                # Запись обратно в сконвертированный файл
                with open(str(out_path_arg), "w") as f:
                    f.write("".join(lines))

        try:  # Основной процесс конвертации
            start_function = datetime.now()  # Время старта конвертации

            gdal.PopErrorHandler()  # Очистка обработчика ошибок gdal
            gdal.PushErrorHandler(handler)  # Инициализация обработчика

            # Распаковка пар-ов
            inp_paths, [dir_path, out_ext], open_options, creation_options = params

            # Создать директорию, без исключения, если существует
            Path.mkdir(dir_path, exist_ok=True)

            # Основной цикл
            for i, inp_path in enumerate(inp_paths):
                # Настройка полосы прогресса для работы с/без callback, и для одного файла
                if i > 1:
                    if "callback" in creation_options:
                        self.signals.change_pbar.emit()
                    else:
                        self.signals.change_pbar_no_cb.emit()
                elif i == 1:
                    self.signals.change_pbar_range.emit()  # Меняем с бесконечного на обычный range после 1ого элем.

                # Конструкция выходной строки
                out_path = dir_path.joinpath(inp_path.stem).with_suffix(out_ext)

                self.signals.text_edit_append.emit("{}: {}\n".format(tr("Входной файл"), str(inp_path)))

                self.err_str_dict.clear()  # Очистка словаря ошибок для нового файла
                # Открытие
                src_ds = gdal.OpenEx(str(inp_path), open_options=open_options)
                mode = self.parent.file_creation_mode
                if mode == 0:
                    self.signals.text_edit_append.emit("{}: {}".format(tr("Выходной файл"), str(out_path)))
                    gdal.VectorTranslate(str(out_path), src_ds, **creation_options)
                elif mode == 1:
                    layer_count = src_ds.GetLayerCount()
                    if layer_count > 1:
                        new_dir_path = dir_path / inp_path.stem
                        Path.mkdir(new_dir_path, exist_ok=True)
                        self.signals.text_edit_append.emit("{}: {}\n".format(tr("Выходная папка"), str(new_dir_path)))
                        for i in range(layer_count):
                            layer = src_ds.GetLayer(i)
                            creation_options["layers"] = layer.GetName()
                            new_out_path = new_dir_path.joinpath(layer.GetName()).with_suffix(out_ext)
                            gdal.VectorTranslate(str(new_out_path), src_ds, **creation_options)
                            self.signals.text_edit_append.emit("{}: {}\n".format(tr("Выходной файл"), str(new_out_path)))
                            change_cs_for_mif(new_out_path)
                    elif layer_count == 1:
                        self.signals.text_edit_append.emit("{}: {}".format(tr("Выходной файл"), str(out_path)))
                        gdal.VectorTranslate(str(out_path), src_ds, **creation_options)
                        change_cs_for_mif(out_path)
                elif mode == 2:
                    if i == 0:
                        if self.parent.file_name_for_append:
                            single_path = out_path.with_name(self.parent.file_name_for_append).with_suffix(out_ext)
                        else:
                            single_path = out_path
                    if i == 1:
                        creation_options["accessMode"] = "append"
                        creation_options.pop("datasetCreationOptions")
                    gdal.VectorTranslate(str(single_path), src_ds, **creation_options)
                    self.signals.text_edit_append.emit("{}: {}".format(tr("Выходной файл"), str(single_path)))

                # Закрытие
                src_ds = None
                print_err_str_dict()  # Вывод ошибок из словаря

                # Проверка отмены конвертации
                if self.task.progress_handler().is_canceled():
                    return None

                self.signals.text_edit_append.emit(self.console_line)  # Вывод разделительной линии в консоль

            # Вывод длительности конвертации
            finish_function = datetime.now()
            self.signals.text_edit_append.emit("{} ({}: {}).".format(
                tr("Конвертация завершена"),
                tr("длительность").capitalize(),
                get_time(start_function, finish_function)
            ))

        except Exception as e:
            src_ds = None
            self.signals.text_edit_append.emit(str(e))

    @Slot()
    def finished(self) -> None:
        """
        Функция вызывается при завершении выполнения задачи в потоке.
        Изменяет клавиши навигации для выбора возможности повторного конвертирования.
        Скрывает полосу прогресса.
        """
        if self.task.progress_handler().is_canceled():
            notify(tr("Конвертация файлов отменена"), 0)
            self.parent.reject()
        layout = [QWizard.Stretch, QWizard.BackButton, QWizard.NextButton, QWizard.CustomButton1, QWizard.CancelButton]
        self.parent.setButtonLayout(layout)
        self.pbar.hide()
        self.setButtonText(QWizard.CancelButton, tr("Завершить"))

    @Slot(str)
    def text_edit_append(self, txt: str) -> None:
        """
        Отправляет полученный сигнал со строкой в поле вывода пользовательского интерфейса.

        :param str txt: Строка, полученная в сигнале из потока.
        """
        self.text_edit.append(txt)

    """
    Слоты для управления полоской прогресса из потока.
    """

    @Slot()
    def change_pbar(self) -> None:
        """
        Изменяет внутреннее значение полосы прогресса.
        """
        self.internal_pbar_value += 100

    @Slot()
    def change_pbar_range(self) -> None:
        """
        Вычисляет длину списка файлов для конвертации.
        Задает разметку кратную 100 для полосы прогресса.
        """
        n = len(self.parent.input_path_list)
        self.pbar.setRange(0, n * 100)
        self.pbar.setValue(100)

    @Slot(int)
    def change_pbar_cb(self, value: int) -> None:
        """
        Движение полосы прогресса для драйверов c callback.
        :param value: Значение callback.
        """
        if value <= 100:
            i = self.internal_pbar_value + value
            self.pbar.setValue(i)

    @Slot()
    def change_pbar_no_cb(self) -> None:
        """
        Движение полосы прогресса для драйверов без callback.
        """
        self.internal_pbar_value += 100
        self.pbar.setValue(self.internal_pbar_value)


class Signals(QObject):
    """
    Сигналы для связи со слотами из потока.
    """
    text_edit_append = Signal(str)
    change_pbar_cb = Signal(int)
    change_pbar_range = Signal()
    change_pbar = Signal()
    change_pbar_no_cb = Signal()
