from axipy.cpp_cs import ShadowCoordSystem, ShadowCoordinateTransformer
from typing import Optional, Union, List
from axipy.utl import Pnt, Rect
from .unit import LinearUnit, AreaUnit, Unit

from PySide2.QtCore import QRectF, QPointF


def _internal_transform(value, func):
    if isinstance(value, QRectF) or isinstance(value, Rect):
        return Rect._rect_value_from_qt(func(Rect._rect_value_to_qt(value)))
    return Pnt._point_value_from_qt(func(Pnt._point_value_to_qt(value)))


class CoordSystem:
    """Система координат (СК). СК описывает каким образом реальные объекты на земной поверхности могут быть представлены
    в виде двумерной проекции.
    Выбор СК для представления данных зависит от конкретных исходных условий по представлению исходных данных.

    Note:
        Проверка на идентичность параметров двух СК производится простым сравнением.

    Note:
        Для получения текстового представления можно воспользоваться функцией :class:`str`.

    Поддерживается создание СК посредством следующих вариантов:

    * Из строки MapInfo PRJ :meth:`from_prj`
    * Из строки PROJ :meth:`from_proj`
    * Из строки WKT :meth:`from_wkt`
    * Из значения EPSG :meth:`from_epsg`
    * План/Схему с указанием единиц измерения и охвата :meth:`from_units`

    .. literalinclude:: /../../tests/doc_examples/test_example_cs.py
        :caption: Пример создания СК разного типа.
        :pyobject: test_run_example_cs_create
        :lines: 2-
        :dedent: 4

    .. literalinclude:: /../../tests/doc_examples/test_example_cs.py
        :caption: Проверка на идентичность координатных систем производится простым сравнением.
        :pyobject: test_run_example_cs_compare
        :lines: 2-
        :dedent: 4
    """

    def __init__(self):
        raise NotImplementedError

    @classmethod
    def _wrap(cls, shadow: ShadowCoordSystem):
        if shadow.isEmpty():
            return None
        result = cls.__new__(cls)
        result.shadow = shadow
        return result

    @staticmethod
    def factory():
        try:
            from axipy.app import csfactory_instance
            return csfactory_instance
        except ImportError:
            raise RuntimeError("axipy is not initialized")

    @property
    def description(self) -> str:
        """Краткое текстовое описание."""
        return self.shadow.description()

    @property
    def rect(self) -> Rect:
        """Максимально допустимый охват."""
        return Rect.from_qt(self.shadow.rect())

    @rect.setter
    def rect(self, r):
        self.shadow.setRect(Rect._rect_value_to_qt(r))

    @property
    def unit(self) -> LinearUnit:
        """Единицы измерения."""
        return LinearUnit._wrap(self.shadow.unit())

    def convert_to_degree(self, value: Union[Pnt, Rect, QPointF, List[QPointF], QRectF]) -> Union[Pnt, List[Pnt], Rect]:
        """Переводит из единиц измерения системы координат в градусы.

        .. literalinclude:: /../../tests/doc_examples/test_example_cs.py
            :caption: Пример.
            :pyobject: test_run_example_cs_to_degree
            :lines: 2-
            :dedent: 4
        """
        return _internal_transform(value, self.shadow.to_degree)

    def convert_from_degree(self, value: Union[Pnt, Rect, QPointF, List[QPointF], QRectF]) -> Union[Pnt, List[Pnt], Rect]:
        """Переводит из градусов в единицы измерения системы координат."""
        return _internal_transform(value, self.shadow.from_degree)

    @property
    def epsg(self) -> Optional[int]:
        """
        Значение EPSG если существует для данной системы координат,
        иначе None.
        """
        return self.factory().toEpsg(self.shadow)

    @property
    def proj(self) -> str:
        """Строка PROJ или пустая строка, если аналога не найдено.
        """
        return self.factory().toProj(self.shadow)

    @property
    def prj(self) -> str:
        """Строка prj формата MapBasic или пустая строка, если аналога не найдено.
        """
        return self.factory().toPrj(self.shadow)

    @property
    def wkt(self) -> str:
        """Строка WKT или пустая строка, если аналога не найдено.
        """
        return self.factory().toWkt(self.shadow)

    @property
    def non_earth(self) -> bool:
        """Является ли данная СК декартовой."""
        return self.shadow.is_non_earth()

    @property
    def lat_lon(self) -> bool:
        """Является ли данная СК широтой/долготой."""
        return self.shadow.is_lat_lon()

    @property
    def name(self) -> str:
        """Наименование системы координат."""
        return self.shadow.name()

    @property
    def semi_major(self) -> float:
        """Большая полуось."""
        return self.shadow.semi_major()

    @property
    def semi_minor(self) -> float:
        """Малая полуось."""
        return self.shadow.semi_minor()

    @property
    def inv_flattening(self) -> float:
        """Полярное сжатие."""
        return self.shadow.inv_flattening()


    @classmethod
    def from_epsg(cls, code: int) -> "CoordSystem":
        """Создает координатную систему по коду EPSG.

        See also:
            Подробнее см. `EPSG <https://epsg.org>`_

        Args:
            code: Стандартное значение EPSG.
        """
        return cls._wrap(cls.factory().fromEpsg(code))

    @classmethod
    def from_wkt(cls, wkt: str) -> "CoordSystem":
        """Создает координатную систему из строки WKT.

        See also:
            Подробнее см. `WKT <https://en.wikipedia.org/wiki/Well-known_text_representation_of_coordinate_reference_systems>`_

        Args:
            wkt: Строка WKT.
        """
        return cls._wrap(cls.factory().fromWkt(wkt))

    @classmethod
    def from_proj(cls, proj: str) -> "CoordSystem":
        """Создает координатную систему из строки proj.

        See also:
            Подробнее см. `PROJ <https://proj.org/usage/quickstart.html>`_

        Args:
            proj: Строка proj.
        """
        return cls._wrap(cls.factory().fromProj(proj))

    @classmethod
    def from_prj(cls, prj: str) -> "CoordSystem":
        """Создает координатную систему из строки MapBasic.

        See also:
            Подробнее см. `PRJ <http://webhelp.infovista.com/Planet/62/Subsystems/MapInfo/Content/pro_help/coordinates/addprojectiontomapinfow.html>`_

        Args:
            prj: Строка MapBasic. Допустима короткая нотация.

        .. literalinclude:: /../../tests/doc_examples/test_example_cs.py
            :caption: Пример.
            :pyobject: test_run_example_cs_create_prj
            :lines: 2-
            :dedent: 4

        """
        return cls._wrap(cls.factory().fromPrj(prj))

    @classmethod
    def from_units(cls, unit: LinearUnit, rect: Optional[Union[Rect, QRectF]] = Rect(-10000, -10000, 10000, 10000)) -> "CoordSystem":
        """Создает декартову систему координат.

        Args:
            unit: Единицы измерения системы координат.
            rect: Охват системы координат.

        .. literalinclude:: /../../tests/doc_examples/test_example_cs.py
            :caption: Пример.
            :pyobject: test_run_example_cs_create_prj_ne
            :lines: 2-
            :dedent: 4
        """
        return cls._wrap(cls.factory().fromUnits(unit.shadow, Rect._rect_value_to_qt(rect) if rect else None))

    @classmethod
    def from_string(cls, string: str) -> "CoordSystem":
        """Создает систему координат из строки. 

        Строка состоит из двух частей: префикса и строки представления СК. Возможные значения префиксов: «proj», «wkt», «epsg», «prj».

        Args:
            string: Строка.
        """
        return cls._wrap(cls.factory().fromString(string))

    def __eq__(self, other) -> bool:
        return other is not None and self.shadow.isEqual(other.shadow)

    def __str__(self):
        return self.shadow.get_str()

    @classmethod
    def current(cls):
        """Текущая установленная система координат"""
        return cls._wrap(cls.factory().current_cs())

    @classmethod
    def set_current(cls, coordsystem : 'CoordSystem'):
        """Устанавливает новую текущую систему координат
        
        Args:
            cs: Новое значение системы координат.
        """
        cls.factory().set_current_cs(coordsystem.shadow)
