"""Модуль приложения.

Данный модуль является основным модулем приложения.

Attributes:
    mainwindow (MainWindow): Готовый экземпляр главного окна ГИС Аксиома.
"""

from PySide2.QtCore import QVersionNumber
from axipy.cpp_app import *
from axipy.cpp_core_core import ShadowInformation
from .main_window import *


class Version:
    """Информация о версии ГИС Аксиома.
    
    Пример использования::
    
        from axipy.app import Version
        print('Версия:', Version.string())
    """

    @staticmethod
    def number() -> int:
        """Возвращает версию в виде одного числа, в котором каждый сегмент
        располагается в отдельном байте."""
        return ShadowInformation.currentVersion()

    @staticmethod
    def qtFormat() -> QVersionNumber:
        """Возвращает версию в Qt формате."""
        return QVersionNumber(*Version.segments())
        
    @staticmethod
    def segments() -> tuple:
        """Возвращает кортеж чисел - сегменты версии: (`major`, `minor`, `patch`)."""
        v = Version.number()
        return (v & 0xFF0000) >> 16, (v & 0xFF00) >> 8, v & 0xFF
        
    @staticmethod
    def string() -> str:
        """Возвращает версию в виде строки."""
        return ShadowInformation.currentVersionString()


class Notifications:
    """Отправление уведомлений в виде всплывающего окна с его последующей регистрацией в окне уведомлений.
    """

    Information = 0
    Warning = 1
    Critical = 2
    Success = 3

    @staticmethod
    def push(title: str, text: str, type_mesage: int = 0):
        """Отправляет уведомление.

        Args:
            title: Заголовок
            text: Текст сообщения.
            type_mesage: Тип сообщения. В зависимости от типа сообщения в окне уведомлений оно помечается соответствующим цветом.

        .. csv-table:: Допустимые значения для типа сообщения:
            :header: Атрибут, Наименование

            `Information`, Информационное сообщение. Устанавливается по умолчанию
            `Warning`, Предупреждение
            `Critical`, Критическая ошибка
            `Success`, Успешное выполнение процесса

        Пример::

            from axipy.app import Notifications
            Notifications.push('Предупреждение', 'Сообщение', Notifications.Warning)
        """
        from axipy.app import ShadowNotifications
        return ShadowNotifications.push(title, text, type_mesage)


# fix sphinx inheritance and typehints
from axipy._util import fixup_module_metadata
fixup_module_metadata(__name__, globals())
del fixup_module_metadata
