"""
Класс wizard и вспомогательные функции для страниц wizard.
"""
from pathlib import Path

from PySide2.QtCore import Qt, QStandardPaths
from PySide2.QtGui import QIcon
from PySide2.QtWidgets import QWizard, QComboBox, QLineEdit, QCheckBox, QDateEdit, QSpinBox, QHBoxLayout, QWizardPage, \
    QWidget, QMessageBox
import axipy
from axipy import tr, CoordSystem, app
from osgeo import gdal

from ru_axioma_gis_vector_converter import jsonloader


class WizardConverter(QWizard):
    """
    В классе создается Wizard, определяются его параметры, происходит добавление страниц и настройка действий кнопок
    навигации внизу страниц ("Далее", "Назад" и т.д.).
    В классе находятся переменные, в которые собирается пользовательский ввод со всех страниц, для последующей передачи
    в функцию конвертирования.
    """
    def __init__(self, parent, drivers):
        """
        Выполняется при создании экземпляра класса.
        """
        super().__init__(parent)

        self.drivers = drivers  # Словарь параметров драйверов из json

        # Название для уведомлений
        self.notification_title = tr('Модуль "Конвертер"')

        # (значения сбрасываются при перезагрузке плагина и перезапуске аксиомы)
        self.documents_path = Path(QStandardPaths.writableLocation(QStandardPaths.DocumentsLocation))
        self.inp_dir_path = self.documents_path  # Входная папка
        # Путь к директории сохранения (по умолч.)
        self.out_dir_path_default = self.documents_path / tr("Аксиома Конвертер")
        # Выходная папка
        self.out_dir_path = self.out_dir_path_default
        self.out_path = None  # Полный путь (Выходная папка + папка результатов)
        # Выходной формат
        self.out_gdal_format = None  # Задается при валидации начальной страницы добавления файлов
        self.skip_failures = False  # Пропуск исключений
        # Значение СК по умолчанию
        self.default_cs = CoordSystem.from_proj("+proj=longlat +ellps=WGS84 +towgs84=0,0,0,0,0,0,0 +no_defs")

        # (значения сбрасываются при рестарте и при закрытии окна)
        self.input_path_list = None  # Список всех входных файлов
        self.validated_files = None  # Список для валидации входных файлов
        self.inp_gdal_format = None  # Формат gdal входных файлов
        self.inp_gui_format = None  # Строка, нужна для проверки вход. файлов на одинак. формат
        # Выходное расширение
        self.out_ext = None
        # Система координат
        self.inp_cs = None  # Входная СК
        self.out_cs = None  # Выходная СК
        self.reproject = None  # Флаг перепроецирования СК
        # Настройки конвертирования вводимые пользователем
        self.open_options, self.dataset, self.layer = [None, None, None]

        self.file_creation_mode = None  # Режим создания файлов
        self.file_creation_mode_list = [
            tr("Аналогично исходному"),
            tr("Каждую таблицу в отдельный файл"),
            tr("Собрать все таблицы в один файл")
        ]
        self.file_name_for_append = None

        self.use_exceptions = None

        # Настройки главного окна
        self.setWindowTitle(tr("Конвертер"))  # Отображаемое имя окна
        # Фикс. размер окна
        self.wizard_width = 550
        self.wizard_height = 700
        self.setFixedSize(self.wizard_width, self.wizard_height)
        # Размер шрифта
        font = self.font()
        font.setPointSize(9)
        self.setFont(font)
        # Модальность окна
        self.setModal(True)
        icon = QIcon(str(Path(__file__).parent / "icon" / "Translate-03.png"))
        self.setWindowIcon(icon)

        # Включение и настройка Пользов. Кнопки 1
        self.setOption(QWizard.HaveCustomButton1)
        self.setButtonText(self.CustomButton1, tr("Сконвертировать ещё"))
        self.button(QWizard.CustomButton1).clicked.connect(self.custom_btn1_press)

        # Флаги
        self.setWindowFlags(self.windowFlags() & ~Qt.WindowContextHelpButtonHint)  # Выключение значка "?"
        # Стиль wizard
        self.setWizardStyle(QWizard.ClassicStyle)
        # self.setWizardStyle(QWizard.ModernStyle)
        # self.setWizardStyle(QWizard.MacStyle)
        # self.setWizardStyle(QWizard.AeroStyle)

        self.results_page = None  # Ссылка на страницу результатов для прерывания процесса конвертации
        self.m_box_q = QMessageBox(
                    QMessageBox.Question, tr("Вопрос"),
                    tr("Прервать процесс конвертации?"),
                    buttons=QMessageBox.Yes | QMessageBox.No, parent=self)

    def load_pages(self) -> None:
        """
        Функция инициализации страниц Wizard. Создана чтобы избежать циклического импорта.
        """
        # Импорт вынесен в функцию, чтобы избежать циклического импорта
        from ru_axioma_gis_vector_converter.wizardpage import WelcomePage, OpenSettingsPage, CreationPage, ResultsPage

        # Создание начальной страницы
        welcome_page = WelcomePage(self)
        # Создание страницы с параметрами открытия входных файлов
        open_settings_page = OpenSettingsPage(self)
        # Создание страницы с параметрами создания выходных файлов
        creation_page = CreationPage(self)
        # Создание страницы результатов конвертирования
        self.results_page = ResultsPage(self)

        # Добавление страниц в wizard
        for elem in (welcome_page, open_settings_page, creation_page, self.results_page):
            self.addPage(elem)

    def reject(self) -> None:
        """
        Отмена задачи из потока при закрытии окна.
        """
        page = self.currentPage()  # Текущая страница wizard
        if hasattr(page, "task"):  # Если у тек. страницы есть аттрибут
            task = self.currentPage().task
            if task is not None:  # Если аттрибут инициализирован

                if not task.progress_handler().is_canceled():  # Если задача не отменена
                    if page == self.results_page and not task.progress_handler().is_finished():
                        self.m_box_q.exec_()
                        if self.m_box_q.result() == QMessageBox.No:
                            return None  # Отклонить закрытие окна
                    task.progress_handler().cancel()  # Отменить

                if task.progress_handler().is_running():  # Если задача еще выполняется
                    # Найти существующий pbar
                    if hasattr(page, "pbar"):
                        pbar = getattr(page, "pbar")
                        pbar.setRange(0, 100)
                        pbar.setValue(0)
                        pbar.setFormat(tr("Идет прерывание процесса..."))
                        pbar.show()
                    if hasattr(page, "canceled_by_dlg"):
                        if page.canceled_by_dlg:
                            super().reject()
                    else:
                        return None  # Отклонить закрытие окна

        # Возвращение значения режима исключений gdal в состояние до запуска окна плагина
        set_use_exeptions = {
            0: gdal.DontUseExceptions,
            1: gdal.UseExceptions
        }
        if self.use_exceptions in set_use_exeptions:
            set_use_exeptions[self.use_exceptions]()
        gdal.PopErrorHandler()  # Очистка обработчика ошибок gdal
        # Закрыть окно
        super().reject()

    def custom_btn1_press(self) -> None:
        """
        Перезапуск wizard с 1 страницы при нажатии кнопки "Сконвертировать еще".
        """
        self.restart()

    def dynamic_form_validation(self, obj: QWizardPage) -> None:
        """
        Проверяет совместимость выбранных пользователем параметров во время работы программы.
        Совместимость параметров проверяется на форме построенной из json. Формы загружаются на странице выбора
        пар-ов открытия и пар-ов создания. Привязывается только к QComboBox и QCheckBox.

        :param QWizardPage obj: Страница Wizard на которой находиться форма, которую нужно проверять на правильность
            польз. ввода.
        """
        # получение пути до sender в json
        sender = obj.sender()
        # Индекс типа настройки
        option_indx = int(sender.objectName()[:1])
        # Имя настройки по индексу
        option_type = jsonloader.options_types_list[option_indx]
        # Имя параметра
        s_name = sender.objectName()[1:]

        # Получение пути data/gdal_format/Имя настройки/Имя параметра
        data = None
        try:
            if option_indx in [1, 2]:
                data = self.drivers[self.out_gdal_format][option_type][s_name]
            elif option_indx in [0]:
                data = self.drivers[self.inp_gdal_format][option_type][s_name]
        except KeyError:
            return None

        # Тип виджета, пославшего сигнал
        assert type(data) is dict
        widget_type = data["type"]
        widget_keys = data.keys()

        # Получение текущего значения виджета, пославшего сигнал
        value = None
        if "disable" in widget_keys:  # Если у пославшего виджета есть ключ "disable"
            if widget_type == "QCheckBox":
                bool_ = bool(sender.checkState())
                value = jsonloader.get_key(data["values"], bool_)
            elif widget_type == "QComboBox":
                txt = sender.currentText()
                value = jsonloader.get_key(data["values"], txt)
            assert value

            # получение виджетов формы
            widgets_list = obj.form_view.widget().children()
            for widget in widgets_list:  # Для всех виджетов формы
                if widget.objectName() != "":  # Если объектное имя не пустое
                    w_name = widget.objectName()[1:]  # Получение имени параметра
                    w_opt_type = jsonloader.options_types_list[int(widget.objectName()[:1])]
                    data = self.drivers[self.out_gdal_format][w_opt_type][w_name]  # Словарь параметра
                    if "disable_if" in data.keys():  # Если у пар-ра есть ключ "disable_if"
                        if s_name in data["disable_if"]:  # Если имя пар-ра отправ. есть в списке по ключу "disable_if"
                            if value in data["disable_if"][s_name]:  # Если значение пар-ра отправителя есть в списке
                                widget.setEnabled(False)  # Выкл. виджет
                            elif not widget.isEnabled():  # Если виджет выключен
                                widget.setEnabled(True)  # Вкл. виджет

    def read_form(self, form: QWidget, d_name: str) -> tuple:
        """
        Собирает данные с формы в списки по имени настроек (open_options, dataset, layer).

        :param form: QWidget с разметкой QFormLayout.
        :param d_name: Имя драйвера в формате gdal.
        :return: Списки (Пар-ры открытия, пар-ры набора данных, пар-ры слоя)
        """
        # Пустые списки пар-ов для заполнения данными из формы
        open_options, dataset, layer = [list(), list(), list()]
        settings_dict = {"open_options": open_options, "dataset": dataset, "layer": layer}

        # Список виджетов формы
        widgets_list = form.children()
        # Цикл по виджетам формы
        for widget in widgets_list:
            if (
                    isinstance(widget, (QComboBox, QLineEdit, QCheckBox, QDateEdit, QSpinBox)) and
                    widget.objectName() not in ["in_line_edit", "out_line_edit"]  # Пропуск полей выбора СК
            ):
                option_type = jsonloader.options_types_list[int(widget.objectName()[:1])]
                name = widget.objectName()[1:]
                gdal_value = None

                if isinstance(widget, QComboBox):
                    values = self.drivers[d_name][option_type][name]["values"]
                    txt = widget.currentText()
                    if txt not in ["По умолчанию", "Default"]:
                        gdal_value = jsonloader.get_key(values, txt)

                elif isinstance(widget, QLineEdit):
                    x = widget.text()
                    # Если поле ввода не пустое и не состоит из пробелов
                    if x != "" and not (" " in x and x == len(x) * x[0]):
                        gdal_value = x

                        if name == "semicolon":  # Поддержка параметров через ;
                            settings_dict[option_type].extend(gdal_value.split(";"))
                            continue

                elif isinstance(widget, QCheckBox):
                    values = self.drivers[d_name][option_type][name]["values"]
                    check_state = widget.isChecked()
                    gdal_value = jsonloader.get_key(values, check_state)

                elif isinstance(widget, QDateEdit):
                    gdal_value = str(widget.date().toPython())

                elif isinstance(widget, QSpinBox):
                    gdal_value = str(widget.value())

                if widget.isEnabled() and gdal_value:
                    elem = f"{name}={gdal_value}"
                    settings_dict[option_type].append(elem)

        return open_options, dataset, layer


def create_hbox(*widgets) -> QHBoxLayout:
    """
    Создает горизонтальную разметку с входными виджетами.

    :param widgets: Список виджетов.
    :return: Горизонтальная разметка (QHBoxLayout).
    """
    hbox = QHBoxLayout()
    for widget in widgets:
        hbox.addWidget(widget)
    return hbox


def add_dot(str_: str):
    """
    Добавляет точку в конце строки, если ее нет.
    """
    if not str_.endswith("."):
        str_ += "."
    return str_


def notify(text: str, msg_type: int = 1) -> None:
    """
    Функция отправляет уведомления с одним и тем же заголовком.
    Warning по умолчанию.

    :param text:
    :param msg_type: 0 Info, 1 Warning
    """
    axipy.app.Notifications.push(
        title=tr("Модуль \"Конвертер\""),
        text=text,
        type_mesage=msg_type
    )
