
import sys
import traceback

import axipy
from axipy.app import Notifications
from axipy.da import provider_manager, data_manager

from axipy.interface import AxiomaInterface
from axipy.sql import geometry_uid, style_uid
from axipy.da import Table, Feature, Style, DefaultKeys, GeometryCollection
from axipy.cs import CoordSystem
from .helper import ensure_editable_table, text_to_float, \
     ensure_sql_init_for_bg_thread
from PySide2.QtUiTools import QUiLoader
from PySide2.QtWidgets import QWidget, QGridLayout
from PySide2.QtGui import QDoubleValidator
from axipy.concurrent import task_manager, AxipyProgressHandler, ProgressSpecification, \
    ProgressGuiFlags
from axipy.gui import view_manager, selection_manager, AxipyAcceptableActiveToolHandler
from typing import List


MIN_POINTS_COUNT = 2


class VoronovSettings:
    """ Настраиваемые параметры, которые можно передать при
    вызове функции из SQL  """

    def __init__(self) -> None:
        # Значения по умолчанию взяты из документации к SpatiaLite
        self.edges_only = False  # type: bool
        self.frame_extra_size = 5  # type: int
        self.tolerance = 0  # type: float


class PolygonsVoronogoSettingsWidget(QWidget):

    def __init__(self, ui_file: str) -> None:
        super().__init__()
        self.__ui = QUiLoader().load(ui_file, self)
        grid_layout = QGridLayout()
        grid_layout.addWidget(self.__ui)
        self.setLayout(grid_layout)
        self.init_tolerance_le()

    def init_tolerance_le(self):
        validator = QDoubleValidator(
            bottom=sys.float_info.min,
            top=sys.float_info.max,
            decimals=15)
        validator.setNotation(QDoubleValidator.StandardNotation)
        self.__ui.tolerance_le.setValidator(validator)
        self.__ui.tolerance_le.setToolTip(axipy.tr(
            "Если координаты точек отличаются друг от друга меньше чем на введенное "
            "значение, то они не будут учитываться при выполнении операции."))

    def update_coordinate_unit(self, cs: CoordSystem):
        self.__ui.unit_tolerance_label.setText(cs.unit.localized_name)

    def get_settings(self) -> VoronovSettings:
        result = VoronovSettings()
        result.edges_only = self.__ui.edges_only.isChecked()
        result.frame_extra_size = self.__ui.frame_extra_size_sb.value()
        if self.__ui.tolerance_gb.isChecked():
            result.tolerance = text_to_float(self.__ui.tolerance_le.text(),
                                             self.__ui.tolerance_le.locale())
        return result


class PolygonsVoronogo:
    """ Построение полигонов Вороного по выбранным точкам """

    def __init__(
            self,
            title: str,
            iface: AxiomaInterface,
            observer_id: DefaultKeys.Key) -> None:
        self.title = title
        self.iface = iface
        ui_file = self.iface.local_file('ui', 'VoronovPolygon.ui')
        panel_service = self.iface.active_tool_panel()
        widget = PolygonsVoronogoSettingsWidget(ui_file)
        self.__tool_panel = panel_service.make_acceptable(
            title=self.title,
            observer_id=observer_id,
            widget=widget) # type: AxipyAcceptableActiveToolHandler
        self.__tool_panel.accepted.connect(self.__run_in_background_thread)

        def update_coordinate_unit():
            widget.update_coordinate_unit(selection_manager.table.coordsystem)
        self.__tool_panel.activated.connect(update_coordinate_unit)
        self.__tmp_tables = []  # type: List[Table]

    def __close_tmp_tables(self):
        for table in reversed(self.__tmp_tables):
            if table is None:
                continue
            table.restore()
            table.close()

    def on_triggered(self) -> None:
        self.__tool_panel.activate()
        ensure_sql_init_for_bg_thread()

    def __run_in_background_thread(self):
        self.__tool_panel.disable()
        settings = self.__tool_panel.widget.get_settings()
        try:
            spec = ProgressSpecification(description=self.title, flags=ProgressGuiFlags.CANCELABLE)
            task_manager.run_and_get(spec, self.__make_voronov_polygons, \
                settings, ensure_editable_table())
        except BaseException:
            traceback.print_exc()
        finally:
            self.__tool_panel.try_enable()

    def __make_voronov_polygons(
            self,
            ph: AxipyProgressHandler,
            settings: VoronovSettings,
            editable_table: Table):
            definition = {
                'src': '', 
                'schema': selection_manager.get_as_table().schema,
                }
            selection = self.__make_table(definition)
            selection.name = 'selection_table_voronoi'
            selection.insert(selection_manager.get_as_table().items())
            try:
                return self.__make_voronov_polygons_impl(ph, settings, selection,\
                    editable_table)
            except BaseException as e:
                tb = sys.exc_info()[2]
                raise e.with_traceback(tb)
            finally:
                self.__close_tmp_tables()

    def __make_voronov_polygons_impl(
            self,
            ph: AxipyProgressHandler,
            settings: VoronovSettings,
            selection_table: Table,
            editable_table: Table):
        view_cs = view_manager.active.coordsystem  # type: CoordSystem
        # Преобразуем геометрии к проекции карты, чтобы можно было построить
        # Полигоны Вороного в проекции карты
        selection_table = self.__reproject_selection_table(
            selection_table, view_cs)
        query_text = f"SELECT AsBinary(VoronojDiagram(Collect(FromAxiGeo({geometry_uid})), \
            {False}, {settings.frame_extra_size}, {settings.tolerance})) \
            FROM {selection_table.name} WHERE Str$({geometry_uid}) == 'Point'"
        table = data_manager.query_hidden(query_text, dialect='sqlite')
        if table is None: 
            self.__notify_about_sql_error(query_text)
            return
        feature = next(table.items())
        if not feature.has_geometry():
            self.__nofify_about_suitable_types()
            return
        ph.raise_if_canceled()
        collection = feature.geometry
        ph.prepare_to_write_changes()
        ph.set_description(axipy.tr("Копируем аттрибутивную информацию"))
        self.__copy_attributes_with_sql(
            ph,
            collection,
            selection_table,
            editable_table,
            settings)

    def column_names(self, attr_table: Table, geom_table: Table) -> List[str]:
        res_attrs = []
        for attr in attr_table.schema:
            res_attrs.append(f"{attr_table.name}.{attr.name}")
        res_attrs.append(f"{geom_table.name}.{geometry_uid}")
        res_attrs.append(f"{geom_table.name}.{style_uid}")
        return ', '.join(res_attrs)

    def __insert_to_table(
            self,
            handler: AxipyProgressHandler,
            editable_table: Table,
            polygons_with_attr_table: Table,
            use_linestring: bool):
        table_size = polygons_with_attr_table.count()
        features = [] # type: List[Feature]
        if use_linestring:
            features = self.__linestring_features(handler, polygons_with_attr_table)
        else:
            features = [f for f in polygons_with_attr_table.items()]
        handler.prepare_to_write_changes()
        editable_table.insert(features)
        self.__notify_about_finish(table_size)

    def __linestring_features(self, ph: AxipyProgressHandler, table: Table) -> List[Feature]:
        table_size = table.count()
        ph.set_description(axipy.tr("Преобразуем полигоны в полилинии"))
        ph.set_max_progress(table_size)
        ph.set_progress(0)

        def to_linestring(f: Feature) -> Feature:
            ph.raise_if_canceled()
            f.geometry = f.geometry.to_linestring()
            f.style = Style.for_geometry(f.geometry)
            ph.add_progress(1)
            return f
        return [to_linestring(f) for f in table.items()]

    def __notify_about_finish(self, count_features: int):
        if count_features == 0:
            self.iface.notifications.push(
                self.title,
                axipy.tr("Не удалось добавить ни один полигон Вороного"),
                Notifications.Critical)
        else:
            self.iface.notifications.push(self.title, axipy.tr(
                f"Было добавленно записей: {count_features}"), Notifications.Success)

    def __reproject_selection_table(
            self,
            table: Table,
            view_cs: CoordSystem) -> Table:
        if table.coordsystem == view_cs:
            return table
        schema = table.schema
        schema.coordsystem = view_cs
        definition = {'src': '', 'schema': schema}
        table_in_view_cs = self.__make_table(definition)
        reprojected_features = []  # type: List[Feature]
        for f in table.items():
            f.geometry = f.geometry.reproject(view_cs)
            reprojected_features.append(f)
        table_in_view_cs.insert(reprojected_features)
        return table_in_view_cs

    def __copy_attributes_with_sql(self, ph: AxipyProgressHandler,
                                   collection: GeometryCollection,
                                   selection_table: Table,
                                   editable_table: Table,
                                   settings: VoronovSettings):
        polygon_features = []  # type: List[Feature]
        for geom in collection:
            polygon_features.append(
                Feature(
                    geometry=geom.clone(),
                    style=Style.for_geometry(geom)))
        definition = {'src': '', 'schema': selection_table.schema}
        polygons_table = self.__make_table(definition)
        polygons_table.insert(polygon_features)
        columns = self.column_names(selection_table, polygons_table)
        query_text = f"SELECT {columns} FROM {polygons_table.name},{selection_table.name}" \
            f" WHERE {selection_table.name}.{geometry_uid}" \
            f" within {polygons_table.name}.{geometry_uid}"
        polygons_with_attr_table = data_manager.query_hidden(
            query_text)  # type: Table
        self.__tmp_tables.append(polygons_with_attr_table)
        if polygons_with_attr_table is None:
            return
        ph.raise_if_canceled()
        self.__insert_to_table(
            ph,
            editable_table,
            polygons_with_attr_table,
            settings.edges_only)

    def __make_table(self, definition: dict) -> Table:
        definition['hidden'] = True
        table = provider_manager.create_open(
            definition)  # type: Table
        self.__tmp_tables.append(table)
        return table 

    def __nofify_about_suitable_types(self):
        self.iface.notifications.push(
            self.title, axipy.tr(
                f"Не удалось выполнить построение Полигонов Вороного. "
                f"Поддерживаемые типы геометрий: Точки. "
                f"Минимальное колличество: {MIN_POINTS_COUNT}"), Notifications.Critical)

    def __notify_about_sql_error(self, query_text: str):
            print(axipy.tr(f'Произошла ошибка при выполнении SQL-запроса {query_text}'))

