
from typing import Callable, List
from axipy.interface import AxiomaInterface
from PySide2.QtWidgets import QAction
import abc


class UnloadActionInterface(metaclass=abc.ABCMeta):
    """
    Интерфейс для действия при отгрузке плагина
    """

    @abc.abstractmethod
    def unload(self) -> None:
        raise NotImplementedError

    def set_description(self, description: str):
        self.__description = description

    def __str__(self) -> str:
        return self.__description


class CompositeUndloadActions(UnloadActionInterface):
    """
    Реализация паттерна Композит. Действие объединяет в себе другие действия
    для отгрузки, чтобы с коллекцией можно было взаимодействовать так-же, как и
    с обычным единичным действием для отгрузки
    """

    def __init__(
            self,
            unloadable_actions: List[UnloadActionInterface]) -> None:
        super().__init__()
        self.__unloadable_actions = unloadable_actions

    def add_action(self, action: UnloadActionInterface):
        self.__unloadable_actions.append(action)

    def unload(self) -> None:
        for action in self.__unloadable_actions:
            action.unload()


class UnloadQtAction(UnloadActionInterface):
    """
    Действие для отгрузки QAction
    """

    def __init__(self, action: QAction, iface: AxiomaInterface) -> None:
        super().__init__()
        self.__action = action
        if (isinstance(action, QAction)):
            self.set_description("Unload QAction: " + action.text())
        self.__iface = iface

    def unload(self):
        self.__iface.menubar.remove(self.__action)


class AnyUnloadableAction(UnloadActionInterface):
    """
    Обертка принимающая любую пользовательскую функцию с аргументами,
    который будут вызваны при выгрузке
    """

    def __init__(self, func: Callable, *args, **kwargs) -> None:
        super().__init__()
        self.__func = func
        self.__args = args
        self.__kwargs = kwargs

    def unload(self) -> None:
        self.__func(*self.__args, **self.__kwargs)
