
import math
import decimal
from decimal import Decimal
from typing import List, Tuple, Callable, Union

from axipy.utl import Pnt

from PySide2.QtCore import QPoint, QPointF, QLineF, Qt
from PySide2.QtGui import QPainter, QPen


def center_and_radius_of_circle(points: List[Pnt]) -> Tuple[Pnt, float]:
    # Уравнение 2D сферы:
    # (x - x0)**2 + (y - y0)**2 = r**2
    # где a и b это координаты её центра. Решая это уравнения для 3х точек
    # можно найти x0, y0 и радиус r. Решено с использованием sympy
    import decimal
    from decimal import Decimal
    # В контексте текущих уравнений максимально необходимая точность 
    # в районе 21 - 22 знаков для целых значений + дробная часть. Чтобы
    # не терять точность на вычислениях используем Decimal
    with decimal.localcontext() as ctx:
        ctx.prec = 50
        assert(len(points) == 3)
        p1 = points[0]
        p2 = points[1]
        p3 = points[2]
        x_1 = Decimal(p1.x)
        x_2 = Decimal(p2.x)
        x_3 = Decimal(p3.x)
        y_1 = Decimal(p1.y)
        y_2 = Decimal(p2.y)
        y_3 = Decimal(p3.y)
        x0 = (Decimal(1/2))*(x_1**2*y_2 - x_1**2*y_3 - x_2**2*y_1 + x_2**2*y_3 + x_3**2*y_1 - x_3**2*y_2 + y_1**2*y_2 - y_1**2*y_3 - y_1*y_2**2 + y_1*y_3**2 + y_2**2*y_3 - y_2*y_3**2)/(x_1*y_2 - x_1*y_3 - x_2*y_1 + x_2*y_3 + x_3*y_1 - x_3*y_2)
        y0 = Decimal(-1/2)*(x_1**2*x_2 - x_1**2*x_3 - x_1*x_2**2 + x_1*x_3**2 - x_1*y_2**2 + x_1*y_3**2 + x_2**2*x_3 - x_2*x_3**2 + x_2*y_1**2 - x_2*y_3**2 - x_3*y_1**2 + x_3*y_2**2)/(x_1*y_2 - x_1*y_3 - x_2*y_1 + x_2*y_3 + x_3*y_1 - x_3*y_2)
        r = Decimal(-1/2)*((x_1**2 - 2*x_1*x_2 + x_2**2 + y_1**2 - 2*y_1*y_2 + y_2**2)*(x_1**2 - 2*x_1*x_3 + x_3**2 + y_1**2 - 2*y_1*y_3 + y_3**2)*(x_2**2 - 2*x_2*x_3 + x_3**2 + y_2**2 - 2*y_2*y_3 + y_3**2))**(Decimal(1/2))/(x_1*y_2 - x_1*y_3 - x_2*y_1 + x_2*y_3 + x_3*y_1 - x_3*y_2)
        # радиус не может быть отрицательным
        r = abs(r)
        return Pnt(float(x0), float(y0)), float(r)

def centers_by_2_points_and_radius(p1: Pnt, p2: Pnt, r: float) -> List[Pnt]:
    # мы знаем две точки и расстояние до центра. Взяв два уравнения расстояния 
    # между точками можно найти два неизвестных параметра (x,y) центра окружности
    x1 = p1.x
    x2 = p2.x
    y1 = p1.y
    y2 = p2.y

    def evaluate(x_1: float, x_2: float, y_1: float, y_2: float) -> List[Pnt]:
        def to_real(value: Union[float, complex]) -> float:
            if isinstance(value, complex):
                return value.real
            return value
        x01 = (1/2)*(x_1**2 - x_2**2 + y_1**2 - y_2**2 + (-2*y_1 + 2*y_2)*((1/2)*y_1 + (1/2)*y_2 - 1/2*((x_1**2 - 2*x_1*x_2 + x_2**2 + y_1**2 - 2*y_1*y_2 + y_2**2)*(4*r**2 - x_1**2 + 2*x_1*x_2 - x_2**2 - y_1**2 + 2*y_1*y_2 - y_2**2))**(1/2)*(x_1 - x_2)/(x_1**2 - 2*x_1*x_2 + x_2**2 + y_1**2 - 2*y_1*y_2 + y_2**2)))/(x_1 - x_2)
        x01 = to_real(x01)
        y01 = (1/2)*y_1 + (1/2)*y_2 - 1/2*((x_1**2 - 2*x_1*x_2 + x_2**2 + y_1**2 - 2*y_1*y_2 + y_2**2)*(4*r**2 - x_1**2 + 2*x_1*x_2 - x_2**2 - y_1**2 + 2*y_1*y_2 - y_2**2))**(1/2)*(x_1 - x_2)/(x_1**2 - 2*x_1*x_2 + x_2**2 + y_1**2 - 2*y_1*y_2 + y_2**2)
        y01 = to_real(y01)
        x02 = (1/2)*(x_1**2 - x_2**2 + y_1**2 - y_2**2 + (-2*y_1 + 2*y_2)*((1/2)*y_1 + (1/2)*y_2 + (1/2)*((x_1**2 - 2*x_1*x_2 + x_2**2 + y_1**2 - 2*y_1*y_2 + y_2**2)*(4*r**2 - x_1**2 + 2*x_1*x_2 - x_2**2 - y_1**2 + 2*y_1*y_2 - y_2**2))**(1/2)*(x_1 - x_2)/(x_1**2 - 2*x_1*x_2 + x_2**2 + y_1**2 - 2*y_1*y_2 + y_2**2)))/(x_1 - x_2)
        x02 = to_real(x02)
        y02 = (1/2)*y_1 + (1/2)*y_2 + (1/2)*((x_1**2 - 2*x_1*x_2 + x_2**2 + y_1**2 - 2*y_1*y_2 + y_2**2)*(4*r**2 - x_1**2 + 2*x_1*x_2 - x_2**2 - y_1**2 + 2*y_1*y_2 - y_2**2))**(1/2)*(x_1 - x_2)/(x_1**2 - 2*x_1*x_2 + x_2**2 + y_1**2 - 2*y_1*y_2 + y_2**2)
        y02 = to_real(y02)
        return [Pnt(x01, y01), Pnt(x02, y02)]

    def is_same_sign(v1: float, v2: float):
        if v1 == v2:
            return True
        return (v1 < 0 and v2 < 0) or (v1 > 0 and v2 > 0)

    if not is_same_sign(x1, x2) or not is_same_sign(y1, y2):
        return evaluate(x1, x2, y1, y2)

    def vector_len(p1: Tuple[Decimal, Decimal], p2: Tuple[Decimal, Decimal]) -> Decimal:
        return ((p2[0] - p1[0])**2 + (p2[1] - p1[1])**2)**(Decimal(1/2))

    # Если координаты по x или y у двух точек одинаковые то в формуле могут возникать ошибки
    # деления на 0 и тд. Поэтому эти случаи обработаем отдельно
    if math.isclose(x1, x2):
        x = Decimal(x1)
        y1d = Decimal(y1)
        y2d = Decimal(y2)
        center = (x, (y2d + y1d) / 2)
        dist_to_center = vector_len((Decimal(p1.x), Decimal(p1.y)), center)
        if math.isclose(dist_to_center, r):
            return [Pnt(center[0],center[1]), Pnt(center[0],center[1])]
        h = (Decimal(r)**2 - dist_to_center**2)**(Decimal(1/2))
        return [Pnt(float(x+h), float(center[1])), Pnt(float(x-h), float(center[1]))]

    if math.isclose(y1,y2):
        y = Decimal(y1)
        x1d = Decimal(x1)
        x2d = Decimal(x2)
        center = ((x1d + x2d) / 2, y)
        dist_to_center = vector_len((Decimal(p1.x), Decimal(p1.y)), center)
        if math.isclose(dist_to_center, r):
            return [Pnt(center[0],center[1]), Pnt(center[0],center[1])]
        h = (Decimal(r)**2 - dist_to_center**2)**(Decimal(1/2))
        return [Pnt(float(center[0]), float(y + h)), Pnt(float(center[0]), float(y - h))]

    # При вычислении центра мы выходим за границу точности float в Python 
    # и получаем ошибку округления. Особенно это заметно когда большие числа
    # различаются незначительно. Чтобы уменьшить влияние этого фактора, смешаем 
    # центр координатной системы в одну из точек, а потом результат смещаем обратно
    # Т.о. мы уменьшаем число значащих цифр и повышаем точность вычислений
    def smallest_diff(v1: float, v2: float) -> Tuple[float, float, float]:
        factor = abs(abs(v2) - abs(v1)) * math.copysign(1, v1) * (-1)
        return v1 + factor, v2 + factor, factor

    x1_f, x2_f, factor_x = smallest_diff(x1, x2)
    y1_f, y2_f, factor_y = smallest_diff(y1, y2)
    p1, p2 = evaluate(x1_f, x2_f, y1_f, y2_f)
    factor_x = factor_x * (-1)
    factor_y = factor_y * (-1)
    return [Pnt(p1.x + factor_x, p1.y + factor_y), Pnt(p2.x + factor_x, p2.y + factor_y)]
    

def angle_by_points(center: Pnt, p: Pnt, to_device: Callable[[Pnt], QPoint]) -> float:
    """
    Возвращает математический угол в градусах.
    """

    def zero_degree_point(p0: Pnt) -> Pnt:
        return Pnt(p0.x + 1, p0.y)

    def to_vec(p1: Pnt, p2: Pnt) -> Pnt:
        return Pnt(p2.x - p1.x, p2.y - p1.y)

    def dot(v1: Pnt, v2: Pnt) -> float:
        return v1.x * v2.x + v1.y * v2.y

    def len(v1: Pnt) -> float:
        return math.sqrt(v1.x**2 + v1.y**2)

    zero_degree_p = zero_degree_point(center)
    v1 = to_vec(center, zero_degree_p)
    v2 = to_vec(center, p)
    rad = math.acos(dot(v1, v2) / (len(v1) * len(v2)))
    degree = math.degrees(rad)

    def is_low_half(equator, p):
        equatord = to_device(equator)
        p1d = to_device(p)
        return p1d.y() > equatord.y()

    if is_low_half(center, p):
        diff = 180 - degree
        return 180 + diff
    return degree

def draw_lines(p: QPainter, points: List[QPointF]):
    lines = [] # type: List[QLineF]
    if len(points) < 2:
        return
    for index in range(len(points) - 1):
        lines.append(QLineF(points[index], points[index + 1]))
    pen = p.pen()
    old_pen = QPen(pen)
    pen.setStyle(Qt.DashLine)
    p.setPen(pen)
    p.drawLines(lines)
    p.setPen(old_pen)
        
