import os
from PyQt5.QtWidgets import QWidget, QFileDialog, QDialog, QMessageBox
from PyQt5.QtGui import QIcon
from PyQt5.QtCore import QStringListModel, Qt, QFileInfo, QCoreApplication, QDir, QRectF
from PyQt5.uic import loadUi
import axioma.app

from axioma.core import *
from axioma.cs import CoordTransform
from axioma.core.dp import *
from axioma.gui import ChooseCoordSystemDialog
from axioma.mapinfo import TabJsonController
from .DoubleProgressWidget import DoubleProgressWidget
from .BoundingRectDialog import BoundingRectDialog
import traceback

class ExportToFileWidget(QWidget):

    def __init__(self,  catalog, parent=None):
        super().__init__(parent)
        cwd = os.path.dirname(__file__)
        self.__ui = loadUi(os.path.join(cwd, "exportToFileWidget.ui"), self)
        self.initUi()
        self.__catalog = catalog
        self.__coordSystem = None
        self.__destinationDataProvider = axioma.app.core.dataProviderById("TabDataProvider")

    def initUi(self):
      self.__ui.tbSelectFiles.setIcon(QIcon.fromTheme("open"))
      self.__ui.tbSelectFiles.clicked.connect(self.onOpenFilesClicked)
      self.__ui.tbDeleteSelected.setIcon(QIcon.fromTheme("delete"))
      self.__ui.tbDeleteSelected.clicked.connect(self.onDeleteSelectedClicked)
      self.__ui.listSourceTables.itemSelectionChanged.connect(self.onSelectionChanged)
      self.__ui.pbExport.clicked.connect(self.exportProcess)
      self.__ui.pbChoiseProjection.clicked.connect(self.onProjectionChangeClicked)
      self.__ui.pbBoundingRect.clicked.connect(self.onProjectionRectChangeClicked)

    def keyPressEvent(self, event):
      super().keyPressEvent(event)
      if (event.key() == Qt.Key_Delete):
        onDeleteSelectedClicked()

    def onOpenFilesClicked(self):
     try:
      settings = axioma.app.core.settings()
      lastPath = settings.value(DefaultSettingsList.LastOpenPath)
      dirName = QFileDialog.getOpenFileNames(self, 'Выбор файлов для экспорта', lastPath, 'Файлы TAB (*.tab);;Все файлы (*.*)')
      if len(dirName[0]):
          settings.setValue(DefaultSettingsList.LastOpenPath, QFileInfo(dirName[0][0]).absolutePath())
          for fn in dirName[0]:
            if len(self.__ui.listSourceTables.findItems(fn, Qt.MatchExactly)) == 0:
              self.__ui.listSourceTables.addItem(fn)
          self.checkEnabledExport()
     except Exception as ex:
       print(ex)
          
    def onProjectionChangeClicked(self):
      dlg = ChooseCoordSystemDialog(self.__coordSystem, self)
      if dlg.exec() == QDialog.Accepted:
        self.setCoordSystem(dlg.chosenCoordSystem())
       
    def onProjectionRectChangeClicked(self):
      rectDialog = BoundingRectDialog(self.__coordSystem.rectNumericCoordSys(), self)
      if rectDialog.exec() == QDialog.Accepted:
        self.__coordSystem.setRectCoordSystem(rectDialog.resultRect())

    def onSelectionChanged(self):
      self.__ui.tbDeleteSelected.setEnabled(len(self.__ui.listSourceTables.selectedItems()))
      
    def coordSystemForTable(self, fileName):
      try:
        table = axioma.core.open_json({'src': fileName})
        return table.coordSystem()
      except Exception as ex:
        print(ex)
      return None

    def setCoordSystem(self, coordSystem):
      if coordSystem is not None:
        self.__ui.pbChoiseProjection.setText(coordSystem.description())
        csCopy = coordSystem.clone()
        if self.__coordSystem is not None and coordSystem.rectNumericCoordSys().isEmpty() and not self.__coordSystem.rectNumericCoordSys().isEmpty():
          ct = CoordTransform(self.__coordSystem, coordSystem)
          rout = ct.forward(self.__coordSystem.rectNumericCoordSys())
          csCopy.setRectCoordSystem(rout)
        self.__coordSystem = csCopy
      else:
        self.__ui.pbChoiseProjection.setText("Проекция")
        self.__coordSystem = None

    def checkEnabledExport(self):
      enabled = self.__ui.listSourceTables.count()
      self.__ui.pbExport.setEnabled(enabled)
      self.__ui.groupBoxProjection.setEnabled(enabled)
      if enabled and self.__coordSystem == None:
        coordSys = self.coordSystemForTable(self.__ui.listSourceTables.item(0).text())
        self.setCoordSystem(coordSys)

    def onDeleteSelectedClicked(self):
      selected_items = self.__ui.listSourceTables.selectedItems()
      for item in selected_items:
        self.__ui.listSourceTables.takeItem(self.__ui.listSourceTables.row(item))
      if self.__ui.listSourceTables.count() == 0:
	      self.setCoordSystem(None)
      self.checkEnabledExport()

    def exportProcess(self):
      settings = axioma.app.core.settings()
      lastPath = settings.value(DefaultSettingsList.LastSavePath)
      dirName = QFileDialog.getExistingDirectory(self, 'Выбор каталога для целевых файлов', lastPath, QFileDialog.ShowDirsOnly)
      if not len(dirName[0]):
        return
      self.__ui.pbExport.setEnabled(False)
      try:
        cnt = self.__ui.listSourceTables.count()
        progress = DoubleProgressWidget(axioma.app.mainWindow)
        progress.setMaximumValueForTotalProgress(cnt)
        progress.setFixedSize(progress.width(), progress.height()) # disabled maximized button
        progress.move(axioma.app.mainWindow.rect().center() - progress.rect().center())
        progress.show()
        for idx in range(0, cnt):
          self.exportFile(dirName, self.__ui.listSourceTables.item(idx).text(), progress)
          progress.setCurrentValueForTotalProgress(idx+1)
          QCoreApplication.processEvents()
        progress.close()
        print('Завершено')
      except Exception as ex:
        print(ex)
      self.__ui.pbExport.setEnabled(True)

    def onProgress(self, count, progress):
      if progress is not None:
        progress.setCurrentValueForCurrentProgress(count)
      QCoreApplication.processEvents()

    def exportFile(self, dirName, fileName, progress):
      print('Исходный файл:', fileName)
      try:
        table = axioma.core.open_json({'src': fileName})
        if not isinstance(table, Table):
          raise TypeError('Исходный файл {} не является таблицей.'.format(fileName))
        outFileName = os.path.join(dirName, os.path.basename(fileName))
        print('Выходной файл:', outFileName)
        
        jd = JsonDefinition()
        if (self.__ui.cbPreserveGeomery.isChecked()):
           TabJsonController.setPreserveGeometryCoordinates(jd, True)
        JsonDataCreationController.setSource(jd, outFileName)
        
        if self.__ui.groupBoxProjection.isChecked():
          if (self.__ui.cbPreserveGeomery.isChecked()):
            # Объединяем MBR
            cs = self.__coordSystem.clone()
            if not cs.rectNumericCoordSys().contains(table.coordSystem().rectNumericCoordSys()):
              if QMessageBox.question(axioma.app.mainWindow, "Подтверждение", "MBR исходной таблица выходит за пределы MBR целевой КС. Расширить?") == QMessageBox.Yes:
                rectUnited = cs.rectNumericCoordSys().united(table.coordSystem().rectNumericCoordSys())
                cs.setRectCoordSystem(rectUnited)
            JsonDataCreationController.setPrj(jd, cs)
          else:
            JsonDataCreationController.setPrj(jd, self.__coordSystem)
        else:
          JsonDataCreationController.setPrj(jd, table.coordSystem())
        JsonDataCreationController.setCreate(jd, True)
        JsonDataCreationController.setOpenWith(jd, self.__destinationDataProvider.class_id())
        JsonDataCreationController.setAccessMode(jd, Access_ReadWrite)
        
        progress.setCurrentTableName(table.name())
        
        context = ProviderExportContext(ModifiedTableSchema(table.tableSchema()))
        cursor = SignalingCursor(table.select(QuerySelectAll( table.tableSchema().attributeNames())))
        cursor.moved.connect(lambda count: self.onProgress(count, progress))
        progress.destroyed.connect(lambda : cursor.abort())
        progress.setMaximumValueForCurrentProgress(table.count())
        self.__destinationDataProvider.exportFeatures(cursor, context, jd)
      except Exception as ex:
        print(ex)
        axioma.app.core.notificationManager().push(Notification('Ошибка', str(ex), Notification.NR_Warning))
