from axipy.cpp_gui import ShadowWorkspace
from axipy.da import DataManager
from .gui_class import gui_instance


class Workspace:
    """Рабочее пространство для сохранения текущего состояния.

    Note:
        Данный класс следует использовать в случае, когда отсутствует главное окно приложения :class:`axipy.app.MainWindow` для сохранения или чтения текущего состояния. 
        Если же главное окно присутствует, то можно воспользоваться методами :meth:`axipy.app.MainWindow.load_workspace` для чтения
        и :meth:`axipy.app.MainWindow.save_workspace` для записи рабочего пространства.

    Рабочее пространство можно как сохранять в файл так и читать из него.
    При чтении из файла рабочего пространства посредством метода :meth:`load_file`
    все источники данных (таблицы) открываются и добавляются в каталог :class:`axipy.da.DataManager`, доступный через переменную :attr:`axipy.da.data_manager`.
    А окна с наполнением добавляются в менеджер окон :class:`axipy.gui.ViewManager`, доступный через переменную :attr:`view_manager`.
    
    В случае записи текущего состояния в файл рабочего пространства последовательность обратная рассмотренной. Состояние каталога и менеджера окон
    записывается в рабочее пространство посредством метода :meth:`save_file`.

    Пример чтения данных::

        print('Before: tables({}), views({})'.format(len(data_manager), len(view_manager)))
        ws.load_file('ws.mws')
        print('After: tables({}), views({})'.format(len(data_manager), len(view_manager)))

        >>> Before: tables(0), views(0)
        >>> After: tables(5), views(3)

    Пример записи рабочего пространства::

        ws = Workspace()
        ws.save_file('ws_out.mws')
    """

    def __init__(self):
        self.shadow = ShadowWorkspace(gui_instance.shadow)

    def load_file(self, fileName: str):
        """Читает из файла рабочего пространства и заносит данные в текущее окружение.
        
        Args:
            fileName: Наименование входного файла.
        """
        self.shadow.load_file(fileName)

    def save_file(self, fileName: str):
        """Сохраняет текущее состояние в файл рабочего пространства.

        Args:
            fileName: Наименование выходного файла.
        """
        self.shadow.save_file(fileName)

