import sys
import inspect, os
import PyQt5.QtGui
from PyQt5.QtCore import *
from PyQt5.QtGui import QMouseEvent, QIcon, QColor, QBrush
from PyQt5.QtWidgets import *
from PyQt5.QtXml import QDomDocument
import axioma.gui.extension
import axioma.app
import axioma.render
from axioma.gui import RibbonId, RibbonActionInfo, upListSelectionForWidget, downListSelectionForWidget
from PyQt5.uic import loadUi
from axioma.core import Core
'''

Расширение, которое позволяет оперативно переключать каталоги со стилями линий заливок и символов.
После загрузки расширение регистрируется на панели "Основные"
Если в каталоге с настройками пользователя присутствует файл StyleCatalog.xml, то он рассматривается как именованный набор шаблонов с каталогами.
Местоположение файла:
C:/Users/<UserName>/AppData/Roaming/ESTI/Axioma.GIS/StyleCatalog.xml - windows
/home/<UserName>/.local/share/settings/Axioma.GIS/StyleCatalog.xml - linux
Эти шаблоны зачитываются в выпадающий список "Наборы".
Для того, чтобы получить возможность переключения между несколькими наборами, необходимо взять как образец файл StyleCatalog.xml.template переименовать 
его в StyleCatalog.xml и заполнить по аналогии с содержимым, указав необходимые для работы каталоги.

Назначение кнопок и элеменов ввода:
Наборы стилей - перечень именованных наборов данных
"Вверх" - выбранное перемещается вверх по списку
"Вниз" - выбранное перемещается вниз по списку
"Добавить" - добавление нового именованного набора. Добавляется набор с пустыми полями. которые в последующем рекомендуется заполнить
"Удалить" - удаление выбранного набора из списка. Реальное удаление производится только после операции "Сохранить в XML"
"По умолчанию" - в поля имен каталогов вносятся значения по умолчанию, которые используются при первоначальном запуске система. Данная функция ничего не меняет, только ачения полей ввода.
"OK" - значения каталогов будут использованы в системе до завершения сеанса


Extension that allow you switch sets with styles of pen, brush and raster symbols.
After it was loaded it has registered on "Options" panel.
If StyleCatalog.xml exists in user settings dir you can use it as template with catalog presets.
File place:
C:/Users/<UserName>/AppData/Roaming/ESTI/Axioma.GIS/StyleCatalog.xml - windows
/home/<UserName>/.local/share/settings/Axioma.GIS/StyleCatalog.xml - linux
Make copy StyleCatalog.xml.template file as StyleCatalog.xml and edit content it.

Purpose of buttons and line edits:
"Manage sets" - list of named sets
"Move up" - move selected up
"Move down" - move selected down
"Append" - append new named set
"Remove" - remove current selected set
"Defaults" - update line gui elements to default system values
"OK" - current values will be used till current session is loaded

'''

def getCurrentDir():
    return os.path.dirname(os.path.abspath(inspect.getfile(inspect.currentframe())))

class StyleCatalogDialog(QDialog):
    _nameKeyName = 'name'
    _penKeyName = 'pen'
    _brushKeyName = 'brush'
    _symbolKeyName = 'symbol'
    __xmlFileName="StyleCatalog.xml"

    def __init__(self, catalogService, resourceDir=None, parent=None):
        super().__init__(parent)
        self._catalogService = catalogService
        self._cwd = getCurrentDir()
        self.__ui = loadUi(os.path.join(self._cwd, 'StyleCatalog.ui'), self)
        self._xmlFileInfo = QFileInfo(Core.appLocalDataLocation().absoluteFilePath(self.__xmlFileName))
        self.__systemCatalogList = self.__findSystemCatalogs(resourceDir)
        #self.initSets()
        self.initUi()
        self.setWindowFlags(self.windowFlags() & ~Qt.WindowContextHelpButtonHint);
        QTimer.singleShot(100, self.initSets) # delay initialization to avoid unnecessary select first item

    def initUi(self):
        self.__ui.tbCatalogPen.setIcon(QIcon.fromTheme("open"))
        self.__ui.tbCatalogPen.clicked.connect(self.onCatalogPenClicked)
        self.__ui.leCatalogPen.setText(self._catalogService.penCatalog())
        self.__ui.tbCatalogBrush.setIcon(QIcon.fromTheme("open"))
        self.__ui.tbCatalogBrush.clicked.connect(self.onCatalogBrushClicked)
        self.__ui.leCatalogBrush.setText(self._catalogService.brushCatalog())
        self.__ui.tbCatalogSymbol.setIcon(QIcon.fromTheme("open"))
        self.__ui.tbCatalogSymbol.clicked.connect(self.onCatalogSymbolClicked)
        self.__ui.leCatalogSymbol.setText(self._catalogService.symbolCatalog())
        self.__ui.pbDefault.clicked.connect(self.onDefaultClicked)
        self.__ui.listWidget.currentRowChanged.connect(self.onCurrentRowChanged)
        self.__ui.tbUp.setIcon(QIcon.fromTheme("arrow_up"))
        self.__ui.tbBottom.setIcon(QIcon.fromTheme("arrow_down"))
        self.__ui.tbAdd.setIcon(QIcon.fromTheme("add"))
        self.__ui.tbRemove.setIcon(QIcon.fromTheme("delete"))
        self.__ui.tbUp.clicked.connect(self.onUp)
        self.__ui.tbBottom.clicked.connect(self.onBottom)
        self.__ui.tbAdd.clicked.connect(self.onAdd)
        self.__ui.tbRemove.clicked.connect(self.onRemove)
        self.__ui.leCatalogPen.textChanged.connect(self.onTextChanged)
        self.__ui.leCatalogBrush.textChanged.connect(self.onTextChanged)
        self.__ui.leCatalogSymbol.textChanged.connect(self.onTextChanged)
        self.__ui.buttonBox.button(QDialogButtonBox.Ok).setEnabled(False)
        
    def __findSystemCatalogs(self, resource_dir):
        """
        Ищем каталоги в стандартных местах
        \resource\ExtendedStyles
        """
        result=list()
        if not resource_dir or not isinstance(resource_dir, QDir):
            return result
        extended_styles_dir=QDir(resource_dir.filePath("ExtendedStyles"))
        if extended_styles_dir.exists():
            catalog_list=extended_styles_dir.entryInfoList(QDir.AllDirs | QDir.NoDotAndDotDot)
            for catalog in catalog_list:
                xml_file=QFileInfo(QDir(catalog.absoluteFilePath()), self.__xmlFileName)
                if xml_file.exists():
                    result.append(xml_file)
        return result

    def onTextChanged(self, text):
       self.__ui.buttonBox.button(QDialogButtonBox.Ok).setEnabled(True)

    def __readItems(self, file_info, is_relative=False):
        """
        :param file_info: Файл xml
        :param is_relative: Все пути являются относительными
        :type is_relative: :class:`bool`
        """
        catalog_list=list()
        file = QFile(file_info.absoluteFilePath())
        if file.open(QIODevice.ReadOnly):
            doc = QDomDocument('')
            doc.setContent(file)
            file.close()
            node = doc.documentElement().firstChild()
            while not node.isNull():
                catalogs = dict()
                el = node.toElement()
                if el.hasAttribute(self._penKeyName):
                    catalogs[self._penKeyName] = el.attribute(self._penKeyName)
                if el.hasAttribute(self._brushKeyName):
                    catalogs[self._brushKeyName] = el.attribute(self._brushKeyName)
                if el.hasAttribute(self._symbolKeyName):
                    catalogs[self._symbolKeyName] = el.attribute(self._symbolKeyName)
                catalogs[self._nameKeyName] = el.attribute(self._nameKeyName)
                catalog_list.append(catalogs)
                node = node.nextSibling()
        item_list=list()
        if is_relative:
            apply_relative=lambda catalog: {key:file_info.absoluteDir().absoluteFilePath(val) if val and self.__is_path_key(key) else val for key, val in catalog.items()}
            catalog_list=[apply_relative(catalog) for catalog in catalog_list]
        for catalog in catalog_list:
            item = QListWidgetItem(catalog[self._nameKeyName])
            self.__set_data_catalog(item, catalog)
            item_list.append(item)
        return item_list

    @staticmethod
    def __set_data_catalog(item, data) -> None:
        item.setData(Qt.UserRole, data)

    @staticmethod
    def __get_data_catalog(item):
        return item.data(Qt.UserRole)

    @staticmethod
    def __is_system(item) -> bool:
        return item.data(Qt.UserRole+1)

    @staticmethod
    def __set_is_system(item) -> None:
        item.setData(Qt.BackgroundRole, QBrush(Qt.lightGray))
        item.setData(Qt.UserRole+1, True)

    @classmethod
    def __is_path_key(cls, key) -> True:
        return key == cls._symbolKeyName or key == cls._brushKeyName or key == cls._penKeyName

# Производим чтение наборов из файла
# retriving sets from xml
    def initSets(self):
        system_item_groups=[self.__readItems(system_catalog, is_relative=True) for system_catalog in self.__systemCatalogList]
        self.__ui.listWidget.blockSignals(True)
        for item_list in system_item_groups:
            for item in item_list:
                self.__set_is_system(item)
                self.__ui.listWidget.addItem(item)
        if self._xmlFileInfo.exists():
            item_list=self.__readItems(self._xmlFileInfo)
            for item in item_list:
                self.__ui.listWidget.addItem(item)
        self.__ui.listWidget.blockSignals(False)
        self.__ui.listWidget.setCurrentItem(None)


# save sets to xml
# Сохранение наборов в файле
    def saveXml(self):
        file = QFile(self._xmlFileInfo.absoluteFilePath())
        if QMessageBox.question(self, QCoreApplication.translate("esti.styleCatalog","Подтверждение"), QCoreApplication.translate("esti.styleCatalog","Файл {} существует. Перезаписать?").format(self._xmlFileInfo.absoluteFilePath()) ) == QMessageBox.Yes:
            if file.open(QIODevice.WriteOnly | QIODevice.Text):
              doc = QDomDocument('')
              doc.appendChild(doc.createProcessingInstruction("xml", "version=\"1.0\" encoding=\"UTF-8\""))
              root = doc.createElement('catalogs')
              doc.appendChild(root)
              for i in range(0, self.__ui.listWidget.count()):
                item=self.__ui.listWidget.item(i)
                if StyleCatalogDialog.__is_system(item):
                    continue
                element = doc.createElement('catalog')
                element.setAttribute(self._nameKeyName, item.text())
                data = self.__get_data_catalog(item)
                if self._penKeyName in data:
                    element.setAttribute(self._penKeyName, data[self._penKeyName])
                if self._brushKeyName in data:
                    element.setAttribute(self._brushKeyName, data[self._brushKeyName])
                if self._symbolKeyName in data:
                    element.setAttribute(self._symbolKeyName, data[self._symbolKeyName])
                root.appendChild(element)
              stream = QTextStream(file)
              stream.setCodec("UTF-8")
              stream << doc.toString()
              file.close()
            else:
              QMessageBox.information(self, QCoreApplication.translate("esti.styleCatalog","Информация"), QCoreApplication.translate("esti.styleCatalog","Ошибка записи файла {}").format(file))

    def catalogCommonClicked(self, name, le, keyName):
        lastDirStylePathTag = "plugins/terplan/lastDirStylePath"
        settings = QSettings()
        currDir = le.text()
        if currDir is None:
            currDir = settings.value(lastDirStylePathTag)
        dirName = QFileDialog.getExistingDirectory(self, name, currDir, QFileDialog.ShowDirsOnly | QFileDialog.DontResolveSymlinks)
        if dirName:
            le.setText(dirName)
            item = self.__ui.listWidget.currentItem()
            if item is not None:
                catalogs = self.__get_data_catalog(item)
                catalogs[keyName] = dirName
                settings.setValue(lastDirStylePathTag, dirName)
                self.__set_data_catalog(item, catalogs)

    def onCatalogPenClicked(self):
        self.catalogCommonClicked(QCoreApplication.translate("esti.styleCatalog","Каталог со стилями линий"), self.__ui.leCatalogPen, self._penKeyName)

    def onCatalogBrushClicked(self):
        self.catalogCommonClicked(QCoreApplication.translate("esti.styleCatalog","Каталог со стилями заливки"), self.__ui.leCatalogBrush, self._brushKeyName)

    def onCatalogSymbolClicked(self):
        self.catalogCommonClicked(QCoreApplication.translate("esti.styleCatalog","Каталог с растровыми символами"), self.__ui.leCatalogSymbol, self._symbolKeyName)

# Сброс на значения по умолчанию
# Reset by defaults
    def onDefaultClicked(self): 
        self.__ui.leCatalogPen.setText(self._catalogService.defaultPenDir())
        self.__ui.leCatalogBrush.setText(self._catalogService.defaultBrushDir())
        self.__ui.leCatalogSymbol.setText(self._catalogService.defaultSymbolDir())
        self.__ui.listWidget.setCurrentItem(None)

# Permanent store 
# Сохранение в настройках
    def saveInSettings(self):
      try:
        settings = QSettings()
        axioma.app.render.renderSettingsManager().setPenStyleCatalog(self.__ui.leCatalogPen.text())
        axioma.app.render.renderSettingsManager().setBrushStyleCatalog(self.__ui.leCatalogBrush.text())
        axioma.app.render.renderSettingsManager().setSymbolStyleCatalog(self.__ui.leCatalogSymbol.text())
        axioma.app.render.renderSettingsManager().saveSettings(settings)
        QMessageBox.information(self, QCoreApplication.translate("esti.styleCatalog","Информация"), QCoreApplication.translate("esti.styleCatalog","Каталоги со стилями сохранены в настройках"))
      except Exception as ex:
        print(ex)

    def changeText(self, keyName, catalogs, le):
        if keyName in catalogs and catalogs[keyName]:
          le.setText(catalogs[keyName])
        else:
          le.clear()

# Устанавливаем данные из наборов
# Set data from sets
    def onCurrentRowChanged(self, index): 
        if index == -1:
          return
        item = self.__ui.listWidget.item(index)
        catalogs=self.__get_data_catalog(item)
        if catalogs is not None:
            self.changeText(self._penKeyName, catalogs, self.__ui.leCatalogPen)
            self.changeText(self._brushKeyName, catalogs, self.__ui.leCatalogBrush)
            self.changeText(self._symbolKeyName, catalogs, self.__ui.leCatalogSymbol)

    def penResult(self):
        return self.__ui.leCatalogPen.text()

    def brushResult(self):
        return self.__ui.leCatalogBrush.text()

    def symbolResult(self):
        return self.__ui.leCatalogSymbol.text()

    def onUp(self):
        upListSelectionForWidget(self.__ui.listWidget)

    def onBottom(self):
        downListSelectionForWidget(self.__ui.listWidget)

    def onAdd(self):
     try:
        name = QInputDialog.getText(self, QCoreApplication.translate("esti.styleCatalog","Название набора"), QCoreApplication.translate("esti.styleCatalog","Набор стилей"), QLineEdit.Normal, '', self.windowFlags() & ~Qt.WindowContextHelpButtonHint)
        if name[0] and name[1]:
          item = QListWidgetItem(name[0])
          self.__set_data_catalog(item, dict())
          self.__ui.listWidget.addItem(item)
     except Exception as ex:
        print(ex)

    def onRemove(self):
        row = self.__ui.listWidget.currentRow()
        if row != -1 and not StyleCatalogDialog.__is_system(self.__ui.listWidget.item(row)):
            item = self.__ui.listWidget.takeItem(row)
            item = None


class StyleCatalogActionExtension(axioma.gui.NativeActionExtension):

    def __init__(self, action_id, ribbon_action_info=RibbonActionInfo(), parent=None):
        super().__init__(action_id, ribbon_action_info, parent)
        self.mainwindow = self.createDependencyGetter(axioma.app.MainWindow)
        self.catalogService = self.createDependencyGetter(axioma.render.StyleCatalogService)
        self.getCore = self.createDependencyGetter(axioma.core.Core)

    def createAction_impl(self, parent):
        action = PyQt5.QtWidgets.QAction(parent)
        action.setText(QCoreApplication.translate("esti.styleCatalog","Каталоги со стилями"))
#        action.setIcon(QIcon(":/icons/share/32px/styles.png"))
        action.setIcon(QIcon(os.path.join(getCurrentDir(), 'StyleCatalog.png')))
        action.triggered.connect(self.slot);
        return action

    def slot(self):
      try:
        catalog = StyleCatalogDialog(self.catalogService(), self.getCore().validDataDir(), axioma.app.mainWindow)
        catalog.setWindowIcon(axioma.app.mainWindow.windowIcon())
        if catalog.exec() == QDialog.Accepted:
            self.catalogService().setPenCatalog(catalog.penResult())
            self.catalogService().setBrushCatalog(catalog.brushResult())
            self.catalogService().setSymbolCatalog(catalog.symbolResult())
            catalog.saveInSettings()
            catalog.saveXml()
      except Exception as ex:
        print(ex)


def createCatalogExtensions():
    ribbonExt = axioma.gui.extension.RibbonExtension()
    ribbonExt.addAction("esti.styleCatalog.actionid", RibbonId.Tab.General, RibbonId.Group.Settings, 2, 2)
    actionExt = StyleCatalogActionExtension("esti.styleCatalog.actionid")
    return [ribbonExt, actionExt]
