import os
from PySide2.QtWidgets import QDialog, QDialogButtonBox
from PySide2.QtUiTools import QUiLoader
from PySide2.QtCore import Qt

from .CoordinateProcess import *
from axipy import *


class LoadPointsDialog(QDialog):

    def __init__(self, iface, coordsystem) -> None:
        super().__init__(iface.window())
        self.__tr = iface.tr
        self._coordsystem = coordsystem
        uiFile = os.path.join(os.path.dirname(__file__), "LoadPointsDialog.ui")
        self.__ui = QUiLoader().load(uiFile, self)
        self.__load_ui()
        self.__update_button_text(coordsystem)
        self.__ui.setWindowFlags(self.windowFlags() & ~Qt.WindowContextHelpButtonHint)

    def __load_ui(self):
        for t in data_manager.tables:
            if len(t.schema) >= 4:
                self.__ui.cbTables.addItem(t.name)
        self.__ui.buttonBox.button(QDialogButtonBox.Ok).setEnabled(self.__ui.cbTables.count())
        self.__ui.cbTables.currentIndexChanged.connect(self.__tablesIndexChanged)
        self.__ui.cbTypeData.currentIndexChanged.connect(self.__typeDataIndexChanged)
        self.__ui.pushButtonSK.clicked.connect(self.__chooseCoordSystem)
        if self.__ui.cbTables.count():
            self.__tablesIndexChanged(0)
        self.__typeDataIndexChanged(0)
    
    def __update_button_text(self, coordsystem):
        if coordsystem is not None:
            self.__ui.pushButtonSK.setText(coordsystem.description)

    def __chooseCoordSystem(self):
        dialog = ChooseCoordSystemDialog(self._coordsystem)
        if dialog.exec() == QDialog.Accepted:
            self._coordsystem = dialog.chosenCoordSystem()
            self.__update_button_text(self._coordsystem)


    def __typeDataIndexChanged(self, idx):
        if idx == 0:
            self.__ui.lblValue1.setText(self.__tr('Х координата:'))
            self.__ui.lblValue2.setText(self.__tr('Y координата:'))
        else:
            self.__ui.lblValue1.setText(self.__tr('Расстояние (м.):'))
            self.__ui.lblValue2.setText(self.__tr('Азимут (град.):'))

    def __set_items(self, cb, attrs, attr):
        cb.clear()
        cb.addItems(attrs)
        cb.setCurrentText(attr)

    def __tablesIndexChanged(self, idx):
        t = self.result_table()
        if t is not None:
            attrs = t.schema.attribute_names
            self.__set_items(self.__ui.cbGeometryId, attrs, CoordinateProcess.IdFeatureField)
            self.__set_items(self.__ui.cbGeometryPartId, attrs, CoordinateProcess.IdGeometryField)
            self.__set_items(self.__ui.cbPointNumber, attrs, CoordinateProcess.IdPointField)
            self.__set_items(self.__ui.cbValue1, attrs, CoordinateProcess.ValueFirstField)
            self.__set_items(self.__ui.cbValue2, attrs, CoordinateProcess.ValueSecondField)

    def result_sql_text(self):
        # Результирующий SQL запрос
        def __ct( cb):
            return cb.currentText() 
        return 'select {}, {}, {}, {}, {} from {} order by 1, 2, 3'.format(__ct(self.__ui.cbGeometryId), __ct(self.__ui.cbGeometryPartId),
            __ct(self.__ui.cbPointNumber), __ct(self.__ui.cbValue1), __ct(self.__ui.cbValue2), self.result_table().name)

    def result_coordsystem(self):
        return self._coordsystem

    def result_table(self):
        # Результирующая базовая таблица
        return data_manager.find(self.__ui.cbTables.currentText())

    def is_simple_coordinates(self):
        return self.__ui.cbTypeData.currentIndex() == 0

    def exec(self):
        return self.__ui.exec()
        
