"""
Класс страница параметров создания wizard
"""
from math import ceil, log10

from PySide2.QtCore import Qt, Slot, QRegExp, QEvent, Signal, QTimer
from PySide2.QtGui import QRegExpValidator
from PySide2.QtWidgets import QWizardPage, QVBoxLayout, QLabel, QPushButton, QLineEdit, QCheckBox, QComboBox, QWidget, \
    QFormLayout, QWizard, QDialog, QMessageBox, QProgressBar
import axipy
from axipy import CoordSystem, ChooseCoordSystemDialog, tr, utl
from axipy.concurrent import AxipyAnyCallableTask, task_manager
from osgeo import gdal

from ru_axioma_gis_vector_converter import jsonloader
from ru_axioma_gis_vector_converter.wizardconverter import WizardConverter, create_hbox, notify


class CreationPage(QWizardPage):
    """
    Страница выбора выходной СК, параметра перепроецирования СК, выбора выходного расширения
    и параметров создания выходных файлов. Реализован автоматический расчет охвата для файлов MapInfo.
    Параметр перепроецирования доступен, если входная и выходная СК не равны.
    """

    lnd_clicked = Signal()

    def __init__(self, parent: WizardConverter) -> None:
        super().__init__(parent)
        self.parent = parent
        width = self.parent.wizard_width * 35 // 100
        # Вертикальная разметка страницы
        self.vbox = QVBoxLayout()
        self.setLayout(self.vbox)

        label = QLabel(tr("Формат выходных файлов"))
        label.setStyleSheet("font-weight: bold")
        self.out_format_label = QLabel()
        self.vbox.addWidget(label)
        self.vbox.addWidget(self.out_format_label)

        label = QLabel(tr("Режим создания файлов"))
        label.setFixedWidth(width)
        self.cbox_file_creation_mode = QComboBox()
        self.cbox_file_creation_mode.currentTextChanged.connect(self.show_file_name_field)

        hbox = create_hbox(label, self.cbox_file_creation_mode)
        self.vbox.addLayout(hbox)
        label = QLabel(tr("Имя файла"))
        label.setFixedWidth(width)
        self.lnd_append_file_name = QLineEdit()
        hbox = create_hbox(label, self.lnd_append_file_name)
        hbox.setContentsMargins(0, 0, 0, 0)
        self.file_name_for_append = QWidget()
        self.file_name_for_append.setLayout(hbox)
        self.vbox.addWidget(self.file_name_for_append)

        # Заголовок
        title_label = QLabel(tr("Выходная система координат"))
        title_label.setStyleSheet("font-weight: bold")
        self.vbox.addWidget(title_label)
        # Отображение и выбор СК
        self.line_edit = QLineEdit()
        self.line_edit.setReadOnly(True)
        btn = QPushButton("...", clicked=self.choose_out_coord, objectName="btn_choose_out_coord")
        btn.setMaximumWidth(20)
        hbox = create_hbox(self.line_edit, btn)
        hbox.setAlignment(Qt.AlignLeft)
        self.vbox.addLayout(hbox)
        # Перепроецирование СК
        self.label = QLabel(tr("Перепроецировать координаты"))
        self.check_box = QCheckBox()
        self.check_box.stateChanged.connect(self.check_state_changed)
        hbox = create_hbox(self.label, self.check_box)
        hbox.setAlignment(Qt.AlignLeft)
        self.vbox.addLayout(hbox)

        # Выбор выходного расширения
        label_out_ext = QLabel(tr("Выходное расширение"))
        label_out_ext.setFixedWidth(width)
        self.cbox_out_ext = QComboBox(objectName="cbox_out_ext")
        self.cbox_out_ext.setEditable(True)
        hbox = create_hbox(label_out_ext, self.cbox_out_ext)
        hbox.setAlignment(Qt.AlignLeft)
        hbox.setContentsMargins(0, 0, 0, 0)
        self.widget_out_ext = QWidget()
        self.widget_out_ext.setLayout(hbox)
        self.vbox.addWidget(self.widget_out_ext)

        self.form_view = None   # Форма с параметрами создания

        self.pbar = QProgressBar()
        self.vbox.addWidget(self.pbar)

        # Расчет охвата для выходного драйвера MapInfo File
        self.line_edit_bounds = None
        self.lnd_bounds_last_text = None
        self.btn_calc_bounds = None
        self.m_box = QMessageBox(
            QMessageBox.Question, tr("Вопрос"),
            tr("Отменить расчет охвата?"),
            buttons=QMessageBox.Yes | QMessageBox.No, parent=self)
        self.canceled_by_dlg = False

        self.task = None

    def initializePage(self) -> None:
        """
        Инициализация страницы.
        """
        # Инициализация надписи выходной формат
        self.out_format_label.setText("{}".format(self.parent.drivers[self.parent.out_gdal_format]["name"]))

        # Режим создания файлов
        # Инициализация режима создания файлов (Сброс параметров).
        mode = self.parent.file_creation_mode_list
        self.cbox_file_creation_mode.clear()
        self.lnd_append_file_name.clear()
        self.parent.file_name_for_append = None
        # Входной формат multilayer
        inp_m = self.parent.drivers[self.parent.inp_gdal_format].get("multilayer", False)
        # Выходной формат multilayer
        out_m = self.parent.drivers[self.parent.out_gdal_format].get("multilayer", False)
        # Выходной формат поддерживает append
        out_a = self.parent.drivers[self.parent.out_gdal_format].get("append", False)
        # Добавление режимов в combobox
        self.cbox_file_creation_mode.addItem(mode[1])
        if out_m and out_a:
            self.cbox_file_creation_mode.addItem(mode[2])
        if inp_m and out_m:
            self.cbox_file_creation_mode.addItem(mode[0])
        # Если элементов combox не больше одного, выключение combox (Для индикации пользователю).
        if self.cbox_file_creation_mode.count() == 1:
            self.cbox_file_creation_mode.setEnabled(False)
        elif self.cbox_file_creation_mode.count() > 1:
            self.cbox_file_creation_mode.setEnabled(True)

        # Синхронизация СК
        self.parent.out_cs = self.parent.inp_cs
        self.line_edit.setText(self.parent.out_cs.name)

        # Обработка исключения для cbox выбора расширения, для выходного формата "MapInfo File"
        widget = self.widget_out_ext
        if self.parent.out_gdal_format == "MapInfo File":
            widget.hide()
        else:
            widget.show()

        # Инициализация cbox с расширениями
        self.cbox_out_ext.clear()
        # Поиск расширения в json по драйверу
        self.cbox_out_ext.addItems(self.parent.drivers[self.parent.out_gdal_format]["out_ext"])

        # Инициализация формы с настройками
        self.form_view = jsonloader.get_form_view(self, self.parent.out_gdal_format, ["dataset", "layer"])
        # Вставка на одну позицию выше полоски прогресса
        i = self.vbox.count() - 1
        self.vbox.insertWidget(i, self.form_view)

        # Добавление кнопки рассчета охвата для выходного формата "MapInfo File"
        if self.parent.out_gdal_format == "MapInfo File":
            # Кнопка
            self.btn_calc_bounds = QPushButton(tr("Рассчитать охват"))
            self.btn_calc_bounds.clicked.connect(self.calc_bounds_in_thread)
            self.btn_calc_bounds.setToolTip(tr("Рассчет охвата с повышенной точностью."))
            # Получение позиции для кнопки
            form = self.findChild(QFormLayout)
            name = str(jsonloader.get_indx_from_type("layer")) + "BOUNDS"
            self.line_edit_bounds = self.findChild(QLineEdit, name)
            self.lnd_bounds_last_text = ""
            row = form.getWidgetPosition(self.line_edit_bounds)[0]
            # Вставка кнопки на позицию ниже BOUNDS
            form.insertRow(row + 1, QLabel(""), self.btn_calc_bounds)

        # не ставить выше инициализации формы
        self.validate_check_box()
        self.validate_bounds()

        self.setButtonText(QWizard.NextButton, tr("Конвертировать"))

        self.pbar.resetFormat()
        self.pbar.hide()

    def cleanupPage(self) -> None:
        """
        Вызывается при нажатии кнопки назад. Удаляет добавленную форму.
        """
        self.form_view.setParent(None)

    def validatePage(self) -> True:
        """
        Валидация страницы при нажатии кнопки далее. Считывает польз. ввод и выходное расширение.
        """
        cbox_txt = self.cbox_file_creation_mode.currentText()
        i = self.parent.file_creation_mode_list.index(cbox_txt)
        self.parent.file_creation_mode = i
        if self.parent.file_creation_mode_list.index(cbox_txt) == 2:
            lnd_txt = self.lnd_append_file_name.text()
            if lnd_txt:
                self.parent.file_name_for_append = lnd_txt
        # Считывание польз. ввода с формы после нажатия кнопки Далее
        form = self.form_view.widget()
        _, self.parent.dataset, self.parent.layer = self.parent.read_form(form, self.parent.out_gdal_format)
        # Считывание расширения
        self.parent.out_ext = self.cbox_out_ext.currentText()

        return True

    def eventFilter(self, watched, event):
        """
        Фильтр ставиться на поле ввода "Охват" для вых. формата MapInfo File во время расчета охвата.
        Нужен для отслеживания клика мыши по полю для отмены активного расчета охвата.
        """
        if event.type() == QEvent.MouseButtonRelease:
            self.lnd_clicked.emit()
        return False

    def dynamic_form_validation(self) -> None:
        """
        Вызов статического метода валидации формы parent.
        """
        self.parent.dynamic_form_validation(self)

    def validate_bounds(self) -> None:
        """
        Функция выполняется при изменении выходной СК. Отвечает за изменение доступности параметра перепроецирования СК.
        Также проверяет необходимость расчета охвата для формата "MapInfo File".
        Вызывается при инициализации страницы и при выборе выходной ск.
        """
        cs_is_equal = self.parent.inp_cs == self.parent.out_cs  # Если входная ск = выходной ск
        out_non_earth = self.parent.out_cs.non_earth

        # Если на выходе "MapInfo File"
        if self.parent.out_gdal_format == "MapInfo File":

            if self.parent.reproject:
                rect = self.parent.out_cs.rect
            else:
                rect = self.parent.inp_cs.rect

            if out_non_earth:
                if cs_is_equal and rect:
                    rect_str = "{},{},{},{}".format(rect.xmin, rect.ymin, rect.xmax, rect.ymax)
                    self.line_edit_bounds.setText(rect_str)
                elif not cs_is_equal or not rect:
                    self.calc_bounds_in_thread()
            else:
                if rect:
                    rect_str = "{},{},{},{}".format(rect.xmin, rect.ymin, rect.xmax, rect.ymax)
                    self.line_edit_bounds.setText(rect_str)
                else:
                    self.calc_bounds_in_thread()

    def validate_check_box(self) -> None:
        """
        Функция для проверки доступности галочки "Перепроецировать".
        """
        cs_is_equal = self.parent.inp_cs == self.parent.out_cs  # Если входная ск = выходной ск
        # Если на входе или на выходе декартова ск
        non_earth = self.parent.inp_cs.non_earth or self.parent.out_cs.non_earth

        # Доступность галочки "Перепроецировать"
        if cs_is_equal or non_earth:
            # Откл. перепроецирование
            self.check_box.setEnabled(False)
            self.check_box.setChecked(False)
            self.label.setStyleSheet("QLabel {color:grey}")
        else:
            # Вкл. перепроецирование
            self.check_box.setEnabled(True)
            self.check_box.setChecked(True)
            self.label.setStyleSheet("QLabel {color:black}")

    def choose_out_coord(self) -> None:
        """
        Выбор выходной СК при нажатии кнопки "...". Также вызывает функцию валидации СК.
        """
        # Обработка декартовых СК
        if self.parent.inp_cs.non_earth:
            cs = CoordSystem.from_wkt(self.parent.out_cs.wkt)
        else:
            cs = CoordSystem.from_proj(self.parent.out_cs.proj)

        # Диалог выбора СК
        dlg = ChooseCoordSystemDialog(cs)
        if dlg.exec_() == QDialog.Accepted:
            cs = dlg.chosenCoordSystem()
            self.parent.out_cs = cs
            # Валидация СК
            self.validate_check_box()
            if self.parent.out_gdal_format == "MapInfo File":
                self.validate_bounds()
            self.line_edit.setText(cs.name)

    def check_state_changed(self) -> None:
        """
        Галочка перепроецирования СК.
        """
        if self.check_box.isChecked():
            self.parent.reproject = True
        else:
            self.parent.reproject = False

        self.validate_bounds()

    def calc_bounds_in_thread(self) -> None:
        """
        Расчет охвата для MapInfo File через GetExtent() в отдельном потоке.
        """
        def calculate_map_info_file_bounds():
            """
            Расчет охвата.

            :return: x_min,y_min,x_max,y_max
            """
            try:
                # Сбор охвата каждого входного файла в список
                extents_list = list()
                path_list = [str(path) for path in self.parent.input_path_list]
                for path in path_list:
                    if self.task.progress_handler().is_canceled():
                        return None
                    ds = gdal.OpenEx(path)
                    layer = ds.GetLayer()
                    extent = layer.GetExtent()
                    layer = None
                    ds = None

                    x_min, x_max, y_min, y_max = extent
                    if x_min > x_max or y_min > y_max:
                        raise Exception("Минимальное значение больше максимального")

                    extents_list.append(extent)

                # Определение минимального и максимального x и y
                x_min = min([extent[0] for extent in extents_list])
                x_max = max([extent[1] for extent in extents_list])
                y_min = min([extent[2] for extent in extents_list])
                y_max = max([extent[3] for extent in extents_list])

                # Расчет ширины и высоты
                width = (x_max - x_min)
                height = (y_max - y_min)

                if self.parent.reproject:
                    # Трансформация точек с учетом СК
                    ct = axipy.cs.CoordTransformer(self.parent.inp_cs, self.parent.out_cs)
                    dx = width / 100
                    dy = height / 100
                    pnts = ct.transform(
                        [utl.Pnt(x_min + i*dx, y_min + j*dy) for i in range(0, 101) for j in range(0, 101)]
                    )
                    x_min = min(pnt.x for pnt in pnts)
                    y_min = min(pnt.y for pnt in pnts)
                    x_max = max(pnt.x for pnt in pnts)
                    y_max = max(pnt.y for pnt in pnts)
                    width = (x_max - x_min)
                    height = (y_max - y_min)

                max_length = max(width, height)
                if max_length == 0:
                    max_length = max(abs(x_max), abs(y_max))
                    if max_length == 0:
                        max_length = 1e8

                p = ceil(log10(abs(max_length)))
                if abs(p) >= 20:
                    raise Exception("Слишком большое значение охвата")

                def in_extent(extent1: int, extent2: int) -> bool:
                    """
                    Функция проверяет нахождение точек во входном промежутке, по x и по y.
                    """
                    if (
                            extent1 < x_min and x_max < extent2 and
                            extent1 < y_min and y_max < extent2):
                        return True
                    else:
                        return False

                if in_extent(-10 ** p, 10 ** p):
                    x_min, y_min, x_max, y_max = (-10 ** p, -10 ** p, 10 ** p, 10 ** p)
                else:
                    if in_extent(0, 2 * 10 ** p):
                        x_min, y_min, x_max, y_max = (0, 0, 2 * 10 ** p, 2 * 10 ** p)
                    else:
                        if in_extent(-2 * 10 ** p, 0):
                            x_min, y_min, x_max, y_max = (-2 * 10 ** p, -2 * 10 ** p, 0, 0)
                        else:
                            def circle_center(arg_1: float, arg_2: float) -> int:
                                """
                                Функция поиска круглой середины.
                                """
                                return round((arg_1 + arg_2) / 2 / 10 ** p) * 10 ** p

                            c_x = circle_center(x_min, x_max)
                            c_y = circle_center(y_min, y_max)

                            x_min, y_min, x_max, y_max = (c_x - 10 ** p, c_y - 10 ** p, c_x + 10 ** p, c_y + 10 ** p)
            except Exception as e:
                notify(tr("Не получилось рассчитать охват"))
            else:
                return f"{x_min},{y_min},{x_max},{y_max}"

        def show_cancel_dlg() -> None:
            """
            Показывает диалог отмены охвата.
            """
            self.m_box.exec_()
            if self.m_box.result() == QMessageBox.Yes:
                if self.task:
                    self.canceled_by_dlg = True
                    self.task.progress_handler().cancel()

        self.canceled_by_dlg = False

        # Блок гуи
        self.findChild(QPushButton, "btn_choose_out_coord").setEnabled(False)
        self.btn_calc_bounds.setEnabled(False)
        QTimer.singleShot(0, lambda: self.parent.button(QWizard.BackButton).setEnabled(False))
        QTimer.singleShot(0, lambda: self.parent.button(QWizard.NextButton).setEnabled(False))

        # Изменение режима поля на "расчет охвата"
        self.line_edit_bounds.setReadOnly(True)
        self.lnd_bounds_last_text = self.line_edit_bounds.text()
        self.line_edit_bounds.setValidator(None)
        self.line_edit_bounds.setText(tr("Вычисление..."))
        self.line_edit_bounds.installEventFilter(self)
        self.lnd_clicked.connect(show_cancel_dlg)

        # Запуск функции расчета охвата в отдельном потоке
        if self.task is None:
            self.task = AxipyAnyCallableTask(calculate_map_info_file_bounds)
            self.task.with_handler(False)
            self.task.progress_handler().finished.connect(self.finished)
            task_manager.start_task(self.task)
        # end

    def show_file_name_field(self, txt) -> None:
        """
        Показ или скрытие поля выбора имени для сборки в один файл.
        """
        if txt:
            if self.parent.file_creation_mode_list.index(txt) == 2:
                self.file_name_for_append.setEnabled(True)
            else:
                self.file_name_for_append.setEnabled(False)
                self.lnd_append_file_name.clear()
                self.parent.file_name_for_append = None

    @Slot()
    def finished(self) -> None:
        """
        Вызывается при завершении расчета охвата в потоке. Отменяет режим расчета охвата (блокировка гуи и т.д.).
        """
        if self.task.progress_handler().is_canceled():
            notify(tr("Расчет охвата отменен."), 0)
            if not self.canceled_by_dlg:
                self.parent.reject()

        self.line_edit_bounds.removeEventFilter(self)
        self.lnd_clicked.disconnect()

        # Ввод результата расчета в поле ввода
        txt = self.task.progress_handler().result
        if txt:
            self.line_edit_bounds.setText(txt)
        else:
            self.line_edit_bounds.setText(self.lnd_bounds_last_text)

        # Возврат валидатора ввода охвата
        self.line_edit_bounds.setValidator(
            QRegExpValidator(QRegExp(r"-?\d+\.?\d+,-?\d+\.?\d+,-?\d+\.?\d+,-?\d+\.?\d+")))

        self.line_edit_bounds.setReadOnly(False)

        # Включение элементов, выключенных во время блока гуи
        self.btn_calc_bounds.setEnabled(True)
        self.findChild(QPushButton, "btn_choose_out_coord").setEnabled(True)

        self.parent.button(QWizard.BackButton).setEnabled(True)
        self.parent.button(QWizard.NextButton).setEnabled(True)
        self.task = None
