
import math
import enum
import sys
import traceback
from typing import Optional, Union, List, Tuple


from PySide2.QtCore import Qt, QPoint, QPointF, QLineF
from PySide2.QtGui import QKeyEvent,  QPaintEvent, QPainter, QPen, QMouseEvent, \
    QDoubleValidator
from PySide2.QtWidgets import QWidget, QGridLayout, QAbstractButton, \
    QMessageBox
from PySide2.QtUiTools import QUiLoader

import axipy
from axipy.gui import MapTool, ActiveToolPanel, view_manager
from axipy.utl import Pnt, Rect, Printer
from axipy.da import Feature, Style, Geometry, DefaultKeys
from axipy.mi import Arc
from axipy.app import Notifications
from axipy.interface import AxiomaInterface
from axipy.cs import Unit

from .helper import editable_table_from_view, text_to_float
from .circle_utils import center_and_radius_of_circle, angle_by_points, \
    centers_by_2_points_and_radius

COUNT_ARC_BY_3_PNT = 3
COUNT_ARC_BY_2_PNT_AND_R = 2
MAX_COUNT_BEFORE_FINILIZE_ARC = 2



class CreateArcMode(enum.Enum):
    """
    Способ задания дуги при наличии двух точек и радиуса
    """
    SemiCircle = 0 # предпологаем, что линия из двух точек проходит через центр окружности
    SmallArc = 1 # для дуги берётся меньшая часть окружности отделенная линией
    LargeArc = 2 # большая

class PositionOnLine(enum.Enum):
    """
    Справа или слева по направлении линии создавать дугу
    """
    Left = 0
    Right = 1

class ArcCreationSettings:

    def __init__(self, mode: CreateArcMode = None, \
            pos: PositionOnLine = None, \
            radius: float = None) -> None:
        self.mode = mode
        self.pos = pos
        self.radius = radius

class ArcByPointsWidget(QWidget):
    """
    Графический элемент для создания дуги по двум точкам и радиусу
    """

    def __init__(self, ui_file: str) -> None:
        super().__init__()
        self.__ui = QUiLoader().load(ui_file, self)
        grid_layout = QGridLayout()
        grid_layout.addWidget(self.__ui)
        self.setLayout(grid_layout)
        self.__init_radius_le()
        self.__ui.semi_circle_rb.setChecked(True)
        self.__ui.left_rb.setChecked(True)
        self.__ui.radius_le.setEnabled(False)
        self.__source_radius = None
        self.set_radius(1)

    def set_unit(self, suffix: str):
        self.__ui.unit_label.setText(suffix)

    def set_radius(self, value: float):
        self.__source_radius = value
        locale = self.__ui.radius_le.locale()
        text = Printer.to_localized_string(value, locale)
        self.__ui.radius_le.setText(text)

    def __mode(self) -> CreateArcMode:
        if self.__ui.semi_circle_rb.isChecked():
            return CreateArcMode.SemiCircle
        elif self.__ui.small_arc_rb.isChecked():
            return CreateArcMode.SmallArc
        elif self.__ui.large_arc_rb.isChecked():
            return CreateArcMode.LargeArc

    def __position_on_line(self) -> PositionOnLine:
        if self.__ui.left_rb.isChecked():
            return PositionOnLine.Left
        else:
            return PositionOnLine.Right
        
    def validate_data_and_notify(self, title: str) -> bool:
        settings = self.get_settings() # type: ArcCreationSettings
        is_wrong_radius = math.isclose(settings.radius, self.__source_radius) or \
            settings.radius < self.__source_radius
        if (self.__source_radius is not None and \
            settings.mode != CreateArcMode.SemiCircle and \
            is_wrong_radius):
            locale = self.__ui.radius_le.locale()
            radius_text = Printer.to_localized_string(settings.radius, locale)
            desctiption = axipy.tr(f"Введите радиус больше чем {radius_text} {self.__ui.unit_label.text()}")
            QMessageBox.information(view_manager.global_parent, title, desctiption)
            return False
        return True

    def get_settings(self) -> ArcCreationSettings:
        settings = ArcCreationSettings()
        settings.mode = self.__mode()
        settings.pos = self.__position_on_line()
        settings.radius = text_to_float(self.__ui.radius_le.text(), \
            self.__ui.radius_le.locale())
        return settings

    def __init_radius_le(self):
        validator = QDoubleValidator(
            bottom=sys.float_info.min,
            top=sys.float_info.max,
            decimals=15)
        validator.setNotation(QDoubleValidator.StandardNotation)
        self.__ui.radius_le.setValidator(validator)
        self.__ui.radius_le.setToolTip(axipy.tr("Радиус окружности по которой строится дуга"))
        def update_enabled_status(button: QAbstractButton, checked: bool):
            if not checked:
                return
            self.__ui.radius_le.setEnabled(button != self.__ui.semi_circle_rb)

        self.__ui.type_of_arc_bg.buttonToggled.connect(update_enabled_status)


class ArcByPoints(MapTool):

    def __init__(self, iface: AxiomaInterface, title: str, \
        observer_id: DefaultKeys.Key) -> None:
        super().__init__()
        self.__title = title # type: str
        self.__observer_id = observer_id
        self.__iface = iface # type: AxiomaInterface
        self.__points = [] # type: List[Pnt]
        self.__current_point = None # type: Pnt
        tool_panel_manager = ActiveToolPanel()
        ui_file = self.__iface.local_file('ui','ArcBy2PointsAndRadius.ui')
        widget = ArcByPointsWidget(ui_file)
        self.__tool_panel = tool_panel_manager.make_acceptable(self.__title, \
            self.__observer_id, widget)
        def make_arc_by_2_points():
            try: 
                self.__make_arc_by_2_points_and_radius()
            except:
                Notifications.push(self.__title, axipy.tr("Не удалось создать дугу,"
                " попробуйте снова изменив введённые коордианты."))
                self.restart()
        self.__tool_panel.accepted.connect(make_arc_by_2_points)
        self.__finilize_state = False

    @property
    def widget(self) -> ArcByPointsWidget:
        return self.__tool_panel.widget

    def pnt_from_event(self, event: QMouseEvent) -> Pnt:
        scene_pos = self.snap(event.pos()) # type: Union[Pnt, QPoint]
        if not self.is_snapped():
            scene_pos = self.to_scene(scene_pos) # type: Pnt
        return scene_pos

    def restart(self):
        self.__clear()
        self.__tool_panel.deactivate()

    def mousePressEvent(self, event: QMouseEvent):
        if event.button() != Qt.LeftButton:
            return self.PassEvent
        # если мы уже завершаем создание дуги то обрабатывать новые нажатия не нужно
        if self.__finilize_state:
            return self.PassEvent
        scene_pos = self.pnt_from_event(event)
        if len(self.__points) >= MAX_COUNT_BEFORE_FINILIZE_ARC:
            return
        self.__points.append(scene_pos)
        self.redraw()

    def radius(self) -> float:
        def length(p1: Pnt, p2: Pnt) -> float:
            return math.sqrt((p2.x - p1.x)**2 + (p2.y - p1.y)**2)
        distance = length(self.__points[0], self.__points[1])
        dist_unit = self.view.map.distanceUnit
        if self.view.coordsystem.lat_lon:
            distance_m, _ = Geometry.distance_by_points(self.__points[0], \
                self.__points[1], self.view.coordsystem)
            distance = Unit.m.to_unit(dist_unit, distance_m)
            return distance / 2
        cs_unit = self.view.coordsystem.unit
        distance = cs_unit.to_unit(dist_unit, distance)
        return distance / 2

    def mouseDoubleClickEvent(self, event: QMouseEvent) -> Optional[bool]:
        if len(self.__points) == 0 or self.__finilize_state:
            # TODO: Notify warning about lackness of points count
            return
        pnt = self.pnt_from_event(event)
        if self.__points[len(self.__points) - 1] != pnt:
            self.__points.append(pnt)
        if len(self.__points) == COUNT_ARC_BY_2_PNT_AND_R:
            self.__finilize_state = True
            self.widget.set_radius(self.radius())
            self.widget.set_unit(self.view.map.distanceUnit.localized_name)
            self.__tool_panel.activate()
        elif len(self.__points) == COUNT_ARC_BY_3_PNT:
            self.__make_arc_by_3_points()
            self.__clear()
        return self.PassEvent

    def mouseMoveEvent(self, event: QMouseEvent) -> Optional[bool]:
        if len(self.__points) == 0:
            return
        self.__current_point = self.pnt_from_event(event)
        self.redraw()


    def right_left_centers(self, p1: Pnt, p2: Pnt, r: float):
        """
        Возвращет правый и левый центр окружности которые можно построить относительно 
        направления заданной линии. Если радиус это половина расстояния между точками
        то центры буду совпадать.
        """
        centers = centers_by_2_points_and_radius(p1, p2, r)
        c1 = centers[0]
        c2 = centers[1]
        right_center = None
        left_center = None
        def is_left(p1, p2, p_test) -> bool:
            return ((p2.x - p1.x) * (p_test.y - p1.y) - (p2.y - p1.y)*(p_test.x - p1.x)) > 0
        if is_left(p1, p2, c1):
            left_center = c1
            right_center = c2
        else:
            left_center = c2
            right_center = c1
        return right_center, left_center

    def __make_arc_by_2_points_and_radius(self):
        is_valid = self.widget.validate_data_and_notify(self.__title)
        if not is_valid:
            return
        settings = self.widget.get_settings()
        p1 = self.__points[0]
        p2 = self.__points[1]
        c0 = Pnt((p1.x + p2.x) / 2, (p1.y + p2.y) / 2)
        radius = None
        if self.view.coordsystem.lat_lon:
            source_distance_m, _ = Geometry.distance_by_points(self.__points[0], \
                self.__points[1], self.view.coordsystem)
            source_distance_degree = math.sqrt((p2.x - p1.x)**2 + (p2.y - p1.y)**2)
            dist_unit = self.view.map.distanceUnit
            current_distance_m = dist_unit.to_unit(Unit.m, settings.radius)
            radius = source_distance_degree * source_distance_m / current_distance_m
        else:
            dist_unit = self.view.map.distanceUnit
            cs_unit = self.view.coordsystem.unit
            radius = dist_unit.to_unit(cs_unit, settings.radius)
        # TODO: Need to protect this function from exception for safe finilizing of the tool. 
        right_center, left_center = self.right_left_centers(p1, p2, radius)
        if settings.mode == CreateArcMode.SemiCircle:
            # берём любой т.к. если радиус равен половине заданной линии, значит она 
            # проходит через центр и решение однозначно
            center = right_center
            angle1 = angle_by_points(center, p1, self.to_device)
            angle2 = angle_by_points(center, p2, self.to_device)
            direction_angle = None
            if settings.pos == PositionOnLine.Left:
                direction_angle = angle2 + 90
            else:
                direction_angle = angle2 - 90
            start, end = self.__angles_rangle_by_direction_angle(angle1, angle2, \
                direction_angle=direction_angle)
            self.insert_arc(start, end, center, radius)
        elif settings.mode == CreateArcMode.LargeArc:
            c0 = Pnt((p1.x + p2.x) / 2, (p1.y + p2.y) / 2)
            center = right_center
            if settings.pos == PositionOnLine.Left:
                center = left_center
            angle1 = angle_by_points(center, p1, self.to_device)
            angle2 = angle_by_points(center, p2, self.to_device)
            direction_angle = angle_by_points(c0, center, self.to_device)
            start, end = self.__angles_rangle_by_direction_angle(angle1, angle2, \
                 direction_angle=direction_angle)
            self.insert_arc(start, end, center, radius)
        elif settings.mode == CreateArcMode.SmallArc:
            center = right_center
            if settings.pos == PositionOnLine.Right:
                center = left_center
            angle1 = angle_by_points(center, p1, self.to_device)
            angle2 = angle_by_points(center, p2, self.to_device)
            small_arc_direction_point = left_center
            if settings.pos == PositionOnLine.Right:
                small_arc_direction_point = right_center
            direction_angle = angle_by_points(c0, small_arc_direction_point, self.to_device)
            start, end = self.__angles_rangle_by_direction_angle(angle1, angle2, \
                 direction_angle=direction_angle)
            self.insert_arc(start, end, center, radius)
        self.__tool_panel.deactivate()
        self.__clear()
    
    def __angles_rangle_by_direction_angle(self, ang1, ang2, direction_angle):
            start = min(ang1, ang2)
            end = max(ang1, ang2)
            if direction_angle > start and direction_angle < end:
                return start, end
            else:
                return end, start

    def __start_end_angles(self, points: List[Pnt], center: Pnt, radius: float) -> Tuple[float, float]:
        p1 = points[0]
        p2 = points[1]
        p3 = points[2]
        ang1 = angle_by_points(center, p1, self.to_device)
        ang2 = angle_by_points(center, p2, self.to_device)
        ang3 = angle_by_points(center, p3, self.to_device)
        return self.__angles_rangle_by_direction_angle(ang1, ang3, direction_angle=ang2)

    def insert_arc(self, start_angle: float, end_angle: float, center: Pnt, \
        radius: float):
        arc = Arc(Rect(0,0,0,0), startAngle=start_angle, \
            endAngle=end_angle, cs=self.view.coordsystem)
        arc.xRadius = radius
        arc.yRadius = radius
        arc.center = center
        f = Feature(geometry=arc, style=Style.for_geometry(arc))
        table = editable_table_from_view(self.view)
        if table is None:
            return
        table.insert(f)
        self.redraw()
        Notifications.push(self.__title, axipy.tr("Дуга успешно добавлена"), \
             Notifications.Success)

    def __make_arc_by_3_points(self):
        try: 
            center, r = center_and_radius_of_circle(self.__points)
            if center is None or r is None:
                return
            start_angle, end_angle = self.__start_end_angles(self.__points, center, r)
            if start_angle is None or end_angle is None:
                return
            self.insert_arc(start_angle, end_angle, center, r)
        except:
            Notifications.push(self.__title, axipy.tr("Не удалось построить дугу "
                "по 3-м точкам. Попробуйте ещё раз изменив координаты."))
            self.restart()

        
    def __clear(self):
        self.__points.clear()
        self.__current_point = None
        self.__finilize_state = False
        self.redraw()

    def keyPressEvent(self, event: QKeyEvent) -> Optional[bool]:
        if event.key() == Qt.Key_Escape:
            self.reset()
            return
        return super().keyPressEvent(event)

    def unload(self):
        # отключаем панель инструментов
        self.__tool_panel.deactivate()
        self.redraw()

    def paintEvent(self, event: QPaintEvent, painter: QPainter):
        painter.save()
        qt_points = [] # type: List[QPointF]
        for point in self.__points:
            qt_points.append(self.to_device(point))
        if self.__current_point is not None and not self.__finilize_state:
            qt_points.append(self.to_device(self.__current_point))
        self.draw_lines(painter, qt_points)
        painter.restore()

    def draw_lines(self, p: QPainter, points: List[QPointF]):
        lines = [] # type: List[QLineF]
        if len(points) < 2:
            return
        for index in range(len(points) - 1):
            lines.append(QLineF(points[index], points[index + 1]))
        pen = p.pen()
        old_pen = QPen(pen)
        pen.setStyle(Qt.DashLine)
        p.setPen(pen)
        p.drawLines(lines)
        p.setPen(old_pen)
            

