from typing import Union
from PySide2.QtWidgets import QWidget
from PySide2.QtCore import Signal, QObject
from axipy.decorator import InitOnce
from .gui_class import gui_instance
from axipy.cpp_gui import AcceptableToolPanelHandler, ActiveToolPanelHandlerBase, \
    CustomToolPanelHandler, ToolPanelHandlerType
from axipy.da import DefaultKeys


class AxipyActiveToolPanelHandlerBase(QObject):

    """
    Базовый класс обработчика панели активного инструмента.
    """

    def __init__(self, shadow_handler) -> None:
        super().__init__()
        self.shadow = shadow_handler
        self._shadow_widget = None

    def __del__(self):
        self.deactivate()

    def activate(self):
        """
        Показывает пользовательский графический элемент в панели активного 
        инструмента.
        """
        self.shadow.setWidget(self._shadow_widget)
        self.shadow.activate()

    def deactivate(self):
        """
        Скрывает пользовательский графический элемент из панели активного инструмента.
        """
        self.shadow.deactivate()

    def set_panel_title(self, title: str):
        """ 
        Устанавливает заголовок панели активного инструмента. 

        Args:
            title: Новый заголовок.
        """
        self.shadow.setPanelTitle(title)

    @property
    def widget(self) -> QWidget:
        """
        Возвращает пользовательский графический элемент.

        Return:
            Переданный ранее пользовательский графический элемент.
        """
        if self._shadow_widget is not None:
            return self._shadow_widget
        else:
            return self.shadow.widget()

    def set_widget(self, widget: QWidget):
        """
        Пользовательский графический элемент будет помещен в панель активного 
        инструмента при активации обработчика. Владение графическим элементом
        передаётся обработчику. Это значит, что не следует использовать и сохранять 
        где-либо ссылку на этот объект. Для получения графического элемента обратно
        используйте :meth:`~axipy.gui.AxipyActiveToolPanelHandlerBase.widget`.
        """
        self._shadow_widget = widget

    def set_observer(self, observer_id: Union[str, DefaultKeys.Key]):
        """
        Метод устанавливает наблюдателя. Если наблюдатель сигнализирует, что
        условия доступности кнопки нарушены, то панель активного инструмента
        сразу же закроется.

        Args:
            observer_id: Идентификатор наблюдателя для управления видимостью и доступностью
            
        .. seealso:: Наблюдатели за состоянием инструмента :class:`~axipy.da.observers`
        """
        is_supported = isinstance(
            observer_id,
            str) or isinstance(
            observer_id,
            DefaultKeys.Key)
        if not is_supported:
            raise TypeError(
                self.tr(f"Неподдерживаемый тип наблюдателя {type(observer_id)}"))
        if isinstance(observer_id, DefaultKeys.Key):
            observer_id = observer_id.name.decode()
        self.shadow.setObserver(observer_id)

    @property
    def activated(self) -> Signal:
        """
        ``Signal[]`` Сигнал испускается когда обработчик панели активного инструмента
        становится активным.
        """
        return self.shadow.activated

    @property
    def deactivated(self) -> Signal:
        """
        ``Signal[]`` Сигнал испускается когда перед тем как обработчик панели активного
        инструмента перестает быть активным.        
        """
        return self.shadow.deactivated

    @property
    def panel_was_closed(self) -> Signal:
        """
        ``Signal[]`` Сигнал испускается после закрытия панели активного инструмента
        """
        return self.shadow.panelClosed


class AxipyAcceptableActiveToolHandler(AxipyActiveToolPanelHandlerBase):
    """
    Обработчик панели активного инструмента, который предоставляет по умолчанию
    блок кнопок Применить/Отменить. При нажатии на эти кнопки испускаются 
    соответствующие сигналы.
    """

    def __init__(
            self,
            shadow_handler: AcceptableToolPanelHandler) -> None:
        super().__init__(shadow_handler)

    @property
    def accepted(self) -> Signal:
        """ 
        ``Signal[]`` Отсылается после того как пользователь нажал кнопку "Применить"
        в панели активного инструмента.
        """
        return self.shadow.accepted

    def try_enable(self):
        """
        Включает доступность блока с кнопками Применить/Отменить если наблюдатель, 
        связанный с панелью активного инструмента, подтверждает доступность.
        """
        self.shadow.enable()

    def disable(self):
        """
        Отключает доступность блока с кнопками Применить/Отменить. Если инструмент
        запускает фоновые задачи с использованием :class:`~axipy.concurrent.TaskManager`,
        то следует вызвать эту функцию перед началом выполнения задачи. Иначе у пользователя
        может быть возможность добавить множество одинаковых задач, несколько раз 
        нажав на кнопку.
        """
        self.shadow.disable()

class AxipyCustomActiveToolPanelHandler(AxipyActiveToolPanelHandlerBase):
    """
    Обработчик панели активного инструмента который не предоставляет никаких встроенных
    по умолчанию элементов управления. 
    """

    def __init__(self, shadow_handler: CustomToolPanelHandler) -> None:
        super().__init__(shadow_handler)


class ActiveToolPanel(QObject):
    """
    Сервис предоставляющий доступ к панели активного инструмента.

    .. literalinclude:: /../../tests/doc_examples/test_example_active_tool_panel.py
        :caption: Пример использования.
        :pyobject: test_make_handler
        :lines: 2-
        :dedent: 4

    Чтобы отобразить переданный ранее графический элемент нужно вызвать
    :meth:`~axipy.gui.AxipyActiveToolPanelHandlerBase.activate`. Например при 
    нажатии на пользовательскую кнопку. 
    
    Панель активного инструмента созданная через :meth:`~axipy.gui.ActiveToolPanel.make_acceptable`
    по умолчанию содержит кнопки "Применить" и "Отмена". По нажатию кнопки "Отмена" 
    посылается сигнал :meth:`~axipy.gui.AxipyAcceptableActiveToolHandler.rejected` 
    и очищается содержимое панели активного инструмента. По нажатию "Применить" 
    отсылается сигнал :meth:`~axipy.gui.AxipyAcceptableActiveToolHandler.accepted`.

    Панель активного инструмента созданная через :meth:`~axipy.gui.ActiveToolPanel.make_custom`
    представляет из себя пустой контейнер в который можно поместить пользовательский
    графический элемент. Это дает больше свободы для реализации управления панелью 
    активного инструмента.

    Переданный идентификатор наблюдателя используется для управления видимостью и 
    доступностью панели. Панель активного инструмента сразу закроется как только
    наблюдатель вернет False.
    """

    def __init__(self) -> None:
        super().__init__(None)

    def make_acceptable(self,
                     title: str,
                     observer_id: Union[str,
                                        DefaultKeys.Key],
                     widget: QWidget = None) -> AxipyAcceptableActiveToolHandler:
        """
        Создает экземпляр обработчика через который можно взаимодействовать с
        панелью активного инструмента. По умолчанию добавляются кнопки "Применить/Отменить".

        Args:
            title: Заголовок панели активного инструмента. Обычно это название инструмента.
            observer_id: Идентификатор наблюдателя для управления видимостью и доступностью.
            widget: Пользовательский виджет который будет отображаться в панели активного инструмента.
        """
        return self.__make_handler(ToolPanelHandlerType.WithAcceptButtonBox, title, observer_id, widget)

    def make_custom(self, 
                     title: str,
                     observer_id: Union[str,
                                        DefaultKeys.Key],
                     widget: QWidget = None) -> AxipyCustomActiveToolPanelHandler:
        """
        Создает экземпляр обработчика панели активного инструмента. В который можно 
        установить любой пользовательский графический элемент. Используется когда пользователю 
        не нужны предустановленные элементы управления.


        Args:
            title: Заголовок панели активного инструмента. Обычно это название инструмента.
            observer_id: Идентификатор наблюдателя для управления видимостью и доступностью.
            widget: Пользовательский виджет который будет отображаться в панели активного инструмента.
        """
        return self.__make_handler(ToolPanelHandlerType.Custom, title, observer_id, widget)

    def __make_handler(self, 
                     handler_type: ToolPanelHandlerType,
                     title: str,
                     observer_id: Union[str,
                                        DefaultKeys.Key],
                     widget: QWidget = None):
        panel_handler = self.__wrap(self.__service().make_handler(handler_type))
        panel_handler.set_observer(observer_id)
        panel_handler.set_panel_title(title)
        panel_handler.set_widget(widget)
        return panel_handler

    @InitOnce
    def __service(self):
        from axipy.cpp_gui import ShadowActiveToolPanelMediator
        shadow_panel = ShadowActiveToolPanelMediator(gui_instance.shadow)
        return shadow_panel

    def __wrap(self, shadow_handler: ActiveToolPanelHandlerBase):
        if isinstance(shadow_handler, AcceptableToolPanelHandler):
            return AxipyAcceptableActiveToolHandler(shadow_handler)
        elif isinstance(shadow_handler, CustomToolPanelHandler):
            return AxipyCustomActiveToolPanelHandler(shadow_handler)
        else:
            raise NotImplementedError()
