from typing import List
from .data_provider import DataProvider, Table
from .source import Source, Destination, Schema
from ..attribute_schema import Schema


class ShapeSource(Source):
    pass


class ShapeDestination(Destination):
    pass


class ShapeDataProvider(DataProvider):
    _identifier = 'OgrDataProvider'

    def get_source(self, filepath: str, encoding: str = 'utf8', prj: str = None) -> Source:
        """Создает источник данных.

        Args:
            filepath: Путь к файлу.
            encoding: Кодировка.
            prj: Строка Системы Координат.

        Пример::

            shp = provider_manager.shp.open('world.shp', prj = '1, 104')
        """
        return ShapeSource(
            Source._provider(self.id),
            Source._table_file(filepath),
            {'charset': encoding, 'prj': prj}
        )

    def get_destination(self, filepath: str, schema: Schema,
                        encoding: str = 'utf8') -> Destination:
        """Создает назначение объекта данных.

        Args:
            filepath: Путь к файлу.
            schema: Схема таблицы.
            encoding: Кодировка.
        """
        return ShapeDestination(schema,
                                Source._provider(self.id),
                                Source._table_file(filepath),
                                {'charset': encoding}
                                )

    def open_temporary(self, schema: Schema) -> Table:
        """Создает временную таблицу.

        Args:
            schema: Схема таблицы.
        """
        return ShapeDestination(schema,
                                Source._provider(self.id),
                                Source._table_file(''),
                                ).create_open()

    def file_extensions(self) -> List[str]:
        return list(filter(lambda v: v in ['shp', 'shz'], super().file_extensions()))
