from PySide2.QtCore import QRectF, QPointF, QRect
from axipy.utl.pnt import Pnt
from typing import Union, Optional


class Rect:
    """Прямоугольник, который не обладает геопривязкой. Используется для различного вида запросов.
    """

    def __init__(self, xmin: float, ymin: float, xmax: float, ymax: float):
        self.__rect = QRectF(xmin, ymin, xmax - xmin, ymax - ymin).normalized()

    @classmethod
    def create_empty(cls):
        obj = cls.__new__(cls)
        obj.__rect = QRectF()
        return obj

    @classmethod
    def from_qt(cls, r: Union[QRectF, QRect]) -> Optional['Rect']:
        """Преобразует из формата Qt. Если класс не соответствует, возвращает None
        
        Args:
            r: Преобразуемый прямоугольник.
        """
        if not isinstance(r, (QRectF, QRect)):
            return None
        obj = cls.__new__(cls)
        obj.__rect = r if isinstance(r, QRectF) else QRectF(r)
        return obj

    def to_qt(self) -> QRectF:
        """Преобразование в формат Qt."""
        return self.__rect

    @classmethod
    def _rect_value_to_qt(cls, v: Union[QRectF, tuple, 'Rect', QRect]) -> QRectF:
        if isinstance(v, tuple) and len(v) == 4:
            return QRectF(QPointF(v[0], v[1]), QPointF(v[2], v[3]))
        elif isinstance(v, Rect):
            return v.to_qt()
        elif isinstance(v, QRect):
            return QRectF(v)
        elif isinstance(v, QRectF):
            return v
        elif isinstance(v, Rect):
            return v.to_qt()
        raise ValueError(f'Bad value {v}')

    @classmethod
    def _rect_value_from_qt(cls, v: Union[QRectF, 'Rect', QRect]) -> 'Rect':
        if isinstance(v, QRectF):
            return Rect.from_qt(v)
        elif  isinstance(v, QRect):
            return Rect.from_qt(QRectF(v))
        elif isinstance(v, Rect):
            return v
        raise ValueError(f'Bad value {v}')

    @property
    def xmin(self) -> float:
        """Минимальное значение X."""
        return self.__rect.left()

    @xmin.setter
    def xmin(self, v: float):
        return self.__rect.setLeft(v)

    @property
    def ymin(self) -> float:
        """Минимальное значение Y."""
        return self.__rect.top()

    @ymin.setter
    def ymin(self, v: float):
        return self.__rect.setTop(v)

    @property
    def xmax(self) -> float:
        """Максимальное значение X."""
        return self.__rect.right()

    @xmax.setter
    def xmax(self, v: float):
        return self.__rect.setRight(v)

    @property
    def ymax(self) -> float:
        """Максимальное значение Y."""
        return self.__rect.bottom()

    @ymax.setter
    def ymax(self, v: float):
        return self.__rect.setBottom(v)

    @property
    def center(self) -> Pnt:
        """Центр прямоугольника."""
        return Pnt.from_qt(self.__rect.center())

    @center.setter
    def center(self, p: Pnt):
        self.__rect.moveCenter(Pnt._fixPnt(p).to_qt())

    def __eq__(self, other):
        if isinstance(other, Rect):
            return self.__rect == other.__rect
        return False

    def __str__(self):
        return "({} {}) ({} {})".format(self.xmin, self.ymin, self.xmax, self.ymax)

    @property
    def is_empty(self) -> bool:
        """Если один или оба размера равны нулю."""
        return self.__rect.isEmpty()

    @property
    def is_valid(self) -> bool:
        """Является ли прямоугольник правильным."""
        return self.__rect.isValid()

    def normalize(self) -> 'Rect':
        """Исправляет прямоугольник, если его ширина или высота отрицательны. """
        self.__rect = self.__rect.normalized()

    @property
    def width(self) -> float:
        """Ширина прямоугольника."""
        return self.__rect.width()

    @property
    def height(self) -> float:
        """Высота прямоугольника."""
        return self.__rect.height()

    def merge(self, other: 'Rect') -> 'Rect':
        """Возвращает прямоугольник, занимаемый обоими прямоугольниками.
        
        Args:
            other: Прямоугольник, с которым производится операция.

        Пример::

            r1 = Rect(2,2,5,5)
            r2 = Rect(3,3,7,7)
            print(r1.merge(r2))
            >>> (2.0 2.0) (7.0 7.0)
        """
        r2 = Rect._rect_value_to_qt(other)
        return Rect.from_qt(self.__rect.united(r2))

    def intersected(self, other: 'Rect') -> 'Rect':
        """Возвращает общий для обоих прямоугольник.
        
        Args:
            other: Прямоугольник, с которым производится операция.

        Пример::

            r1 = Rect(2,2,5,5)
            r2 = Rect(3,3,7,7)
            print(r1.intersected(r2))
            >>> (3.0 3.0) (5.0 5.0)
        """
        r2 = Rect._rect_value_to_qt(other)
        return Rect.from_qt(self.__rect.intersected(r2))

    def translated(self, dx: float, dy: float) -> 'Rect':
        """Возвращает прямоугольник, смещенный на заданную величину.
        
        Args:
            dx: смещение по X
            dy: смещение по Y

        Пример::

            r = Rect(2,2,5,5)
            print(r.translated(10, -10))
            >>> (12.0 -8.0) (15.0 -5.0)
        """
        return Rect.from_qt(self.__rect.translated(dx, dy))

    def expanded(self, dx: float, dy: float) -> 'Rect':
        """Возвращает прямоугольник, увеличенный на заданные величины. Увеличение размеров производится по отношению к центру,
        который не меняется в результате операции.
        
        Args:
            dx: расширение по X
            dy: расширение по Y

        Пример::

            r = Rect(2,2,5,5)
            print(r.expanded(2, 4))
            >>> (1.0 0.0) (6.0 7.0)
        """
        dx2 = dx / 2
        dy2 = dy / 2
        return Rect.from_qt(self.__rect.adjusted(-dx2, -dy2, dx2, dy2))

    def contains(self, other: Union[Pnt, QPointF, 'Rect', QRectF]) -> bool:
        """Содержит ли полностью в своих границах переданный объект.
        
        Args:

            other: Переданный объект - точка или прямоугольник.

        Пример::

            r = Rect(2,2,5,5)
            print(r.contains((3,3)))
            print(r.contains((3,10)))
            print(r.contains(Rect(3,3,7,7)))
            print(r.contains(Rect(3,3,4,4)))
            >>> True
            >>> False
            >>> False
            >>> True
        """
        try:
            r = Rect._rect_value_to_qt(other)
            return self.__rect.contains(r)
        except ValueError:
            try:
                p = Pnt._point_value_to_qt(other)
                return self.__rect.contains(p)
            except ValueError:
                pass
        return False

