from axipy.cpp_core_geometry import (ShadowStyle, ShadowPointStyle, ShadowLineStyle,
                                     ShadowPolygonStyle, ShadowTextStyle, ShadowCollectionStyle)
from axipy.da.Geometry import Geometry
from PySide2.QtGui import QColor, QPainter
from PySide2.QtCore import Qt


class Style:
    """Абстрактный класс стиля оформления геометрического объекта.

    Определяет как будет отрисован геометрический объект.

    Note:
        Для получения текстового представления стиля можно воспользоваться функцией :class:`str`.

    """

    def __init__(self):
        if type(self) is Style:
            raise NotImplementedError

    @classmethod
    def from_mapinfo(cls, mapbasic_string: str) -> 'Style':
        """Получает стиль из строки формата MapBasic.

        Args:
            mapbasic_string: Строка в формате MapBasic.

        .. literalinclude:: /../../tests/doc_examples/test_example_style.py
            :pyobject: test_run_example_style_from_mi
            :lines: 2
            :dedent: 4

        """
        result = ShadowStyle.from_mapinfo(mapbasic_string)
        if result is None or result.isEmpty():
            return None
        return Style._wrap(result)

    def to_mapinfo(self) -> str:
        """Возвращает строковое представление в формате MapBasic.

        .. literalinclude:: /../../tests/doc_examples/test_example_style.py
            :pyobject: test_run_example_style_from_mi
            :lines: 3-
            :dedent: 4
        """
        return self.shadow.to_mapinfo().strip()

#    def to_ogr(self) -> str:
#        """Возвращает строковое представление в формате OGR."""
#        return self.shadow.to_ogr().strip()

    def __eq__(self, other) -> bool:
        return other is not None and self.shadow.isEqual(other.shadow)

    @classmethod
    def for_geometry(cls, geom: Geometry) -> 'Style':
        """Возвращает стиль по умолчанию для переданного объекта.

        Args:
            geom: Геометрический объект, для которого необходимо получить соответствующий ему стиль.
        """
        return Style._wrap(ShadowStyle.for_geometry(geom.shadow))

    def draw(self, geometry: Geometry, painter: QPainter):
        """Рисует геометрический объект с текущим стилем в произвольном контексте вывода. Это может быть востребовано при 
        желании отрисовать геометрию со стилем на форме или диалоге.

        Args:
            geometry: Геометрия. Должна соответствовать стилю. Т.е. если объект полигон, а стиль для рисования точечных
                объектов, то ничего нарисовано не будет.
            painter: Контекст вывода.

        .. literalinclude:: /../../tests/doc_examples/test_example_style.py
            :caption: Пример отрисовки в растре и сохранение результата в файле.
            :pyobject: test_run_example_style_draw
            :lines: 7-
            :dedent: 4

        """
        self.shadow.draw(geometry.shadow, painter)

    def _assign_shadow(self, shadow):
        self.shadow = shadow

    @classmethod
    def _wrap_typed(cls, shadow):
        result = cls.__new__(cls)
        result.shadow = shadow
        return result

    @classmethod
    def _wrap(cls, shadow):
        if shadow is None:
            return None
        if isinstance(shadow, ShadowPointStyle):
            return PointStyle._wrap_typed(shadow)
        elif isinstance(shadow, ShadowLineStyle):
            return LineStyle._wrap_typed(shadow)
        elif isinstance(shadow, ShadowPolygonStyle):
            return PolygonStyle._wrap_typed(shadow)
        elif isinstance(shadow, ShadowTextStyle):
            return TextStyle._wrap_typed(shadow)
        elif isinstance(shadow, ShadowCollectionStyle):
            return CollectionStyle._wrap_typed(shadow)
        else:
            return Style._wrap_typed(shadow)

    def __str__(self) -> str:
        return self.to_mapinfo()


class PointStyle(Style):
    """Стиль оформления точечных объектов.

    По умолчанию создается стиль на базе шрифта True Type, а параметры аналогичны значениям по умолчанию
    в методе :meth:`create_mi_font`.

    Поддерживается 3 вида оформления:

    * Совместимое с MapInfo версии 3. Для создания такого стиля необходимо использовать :meth:`create_mi_compat`.
    * На базе шрифтов True Type. Задано по умолчанию. Стиль создается посредством :meth:`create_mi_font`.
    * На базе растрового файла. Стиль можно создать с помощью :meth:`create_mi_picture`.

    .. literalinclude:: /../../tests/doc_examples/test_example_style.py
        :caption: Пример.
        :pyobject: test_run_example_style_point
        :lines: 2-
        :dedent: 4
    """

    def __init__(self):
        super().__init__()
        self._assign_shadow(ShadowPointStyle())

    @staticmethod
    def create_mi_compat(symbol: int = 35, color: QColor = Qt.red, pointSize: int = 8) -> 'PointStyle':
        """Создание стиля в виде совместимого с MapInfo 3.

        Args:
            symbol: Номер символа, который будет отображен при отрисовке. Для создания невидимого символа используйте значение 31. Стандартный набор условных
                    знаков включает символы от 31 до 67,
            color: Цвет символа
            pointSize: Целое число, размер символа в пунктах от 1 до 48.

        В системе доступны следующие стили:

        .. image:: /../images/style/style_point_mi3.png

        Подробнее: `Стиль MapInfo <http://webhelp.infovista.com/Planet/62/Subsystems/MapInfo/Content/pro_help/datainterchange/symbolclausemapinfo30syntax.html>`_

        """
        return Style._wrap(ShadowPointStyle.create_mi_compat(symbol, color, pointSize))

    @staticmethod
    def create_mi_font(symbol: int = 36, color: QColor = Qt.red, size: int = 8,
                       fontname: str = 'Axioma MI MapSymbols', fontstyle: int = 0, rotation: float = 0.0) -> 'PointStyle':
        """Создание стиля на базе шрифта True Type.

        Args:
            symbol: Целое, имеющее значение 31 или больше, определяющее, какой используется символ из шрифтов TrueType. Для создания невидимого символа
                    используйте значение 31.
            color: Цвет символа
            pointSize: Целое число, размер символа в пунктах от 1 до 48;
            fontname: Строка с именем шрифта TrueType (например, значение по умолчание 'Axioma MI MapSymbols')
            fontstyle: Стиль дополнительного оформления, например, курсивный текст. Возможные параметры см. в таблице ниже. Для указания нескольких параметров 
                       их суммируют между собой.
            rotation: Угол поворота символа в градусах.

         .. csv-table:: Возможные значения параметра `fontstyle`
            :header: Значение, Наименование
            :widths: 10, 50

            0, Обычный текст
            1, Жирный текст
            16, Черная кайма вокруг символа
            32, Тень
            256, Белая кайма вокруг символа

        В системе доступны следующие стили:

        .. image:: /../images/style/style_point_mi_tt.png


        Подробнее: `Стиль True Type <http://webhelp.infovista.com/Planet/62/Subsystems/MapInfo/Content/pro_help/datainterchange/symbolclausetruetypefontsyntax.html>`_

        """
        return Style._wrap(ShadowPointStyle.create_mi_font(symbol, color, size, fontname, fontstyle, rotation))

    @staticmethod
    def create_mi_picture(filename: str, color: QColor = Qt.black, size: int = 12, customstyle: int = 0) -> 'PointStyle':
        """Создание стиля с ссылкой на растровый файл.

        Args:
            filename: Наименование растрового файла. Строка до 31 символа длиной. Данный файл должен находится в каталоге CustSymb с ресурсами. Например, `“Arrow.BMP”`.
            color: Цвет символа.
            size: Размер символа в в пунктах от 1 до 48.
            customstyle:  Задание дополнительных параметров стиля оформления. 

        .. csv-table:: Возможные значения параметра `customstyle`
            :header: Значение, Наименование
            :widths: 10, 70

            0, "Флажки Фон и Покрасить одним цветом не установлены. Символ показывается стандартно. Все белые точки изображения становятся прозрачными и под ними видны объекты Карты."
            1, "Установлен флажок Фон; все белые точки изображения становятся непрозрачными."
            2, "Установлен флажок Покрасить одним цветом все не белые точки изображения красятся в цвет символа."
            3, "Установлены флажки Фон и Покрасить одним цветом."


        Подробнее: `Стиль растрового символа <http://webhelp.infovista.com/Planet/62/Subsystems/MapInfo/Content/pro_help/datainterchange/symbolclausecustombitmapfilesyntax.html>`_
        """
        return Style._wrap(ShadowPointStyle.create_mi_picture(filename, color, size, customstyle))


class LineStyle(Style):
    """Стиль линейного объекта, совместимый с MapInfo.

    Args:
        pattern: Тип линии. Типы линий обозначаются кодами от 1 до 118. Тип 1 представляет собой невидимую линию.
        color: Цвет линии
        width: Толщина линии. Задается числом от 0 до 7, при этом линия нулевой ширины невидима на экране.
               11-2047 - это значения, которые могут быть преобразованы в пункты:
               ширина линии = (число пунктов * 10) + 10
               Значение 0 допустимо только для типа линии 1 или невидимых линий.

    В системе доступны следующие стили линии:

    .. image:: /../images/style/style_line.png

    Подробнее: `Стиль линейных объектов <http://webhelp.infovista.com/Planet/62/Subsystems/MapInfo/Content/pro_help/datainterchange/penstyles.html>`_

    .. literalinclude:: /../../tests/doc_examples/test_example_style.py
        :caption: Пример.
        :pyobject: test_run_example_style_line
        :lines: 2-
        :dedent: 4
    """

    def __init__(self, pattern: int = 2, color: QColor = Qt.black, width: int = 1):
        super().__init__()
        self._assign_shadow(ShadowLineStyle(pattern, color, width))


class PolygonStyle(Style):
    """Стиль площадного объекта. По умолчанию создается прозрачный стиль с черной окантовкой.

    .. literalinclude:: /../../tests/doc_examples/test_example_style.py
        :caption: Пример.
        :pyobject: test_run_example_style_poly
        :lines: 2-
        :dedent: 4

    Args:
        pattern: Номер стиля заливки.
        color: Цвет основной заливки.
        pattern_pen: Цвет обводки. По умолчанию обводка отсутствует.
    """

    def __init__(self, pattern: int = 1, color: QColor = Qt.white, pattern_pen: int = 1):
        super().__init__()
        self._assign_shadow(ShadowPolygonStyle(pattern, color, pattern_pen))

    def set_pen(self, pattern: int = 2, color: QColor = Qt.black, width: int = 1):
        """Задание стиля обводки. Параметры аналогичны при задании стиля линии :meth:`LineStyle`

        Args:
            pattern: Номер стиля линии.
            color: Цвет линии
            width: Толщина линии.
        """
        self.shadow.set_pen(pattern, color, width)

    def set_brush(self, pattern: int = 1, color: QColor = Qt.white, bgColor: QColor = Qt.transparent):
        """Задание стиля заливки площадного объекта.

        Args:
            pattern: Номер стиля заливки. Шаблон задается числом от 1 до 71, при этом в шаблоне с номером 1 оба цвета отсутствуют,
                     а в шаблоне 2 отсутствует цвет фона. Шаблоны с кодами 9-11 зарезервированы для внутренних
                     целей.
            color: Цвет основной заливки.
            bgColor: Цвет заднего фона, если заливка неполная.

        В системе доступны следующие стили заливки:

        .. image:: /../images/style/style_fill.png

        Подробнее: `Стиль заливки полигона <http://webhelp.infovista.com/Planet/62/Subsystems/MapInfo/Content/pro_help/datainterchange/brushstyles.html>`_
        """
        self.shadow.set_brush(pattern, color, bgColor)


class TextStyle(Style):
    """Стиль текстового объекта.

    Args:
        fontname: Наименование шрифта.
        size: Размер шрифта в пунктах. Может принимать значение 0 для подписей в окне карты, так как они являются атрибутами карты и их размер определяется динамически.
        style: Дополнительные параметры стиля. Подробнее см. в таблице ниже.
        color: Цвет шрифта
        backcolor: Цвет заднего фона, если он задан.

     .. csv-table:: Возможные значения параметра style
        :header: Значение, Наименование
        :widths: 10, 70

            0, Обычный
            1, Жирный
            2, Курсив
            4, Подчеркнутый
            16, Контур (только для `Macintosh`)
            32, Тень
            256, Кайма
            512, Капитель
            1024, Разрядка

    Подробнее: `Стиль текста <http://webhelp.infovista.com/Planet/62/Subsystems/MapInfo/Content/pro_help/datainterchange/fontstylesforsymbols.html>`_
    """

    def __init__(self, fontname: str, size: int, style: int = 0, forecolor: QColor = Qt.black, backcolor: QColor = Qt.transparent):
        super().__init__()
        self._assign_shadow(ShadowTextStyle(
            fontname, size, style, forecolor, backcolor))


class CollectionStyle(Style):
    """Смешанный стиль для разнородного типа объектов.

    Данный стиль представляет собой контейнер стилей. может применяться в купе с геометрическим объектом
    типа разнородная коллекция :class:`axipy.da.GeometryCollection`.
    Для задания или переопределения стилей простейших объектов, необходимо вызывать соответствующие методы для необходимых типов объектов.
    """

    def __init__(self):
        super().__init__()
        self._assign_shadow(ShadowCollectionStyle())

    def for_point(self, style: PointStyle):
        """Задание стиля для точечных объектов :class:`PointStyle`.  
        """
        self.shadow.for_point(style.shadow)

    def for_line(self, style: LineStyle):
        """Задание стиля для линейных объектов :class:`LineStyle`.  
        """
        self.shadow.for_line(style.shadow)

    def for_polygon(self, style: PolygonStyle):
        """Задание стиля для полигональных объектов :class:`PolygonStyle`.  
        """
        self.shadow.for_polygon(style.shadow)

    def for_text(self, style: TextStyle):
        """Задание стиля для текстовых объектов :class:`TextStyle`.  
        """
        self.shadow.for_text(style.shadow)

    def text(self):
        """Стиль для текстовых объектов :class:`TextStyle`.  
        """
        return Style._wrap(self.shadow.text())

    def line(self):
        """Стиль для линейных объектов :class:`LineStyle`.  
        """
        return Style._wrap(self.shadow.line())

    def polygon(self):
        """Стиль для полигональных объектов :class:`PolygonStyle`.  
        """
        return Style._wrap(self.shadow.polygon())

    def point(self):
        """Стиль для точечных объектов :class:`PointStyle`.  
        """
        return Style._wrap(self.shadow.point())

    def find_style(self, geom: Geometry):
        """Пытаемся найти стиль подходящий для переданной геометрии
        """
        return Style._wrap(self.shadow.style(geom.shadow))
