
from abc import ABCMeta, abstractmethod

import axipy
from axipy.cs.CoordSystem import CoordSystem
from axipy.da import Feature, Table, Type, GeometryCollection
from axipy.da.Geometry import Geometry
from axipy.interface import AxiomaInterface
from axipy.utl import Rect
from axipy.gui import selection_manager, view_manager, MapView
from axipy.app import Notifications
from typing import Iterator, List
from PySide2.QtGui import QTransform
from .helper import ensure_editable_table, quick_envelope
from axipy.concurrent import task_manager, AxipyProgressHandler, ProgressSpecification, ProgressGuiFlags

unsupported_types_string = ['Текст']
unsupported_types = [Type.Text]
unsupported_single_geom_types = [
    Type.Point,
    Type.Ellipse,
    Type.Rectange,
    Type.RoundedRectange]
unsupported_single_geom_types_string = [
    'Точка',
    'Эллипс',
    'Прямоугольник',
    'Скругленный прямоугольник']




class Mirror(metaclass=ABCMeta):
    """
    Базовый класс инструмента отражения геометрий.
    """

    def __init__(self, title: str, iface: AxiomaInterface) -> None:
        self.title = title
        self.iface = iface
    
    def mirror( self,
            handler: AxipyProgressHandler,
            editable_table: Table, view_cs: CoordSystem):
        features = self.features_to_map_view_cs( selection_manager.get_as_cursor(), view_cs)
        if not self.is_allowed_selection(features, view_cs):
            return
        # Получаем границы геометрии в координатах карты
        bound = quick_envelope([f.geometry for f in features]) # type: Rect
        transform = self.transform_impl(bound)
        handler.set_max_progress(len(features))
        for feature in features:
            handler.raise_if_canceled()
            geom = feature.geometry.affine_transform(transform)
            feature.geometry = geom
            handler.add_progress(1)
        handler.prepare_to_write_changes()
        editable_table.update(features)
        self.send_notification_about_finish(len(features))

    @abstractmethod
    def transform_impl(self, bound: Rect) -> QTransform:
        pass

    def on_triggered(self):
        editable_table = ensure_editable_table()
        map_view = view_manager.active  # type: MapView
        spec = ProgressSpecification(
            description=self.title,
            flags=ProgressGuiFlags.CANCELABLE)
        task_manager.run_and_get(
            spec,
            self.mirror,
            editable_table,
            map_view.coordsystem)


    def is_unsupported_geometry(self, geom: Geometry, view_cs: CoordSystem) -> bool:
        if geom.type not in unsupported_single_geom_types:
            return False
        if geom.type == Type.Point:
            return True
        else:
            # Если КС карты и геометрии отличаются то фигура будет полигоном и вероятно в
            # зеркальном отражении будет смысл
            return geom.coordsystem == view_cs


    def is_allowed_selection(self,
            features: List[Feature],
            view_cs: CoordSystem) -> bool:
        if len(features) == 0:
            self.send_notification_no_supported_geometry()
            return False
        if len(features) == 1 and features[0].has_geometry() \
                and self.is_unsupported_geometry(features[0].geometry, view_cs):
            self.send_notification_about_useless_signle_one_geom_mirror()
            return False
        return True


    def is_allowed(self, feature: Feature) -> bool:
        return feature.has_geometry() and feature.geometry.type not in unsupported_types \
            or issubclass(type(feature.geometry), GeometryCollection)


    def send_notification_no_supported_geometry(self):
        self.iface.notifications.push(title=self.title, text=axipy.tr(
            f'Не удалось обновить ни одну геометрию. Список запрещенных типов: '
            f'{", ".join(unsupported_types_string)}.'),
            type_mesage=Notifications.Critical)


    def send_notification_about_geom_conversation(self):
        self.iface.notifications.push(
            title=self.title,
            text=axipy.tr(
                'Часть геометрий были '
                'преобразованы в другой тип для выполнения операции.'),
            type_mesage=Notifications.Information)


    def send_notification_about_useless_signle_one_geom_mirror(self):
        self.iface.notifications.push(
            title=self.title,
            text=axipy.tr(
                f"Зеркальное отражение не имеет смысла для одиночных геометрий "
                f"типа: {', '.join(unsupported_single_geom_types_string)}."),
            type_mesage=Notifications.Information)


    def send_notification_about_finish(self, updated_cout: int):
        if updated_cout == 0:
            self.send_notification_no_supported_geometry()
            return
        self.iface.notifications.push(
            title=self.title,
            text=axipy.tr(f'Было преобразовано геометрий: {updated_cout}'),
            type_mesage=Notifications.Success)


    # Не зависимо от КС данных пользователь хочет чтобы геометрии отражались в текущих
    # координатах окна карты
    def features_to_map_view_cs(self,
            featuers: Iterator[Feature],
            view_cs: CoordSystem) -> List[Feature]:
        res = []  # type: List[Feature]
        convert_geom_counter = 0
        for feature in featuers:
            if not self.is_allowed(feature):
                continue
            if feature.geometry.type == Type.Arc:
                feature.geometry = feature.geometry.to_linestring()
                convert_geom_counter += 1
            if feature.geometry is None:
                continue
            # Преобразуем всю геометрию к координатам карты
            if feature.geometry.coordsystem != view_cs:
                feature.geometry = feature.geometry.reproject(view_cs)
            res.append(feature)
        if convert_geom_counter != 0:
            self.send_notification_about_geom_conversation()
        return res


class HorizontalMirror(Mirror):
    """
    Отразить по горизонтали.
    """

    def __init__(self, title: str, iface: AxiomaInterface) -> None:
        super().__init__(title, iface)

    def transform_impl(self, bound: Rect) -> QTransform:
        dy = bound.height / 2 + bound.ymin
        transform = QTransform().fromTranslate(0, 2 * dy)
        transform = transform.scale(1, -1)
        return transform


class VerticalMirror(Mirror):
    """
    Отразить по вертикали.
    """

    def __init__(self, title: str, iface: AxiomaInterface) -> None:
        super().__init__(title, iface)

    def transform_impl(self, bound: Rect) -> QTransform:
        dx = bound.width / 2 + bound.xmin
        transform = QTransform().fromTranslate(2 * dx, 0)
        transform = transform.scale(-1, 1)
        return transform