from PySide2.QtCore import QRectF, QObject, Signal
from PySide2.QtGui import QColor
from typing import List, Union

from axipy.cpp_render import Render, ShadowLayer, ShadowVectorLayer, ShadowRasterLayer, ShadowThematicList, ShadowThematic
from axipy.cpp_core_geometry import ShadowStyle

from axipy.da import DataObject
from axipy.da import Style
from axipy.cs import CoordSystem
from axipy.render.context import Context
from axipy.utl import Pnt, Rect
from axipy.decorator import axi_deprecated


class Layer:
    """Абстрактный базовый класс для слоя карты.

    Для создания нового экземпляра для векторного или растрового источника данных необходимо использовать метод :meth:`Layer.create`.
    Для тематических слоев - использовать соответствующие им конструкторы.
    """

    def __init__(self):
        super().__init__()
        raise NotImplementedError

    @classmethod
    def _wrap_typed(cls, _shadow):
        result = cls.__new__(cls)
        result.shadow = _shadow
        return result

    @classmethod
    def _wrap(cls, _shadow: ShadowLayer):
        if isinstance(_shadow, ShadowVectorLayer):
            return VectorLayer._wrap_typed(_shadow)
        if isinstance(_shadow, ShadowRasterLayer):
            return RasterLayer._wrap_typed(_shadow)
        if isinstance(_shadow, ShadowThematic):
            return ShadowThematic._wrap_typed(_shadow)
        return None

    @classmethod
    def create(cls, dataObject: DataObject) -> 'Layer':
        """Создает слой на базе открытой таблицы или растра.

        Args:
            dataObject: Таблица или растр. В зависимости от переданного объекта будет создан
                :class:`VectorLayer`  или :class:`RasterLayer`.

        .. literalinclude:: /../../tests/doc_examples/test_example_layer.py
            :caption: Пример создания слоя на базе файла.
            :pyobject: test_run_example_layer
            :lines: 3-
            :dedent: 4
        """
        from axipy.app import render_instance
        from axipy.da.DataObjectWrapper import Raster, Table
        if not render_instance:
            raise RuntimeError("axipy is not initialized")
        if isinstance(dataObject, Raster):
            shadow = ShadowRasterLayer(render_instance, dataObject.shadow)
        elif dataObject.is_spatial:
            if isinstance(dataObject, Table):
                shadow = ShadowVectorLayer(render_instance, dataObject.shadow)
            else:
                shadow = ShadowRasterLayer(render_instance, dataObject.shadow)
        else:
            raise ValueError("Non spatial table")
        if shadow is None:
            msg = 'Cannot create layer.'
            if dataObject.shadow is not None and not dataObject.shadow.isSpatial():
                msg = msg + ' Table is not spatial.'
            raise ValueError(msg)
        res = cls._wrap(shadow)
        if res is None:
            raise TypeError("Invalid layer type")
        return res

    @property
    def data_changed(self) -> Signal:
        """``Signal[]`` Сигнал об изменении контента слоя."""
        return self.shadow.dataChanged

    @property
    def need_redraw(self) -> Signal:
        """``Signal[]`` Сигнал о необходимости перерисовать слой.
        """
        return self.shadow.needRedraw

    @property
    def title(self) -> str:
        """Наименование слоя."""
        return self.shadow.get_name()

    @title.setter
    def title(self, n: str):
        self.shadow.set_name(n)

    @property
    def coordsystem(self) -> CoordSystem:
        """Координатная система, в которой находятся данные, отображаемые слоем."""
        return CoordSystem._wrap(self.shadow.cs())

    @property
    @axi_deprecated('get_bounds')
    def bounds(self) -> Rect:
        return self.get_bounds()

    def get_bounds(self) -> Rect:
        """Возвращает область, в которую попадают все данные, которые могут быть отображены на слое."""
        return Rect.from_qt(self.shadow.bounds())

    @property
    def data_object(self) -> DataObject:
        """Источник данных для слоя."""
        return DataObject._wrap(self.shadow.data_object())

    @property
    def opacity(self) -> int:
        """Прозрачность слоя в составе карты. Доступные значения от 0 до 100."""
        return self.shadow.get_opacity()

    @opacity.setter
    def opacity(self, n: int):
        self.shadow.set_opacity(n)

    @property
    def zoom_restrict(self) -> bool:
        """Будет ли использоваться ограничение по отображению. Если установлено True, то для ограничения отображения слоя в 
        зависимости от масштаба используются значения свойств `zoom_min` и `zoom_max`"""
        return self.shadow.zoomRestrict()

    @zoom_restrict.setter
    def zoom_restrict(self, v: bool):
        self.shadow.setZoomRestrict(v)

    @property
    def min_zoom(self) -> float:
        """Минимальная ширина окна, при котором слой отображается на карте. Учитывается только при установленном `zoom_restrict=True`"""
        return self.shadow.minZoom()

    @min_zoom.setter
    def min_zoom(self, v: float):
        self.shadow.setZoomMin(v)

    @property
    def max_zoom(self) -> float:
        """Максимальная ширина окна, при котором слой отображается на карте. Учитывается только при установленном `zoom_restrict=True`"""
        return self.shadow.maxZoom()

    @max_zoom.setter
    def max_zoom(self, v: float):
        self.shadow.setZoomMax(v)

    def __eq__(self, other) -> bool:
        if not isinstance(other, Layer):
            return False
        return ShadowLayer.isEquals(self.shadow, other.shadow if other.shadow is not None else None)

    def __str__(self):
        return str(type(self))

    @property
    def visible(self):
        """Управляет видимостью слоя.
        
        Выключение видимости верхнего слоя для активной карты::

            if view_manager.active is not None:
                view_manager.active.map.layers[0].visible = False
        """
        return self.shadow.is_visible()

    @visible.setter
    def visible(self, v):
        self.shadow.set_visible(v)


class ThematicLayer(Layer):
    """Абстрактный класс слоя с тематическим оформлением векторного слоя карты на базе атрибутивной информации.
    """
    @classmethod
    def _wrap(cls, shadow: ShadowThematic):
        from .thematic_layer import (RangeThematicLayer,
                                     PieThematicLayer,
                                     BarThematicLayer,
                                     SymbolThematicLayer,
                                     DensityThematicLayer,
                                     IndividualThematicLayer)
        from axipy.cpp_render import (ShadowThematicRange,
                                      ShadowThematicPie,
                                      ShadowThematicBar,
                                      ShadowThematicSymbol,
                                      ShadowThematicIndividual,
                                      ShadowThematicDensity)
        if isinstance(shadow, ShadowThematicRange):
            return RangeThematicLayer._wrap(shadow)
        elif isinstance(shadow, ShadowThematicPie):
            return PieThematicLayer._wrap(shadow)
        elif isinstance(shadow, ShadowThematicBar):
            return BarThematicLayer._wrap(shadow)
        elif isinstance(shadow, ShadowThematicSymbol):
            return SymbolThematicLayer._wrap(shadow)
        elif isinstance(shadow, ShadowThematicIndividual):
            return IndividualThematicLayer._wrap(shadow)
        elif isinstance(shadow, ShadowThematicDensity):
            return DensityThematicLayer._wrap(shadow)
        print('Shadow is unknown or null:', shadow)
        return None


class ListThematic:
    """Список тематических слоев (тематик) карты."""

    def __init__(self):
        raise NotImplementedError

    @classmethod
    def _wrap(cls, _shadow: ShadowThematicList):
        result = ListThematic.__new__(ListThematic)
        result.shadow = _shadow
        return result

    def append(self, lay: ThematicLayer):
        """Добавить тематику.

        Args:
            lay: Добавляемый тематический слой.
        """
        self.shadow.add(lay.shadow)

    def add(self, lay: ThematicLayer):
        self.shadow.add(lay.shadow)

    def remove(self, idx: int):
        """Удалить тематику.

        Args:
            idx: Индекс удаляемого слоя.
        """
        self.shadow.remove(idx)

    def move(self, fromIdx: int, toIdx: int):
        """Поменять тематики местами.

        Args:
            fromIdx: Текущий индекс.
            toIdx: Новое положение.
        """
        self.shadow.move(fromIdx, toIdx)

    @property
    def count(self) -> int:
        """Количество тематик слоя."""
        return self.__len__()

    def at(self, idx: int) -> ThematicLayer:
        """Получение тематики по ее индексу.

        Args:
            idx: Индекс запрашиваемой тематики.
        """
        return ThematicLayer._wrap(self.shadow.at(idx))

#    def by_name(self, name: str) -> ThematicLayer:
#        """Получение тематики по ее наименованию.
#
#        Args:
#            name: Наименование запрашиваемой тематики.
#
#        Returns:
#            Если не найдено, возвращает None.
#        """
#        return ThematicLayer._wrap(self.shadow.by_name(name))

    def __getitem__(self, item: Union[int, str]):
        if isinstance(item, str):
            return self.shadow.by_name(item)
        return self.at(item)

    def __len__(self):
        return self.shadow.count()

    def __iter__(self):
        return (self.at(idx) for idx in range(self.__len__()))


class Label:
    """Метки слоя. Доступны через свойство векторного слоя :attr:`label`.
    """

    def __init__(self):
        raise NotImplementedError

    @classmethod
    def _wrap(cls, shadow):
        result = cls.__new__(cls)
        result.shadow = shadow
        return result

    @property
    def text(self) -> str:
        """Наименование атрибута таблицы либо выражение для метки, которое может основываться на одном или нескольких атрибутах."""
        return self.shadow.get_label()

    @text.setter
    def text(self, n: str):
        self.shadow.set_label(n)

    @property
    def visible(self) -> bool:
        """Управляет видимостью меток."""
        return self.shadow.get_label_visible()

    @visible.setter
    def visible(self, v: bool):
        self.shadow.set_label_visible(v)

    ALLOW_OVERLAP = 0
    DISALLOW_OVERLAP = 1
    TRY_OTHER_POSITION = 2

    @property
    def placementPolicy(self) -> int:
        """Принцип наложения меток на слой карты.

        .. csv-table:: Допустимые значения:
            :header: "Константа", "Значение", "Описание"
            :align: left
            :widths: 10, 5, 40

            ALLOW_OVERLAP, 0, "Допускать перекрытие меток (по умолчанию)"
            DISALLOW_OVERLAP, 1, "Не допускать перекрытие меток"
            TRY_OTHER_POSITION, 2, "Пробовать найти для метки новую позицию"

        """
        return self.shadow.get_label_placement_policy()

    @placementPolicy.setter
    def placementPolicy(self, n: int):
        self.shadow.set_label_placement_policy(n)


class VectorLayer(Layer):
    """Слой, основанный на базе векторных данных.

    Note:
        Создание слоя производится посредством метода вызова :meth:`Layer.create`

    .. literalinclude:: /../../tests/doc_examples/test_example_layer.py
        :caption: Примеры работы со свойствами слоя.
        :pyobject: test_run_example_layer_props
        :lines: 3-
        :dedent: 4

    """

    @classmethod
    def _wrap_typed(cls, _shadow: ShadowVectorLayer):
        result = cls.__new__(cls)
        result.shadow = _shadow
        #self.label = self.Label(self)
        result._thematics = None
        result._label = None
        return result

    @property
    def showCentroid(self) -> bool:
        """Показ центроидов на слое."""
        return self.shadow.get_show_centroid()

    @showCentroid.setter
    def showCentroid(self, n: bool):
        self.shadow.set_show_centroid(n)

    @property
    def nodesVisible(self) -> bool:
        """Показ узлов линий и полигонов."""
        return self.shadow.get_nodesVisible()

    @nodesVisible.setter
    def nodesVisible(self, n: bool):
        self.shadow.set_nodesVisible(n)

    @property
    def linesDirectionVisibile(self) -> bool:
        """Показ направлений линий."""
        return self.shadow.get_linesDirectionVisibile()

    @linesDirectionVisibile.setter
    def linesDirectionVisibile(self, n: bool):
        self.shadow.set_linesDirectionVisibile(n)

    @property
    def overrideStyle(self) -> Style:
        """Переопределяемый стиль слоя. Если задан как None (по умолчанию), объекты будут отображены на 
        основании оформления источника данных.
        """
        return Style._wrap(self.shadow.get_override_style())

    @overrideStyle.setter
    def overrideStyle(self, style: Style):
        self.shadow.set_override_style(
            ShadowStyle() if style is None else style.shadow)

    @property
    def thematic(self) -> ListThematic:
        """Перечень тематик для данного слоя. Работа с тематическими слоями похожа на работу со списком `list`.
        
        .. literalinclude:: /../../tests/doc_examples/test_example_layer.py
            :caption: Пример.
            :pyobject: test_run_example_layer_thematic
            :lines: 3-11
            :dedent: 4
        
        """
        if self._thematics is None:
            self._thematics = ListThematic._wrap(self.shadow.get_thematics())
        return self._thematics

    @property
    def label(self) -> Label:
        """Метки слоя. В качестве формулы может использоваться или наименование поля таблицы или выражение."""
        if self._label is None:
            self._label = Label._wrap(self.shadow)
        return self._label


class RasterLayer(Layer):
    """Класс, который должен использоваться в качестве базового класса для тех слоев, в которых используются
    свойства отрисовки растрового изображения.

    Note:
        Создание слоя производится посредством метода вызова :meth:`Layer.create`

    .. literalinclude:: /../../tests/doc_examples/test_example_layer.py
        :caption: Примеры создания растрового слоя.
        :pyobject: test_run_example_layer_raster
        :lines: 3-
        :dedent: 4
    """

    @property
    def transparentColor(self) -> QColor:
        """Цвет растра, который обрабатывается как прозрачный."""
        return self.shadow.get_transparentColor()

    @transparentColor.setter
    def transparentColor(self, cl: QColor):
        self.shadow.set_transparentColor(cl)

    @property
    def brightness(self) -> int:
        """Яркость. Значение может быть в пределах от -100 до 100."""
        return self.shadow.get_brightness()

    @brightness.setter
    def brightness(self, v: int):
        self.shadow.set_brightness(v)

    @property
    def contrast(self) -> int:
        """Контраст. Значение может быть в пределах от -100 до 100."""
        return self.shadow.get_contrast()

    @contrast.setter
    def contrast(self, v: int):
        self.shadow.set_contrast(v)

    @property
    def grayscale(self) -> bool:
        """Является ли данное изображение черно-белым."""
        return self.shadow.get_grayscale()

    @grayscale.setter
    def grayscale(self, v: bool):
        self.shadow.set_grayscale(v)
