import inspect, os
import gc
from PyQt5.QtCore import QObject, QDate,  Qt
from PyQt5.QtWidgets import QWidget, QTableWidgetItem
from PyQt5.uic import loadUi
import axioma.app
import axioma.render
from axioma.core.geometry import *

'''
Пример просмотра информации по открытым таблицам
'''


class TableInspector(QWidget):

    def __init__(self, axiomaWindow, parent=None):
        QWidget.__init__(self, parent)
        self.setAttribute(Qt.WA_DeleteOnClose)
        assert(type(axiomaWindow) is axioma.app.MainWindow)
        # trick to get .ui file from dir w/ script
        cwd = os.path.dirname(os.path.abspath(inspect.getfile(inspect.currentframe())))
        self.__ui = loadUi(os.path.join(cwd, "TableInspector.ui"), self)
        self.__tables = dict()
        self.__axiomaWindow = axiomaWindow
        self.initUi()
        self.onRefresh()


    def initUi(self):
        #connects
        self.__ui.btnQuit.clicked.connect(self.close)
        self.__axiomaWindow.signalClosed.connect(self.close)
        self.__ui.btnRefresh.clicked.connect(self.onRefresh)
        self.__ui.cbTables.currentIndexChanged[str].connect(self.onCurrentTableChanged)

    def clear_data(self):
        table_widget = self.__ui.tableWidget
        table_widget.clear()
        table_widget.setColumnCount(0)
        table_widget.setRowCount(0)
        self.__ui.leFeatureCount.clear()

    def set_data_from_table(self, table): # Запрос и представление данных в табличной форме.
        tableWidget = self.__ui.tableWidget
        if not table: # if table is None
            self.clear_data()
            return
        features = table.allFeatures()
        row_count = table.count()
        #exclude geometry and style (-2)
        col_count = table.tableSchema().numAttributes() - 2
        self.__ui.leFeatureCount.setText(str(row_count))
        tableWidget.setRowCount(row_count)
        tableWidget.setColumnCount(col_count)
        tableWidget.setHorizontalHeaderLabels(table.attributeNames()[:-2])
        for i in range(row_count):
            feature = features[i]
            for j in range(col_count):
                try:
                    attr = feature.getAttribute(j)
                    if type(attr) is QDate:
                        attr = attr.toString("dd.MM.yyyy")
                except TypeError:
                    attr = "N/A"
                tableWidget.setItem(i, j, QTableWidgetItem(str(attr)))
        tableWidget.setVisible(False)
        tableWidget.resizeColumnsToContents()
        tableWidget.setVisible(True)

    #SLOTS
    def updateForm(self):
        self.__ui.cbTables.clear()
        self.__ui.cbTables.addItems(list(self.__tables.keys()))

    # Запрос перечня таблиц
    def onRefresh(self):
        #get tables and update
        self.__tables.clear()
        self.__tables = {table.name():table for table in self.__axiomaWindow.listTables()}
        self.updateForm()

    def onCurrentTableChanged(self, key):
        curTable = self.__tables.get(key)
        self.set_data_from_table(curTable)


class Starter(QObject):
    """Helper class to collect garbage after execution"""

    def __init__(self, parent=None):
        QObject.__init__(self, parent)
        gc.collect()

        #Execution code
        inspector = TableInspector(axioma.app.mainWindow)
        inspector.setWindowIcon(axioma.app.mainWindow.windowIcon())
        inspector.show()
        inspector.move(300, 200)
        self.handled_object = inspector
        #Execution code

        self.handled_object.destroyed.connect(self.clean_up)

    def clean_up(self):
        self.handled_object = None
        gc.collect()

starter = Starter()
