import axioma
import axioma.gui
import sys, os
from PyQt5.QtCore import Qt
from PyQt5.QtGui import QIcon
from PyQt5.QtWidgets import QMainWindow, QApplication, QFileDialog, QMessageBox, QAction
from PyQt5.uic import loadUi
from PyQt5.QtCore import pyqtSlot
import axioma.core
from axioma.core.dp import DataCatalog, FileDefinition, FileDataProvider
import axioma.render
from axioma.gui import tool_id
import rc
from ui_MainWindow import Ui_MainWindow

'''
Пример простейшего приложения с использованием SDK.
Данное приложение позволяет открыть карту и использовать на ее основе некоторые базовые инструменты.
Запускается из сторонней среды разработки (например, PyCharm).   

Файл ui_MainWindow.py получен выполнением команды:
pyuic5 MainWindow.ui > ui_MainWindow.py

'''

# Главное окно приложения
class MainWindow(QMainWindow):
    def __init__(self, gui, parent=None):
        """Конструктор"""
        assert isinstance(gui, axioma.gui.Gui)
        QMainWindow.__init__(self, parent)
        self.setAttribute(Qt.WA_DeleteOnClose)
        self.__ui = Ui_MainWindow()
        self.__ui.setupUi(self)

        self.__gui = gui
        self.__data_catalog = DataCatalog(self)
        self.initUi()


    def initUi(self):
        self.setWindowIcon(QIcon("://icons/share/32px/logo.png"))
        self.__ui.actionOpen.setIcon(QIcon.fromTheme("open"))
        self.__ui.actionQuit.setIcon(QIcon.fromTheme("exit"))
        # actions
        self.__ui.toolBar.addAction(self.__ui.actionOpen)
        self.__ui.toolBar.addAction(self.__ui.actionQuit)

        # Сигналы - слоты
        self.__ui.actionOpen.triggered.connect(self.onOpenTriggered)
        self.__data_catalog.dataObjectAdded.connect(self.onDataObjectAdded)
        self.__ui.tabWidget.tabCloseRequested[int].connect(self.onTabCloseRequest)

    def check_file_already_open(self, file_def):
        """Проверка, если данные уже открыты"""
        for dsrc in self.__data_catalog.allDataSources():
            if dsrc.dataSourceDefinition().isSameSource(file_def):
                return True
        return False

    def open_file(self, file):
        """Открыть файл"""
        if self.check_file_already_open(FileDefinition(file)):
            QMessageBox.warning(self, "Предупреждение", "Файл %s уже открыт" % file)
            return

        def quick_open_file(gui, file):
            """Функция открытия файла"""
            ext = os.path.splitext(file)[1][1:].strip().lower()
            data_providers = gui.dataProviders()
            file_providers = [file_dp for file_dp in data_providers if isinstance(file_dp, FileDataProvider)]
            for provider in file_providers:
                if len(provider.fileTypeCategories().categoriesForExtension(ext)):
                    data_source = provider.quickOpenFile(file)
                    return data_source.openDataObject()

        data_object = quick_open_file(self.__gui, file)
        assert data_object
        self.__data_catalog.addDataObject(data_object)

    def init_tool_bar(self):
        tool_names = [tool_id.select, tool_id.pan, tool_id.info, tool_id.ruler, tool_id.zoom_in, tool_id.zoom_out]
        action_manager = self.__gui.actionManager()
        tool_bar = self.__ui.toolBar
        tool_bar.addSeparator()
        for tool_name in tool_names:
            action = action_manager.actionByName(tool_name)
            if action:
                self.__ui.toolBar.addAction(action)
        tool_bar.addSeparator()
        snap_action = tool_bar.addAction(QIcon(), "Узлы")
        snap_action.setCheckable(True)
        snap_action.setObjectName("snapAction")
        snap_action.toggled.connect(self.onSnapChecked)

    @pyqtSlot(bool)
    def onSnapChecked(self, checked):
        """Режим привязки к координатам"""
        map_views = self.__ui.tabWidget.findChildren(axioma.gui.MapView)
        for map_view in map_views:
            map_view.setSnapMode(checked)

    # slots
    @pyqtSlot()
    def onOpenTriggered(self):
        """Действие на нажатие кнопки Открыть"""
        file_dialog = QFileDialog(self)
        file_dialog.setFileMode(QFileDialog.ExistingFiles)
        cwd = os.path.dirname(__file__)
        filter_str = "Tab файлы (*.tab);;Растры (*.jpg *.png *.bmp *.tif);;" \
                     "Файлы рабочего набора (*.mws);;Все файлы (*.*)"
        file = QFileDialog.getOpenFileName(self, "Открыть файл ...", cwd, filter_str)
        if len(file[0]):
            self.open_file(file[0])

    @pyqtSlot(axioma.core.dp.DataObject)
    def onDataObjectAdded(self, dobj):
        """ Отрабатывается при добавлении объекта в каталог данных"""
        self.__ui.widgetDataObjectList.addItem(dobj.name())
        layer = self.__gui.createLayerForDataObject(dobj)
        map = axioma.render.Map([layer])
        map_view = self.__gui.createMapView(map)
        idx = self.__ui.tabWidget.addTab(map_view, dobj.name())
        snap_action = self.__ui.toolBar.findChild(QAction, "snapAction")
        map_view.setSnapMode(snap_action.isChecked())

        select_action = self.__gui.actionManager().actionByName("select")
        select_action.trigger()


    @pyqtSlot(int)
    def onTabCloseRequest(self, index): # По закрытию вкладки
        """Запрос на закрытие"""
        self.__ui.widgetDataObjectList.takeItem(index) # удаляем из списка
        tab_name = self.__ui.tabWidget.tabText(index)
        tab_name = tab_name.replace("&", "")
        obj = self.__data_catalog.findDataObjectByName(tab_name)
        if obj is not None:
            obj.closeRequest()


if __name__ == '__main__':
    a = QApplication(sys.argv)

    # Создаем и инициализируем ядро
    gui = axioma.gui.Gui(a)
    gui.initialize()

    # Создаем главное окно
    mainWindow = MainWindow(gui)
    mainWindow.init_tool_bar()

    mainWindow.show()
    sys.exit(a.exec())
