from PySide2.QtCore import QPointF
from typing import Union, List


class Pnt:

    """Точка без геопривязки. Может быть использована в качестве параметра геометрии (точки полигона) или при получении параметров, 
    где результат представлен в виде точки (центр карты или элемента отчета).

    Args:
        x: X координата.
        y: Y координата.
    """

    def __init__(self, x: float, y: float):
        self._point = QPointF(x, y)

    @classmethod
    def from_qt(cls, p: QPointF):
        if not isinstance(p, QPointF):
            return None
        obj = cls.__new__(cls)
        obj._point = p
        return obj

    def to_qt(self):
        return self._point

    # Преобразование из tuple в Pnt если это возможно
    @staticmethod
    def _fixPnt(v) -> 'Pnt':
        if isinstance(v, tuple) and len(v) == 2:
            return Pnt(v[0], v[1])
        return v

    @classmethod
    def _point_value_to_qt(cls, v: Union[QPointF, tuple, 'Pnt', List]) -> Union[QPointF, List[QPointF]]:
        if isinstance(v, tuple) and len(v) == 2:
            return QPointF(v[0], v[1])
        elif isinstance(v, Pnt):
            return v.to_qt()
        elif isinstance(v, list):
            return [cls._point_value_to_qt(p) for p in v]
        return v

    @classmethod
    def _point_value_from_qt(cls, v: Union[QPointF, tuple, 'Pnt', List]) -> Union['Pnt', List['Pnt']]:
        if isinstance(v, tuple) and len(v) == 2:
            return Pnt(v[0], v[1])
        elif isinstance(v, QPointF):
            return Pnt.from_qt(v)
        elif isinstance(v, list):
            return [cls._point_value_from_qt(p) for p in v]
        return v

    @property
    def x(self) -> float:
        """Координата X."""
        return self._point.x()

    @x.setter
    def x(self, v: float):
        self._point.setX(v)

    @property
    def y(self) -> float:
        """Координата Y."""
        return self._point.y()

    @y.setter
    def y(self, v: float):
        self._point.setY(v)

    def __eq__(self, other):
        import math
        if isinstance(other, Pnt):
            return math.isclose(self.x, other.x) and math.isclose(self.y, other.y)
        if isinstance(other, tuple) and len(other) == 2:
            return math.isclose(self.x, other[0]) and math.isclose(self.y, other[1])
        return False

    def __str__(self):
        return "({} {})".format(self.x, self.y)
