"""Модуль меню главного окна ГИС "Аксиома"."""

from typing import Union, Type, Optional, Any, Callable
import inspect
from PySide2.QtWidgets import QAction
from PySide2.QtGui import QIcon
from axipy.cpp_gui import ShadowMenuBar as MenuBar, ShadowToolFactory as IToolFactory
from axipy.decorator import InitOnce
from axipy.gui import gui_instance, MapTool
from axipy.da import DefaultKeys, state_manager


__all__ = [
    'Button',
    'ActionButton',
    'ToolButton',
    'Position',
    'create_button',
    'remove',
    'get_position',
]


def title_to_id(title: str) -> str:
    # Creates identifier from title.
    table = str.maketrans(
        "абвгдеёжзийклмнопрстуфхцчшщъыьэюяАБВГДЕЁЖЗИЙКЛМНОПРСТУФХЦЧШЩЪЫЬЭЮЯ ",
        "abvgdeejzijklmnoprstufhzcss_y_euaABVGDEEJZIJKLMNOPRSTUFHZCSS_Y_EUA_"
    )
    return title.translate(table)


@InitOnce
def _service():
    return MenuBar(gui_instance.shadow)


class Button:
    """Кнопка с инструментом для добавления в меню. Абстрактный класс.

    Для создания объекта этого класса используйте
    :class:`axipy.menubar.ActionButton`, :class:`axipy.menubar.ToolButton`.
    """
    def __init__(self):
        if type(self) is Button:
            raise NotImplementedError

    def _assign(self, action: QAction, factory: Callable[[], MapTool] = None, observer_id: str = ''):
        self.__action = action
        self.factory = factory
        self.__observer_id = observer_id

    @classmethod
    def create_action(cls, title: str, icon: Union[str, QIcon] = ''):
        if not isinstance(icon, QIcon):
            icon = QIcon(icon)
        action = QAction(icon, title)
        action.setObjectName(title_to_id(title))
        return action

    @property
    def action(self) -> QAction:
        """Ссылка на объект :class:`~PySide2.QtWidgets.QAction`. Через него можно производить дополнительные необходимые действия через объект `Qt`.

        Пример задания всплывающей подсказки, используя метод класса :class:`~PySide2.QtWidgets.QAction`::

            button.action.setToolTip('Всплывающая подсказка')
        """
        return self.__action

    @property
    def observer_id(self) -> str:
        """Идентификатор наблюдателя для определения доступности инструмента."""
        return self.__observer_id

    def remove(self):
        """Удаляет кнопку из меню."""
        _service().remove(self.action)

class ActionButton(Button):
    """Кнопка с действием.
    
    Args:
        title: Текст.
        on_click: Действие на нажатие. Делегируется функция, которая будет вызвана при активации инструмента.
        icon: Иконка. Может быть путем к файлу или адресом ресурса.
        enable_on: Идентификатор наблюдателя для определения доступности кнопки. Если это пользовательский наблюдатель,
        то указывается его наименование при создании.
        tooltip: Строка с дополнительной короткой информацией по данному действию.

    See also:
        :class:`axipy.da.StateManager`.

    .. literalinclude:: /../../tests/doc_examples/test_example_menubar.py
        :caption: Пример со встроенным наблюдателем.
        :pyobject: test_run_example_menubar_button
        :lines: 2-
        :dedent: 4

    .. literalinclude:: /../../tests/doc_examples/test_example_menubar.py
        :caption: Пример со пользовательским наблюдателем.
        :pyobject: test_run_example_menubar_button_observer
        :lines: 2-
        :dedent: 4
    """
    def __init__(self, title: str, on_click: Callable[[], Any], icon: Union[str, QIcon] = '', enable_on: Union[str, DefaultKeys] = None, tooltip: str = None):
        action = self.create_action(title, icon)
        if on_click is not None:
            action.triggered.connect(on_click)
        if tooltip is not None:
            action.setToolTip(tooltip)
        self._assign(action, None, state_manager._to_name(enable_on))


class ToolButton(Button):
    """Кнопка с инструментом.
    
    Args:
        title: Текст.
        on_click: Класс инструмента.
        icon: Иконка. Может быть путем к файлу или адресом ресурса.
        enable_on: Идентификатор наблюдателя для определения доступности кнопки.

    See also:
        :class:`axipy.da.StateManager`.

    .. literalinclude:: /../../tests/doc_examples/test_example_menubar.py
        :caption: Пример
        :pyobject: test_run_example_menubar_tbutton
        :lines: 6-
        :dedent: 4
    """
    def __init__(self, title: str, on_click: Callable[[], MapTool], icon: Union[str, QIcon] = '', enable_on: Union[str, DefaultKeys] = None):
        action = self.create_action(title, icon)
        if inspect.isclass(on_click) and issubclass(on_click, MapTool):
            enable_on = enable_on or on_click.enable_on
        else:
            if not isinstance(on_click, Callable):
                raise ValueError('Object is not callable')
            signature = inspect.signature(on_click)
            if len(signature.parameters) != 0:
                raise ValueError('Callable must be without parameters')
        self._assign(action, on_click, state_manager._to_name(enable_on))


def create_button(title: str, on_click: Callable[[], MapTool], icon: Union[str, QIcon] = '', enable_on: Union[str, DefaultKeys] = None) -> Button:
    """Создает кнопку с заданными параметрами.

    От параметра `on_click` зависит, будет создан инструмент или простое действие.

    Args:
        title: Текст.
        on_click: Класс инструмента или действие на нажатие.
        icon: Иконка. Может быть путем к файлу или адресом ресурса.
        enable_on: Идентификатор наблюдателя для определения доступности кнопки.
    Return:
        Кнопка с инструментом.

    See also:
        :class:`axipy.da.StateManager`.

    Пример::

        tool = create_button("Мой инструмент", on_click=MyTool)
        action = create_button("Мое действие", on_click=lambda: print('clicked'))

    Warning:
    
        .. deprecated:: 3.5
            Используйте конструктор наследников :class:`axipy.menubar.Button`.
    """
    if inspect.isclass(on_click) and issubclass(on_click, MapTool):
        return ToolButton(title, on_click, icon, enable_on)
    return ActionButton(title, on_click, icon, enable_on)

def remove(button: Union[QAction, Button]):
    """Удаляет кнопку из меню.

    Args:
        button: Удаляемая кнопка.

    Warning:
    
        .. deprecated:: 3.5
            Используйте метод :meth:`axipy.menubar.Button.remove`.
    """
    if isinstance(button, Button):
        button = button.action
    _service().remove(button)


def _find_tab_id(tab_name: str):
    tabs = _service().tabs()
    if tab_name not in tabs:
        for key, value in tabs.items():
            if tab_name == value:
                return key
    return None


def _find_group_id(name: str):
    pairs = _service().groups()
    if name not in pairs:
        for key, value in pairs.items():
            if name == value:
                return key
    return None


def _create_tool(factory: Callable[[], MapTool]):
    class ToolFactory(IToolFactory):
        def create(self) -> MapTool:
            tool = factory()
            tool.setParent(self)
            return tool

    return ToolFactory()


class Position:
    """Положение кнопки в меню.

    Args:
        tab: Название вкладки.
        group: Название группы.

    Пример::

        pos = Position("Основные", "Команды")
    """

    def __init__(self, tab: str, group: str):
        self._tab = tab
        self._group = group

    def add(self, button: Union[QAction, Button], size: int = 2):
        """Добавляет кнопку текущее положение.

        Args:
            button: Кнопка.
            size: Размер кнопки. Маленькая кнопка - ``1``. Большая кнопка - ``2``.
        """
        button_def = button
        if isinstance(button_def, QAction):
            button_def = Button(button_def)
        tab_name, group_name = self._tab, self._group
        tab_id = _find_tab_id(tab_name)
        if tab_id is None:
            tab_id = title_to_id(tab_name)
        group_id = _find_group_id(group_name)
        if group_id is None:
            group_id = title_to_id(group_name)
        tool_factory = None
        if button_def.factory is not None:
            tool_factory = _create_tool(button_def.factory)
        _service().add_button(button_def.action, tool_factory, tab_id,
                              tab_name, group_id, group_name,
                              button_def.observer_id, size)


def get_position(tab: str, group: str) -> Position:
    """Возвращает положение в меню. Может заранее не существовать.

    Args:
        tab: Название вкладки.
        group: Название группы.

    Return:
        Положение для кнопки.

    Пример::

        pos = get_position("Основные", "Команды")

    Warning:
    
        .. deprecated:: 3.5
            Используйте конструктор :class:`axipy.menubar.Position`.
    """
    return Position(tab, group)


def add(button: Union[QAction, Button], position: Position, size=2):
    return position.add(button, size)
