from axipy.cpp_gui import ShadowView, ShadowMapView, ShadowTableView, ShadowReportView, ShadowDrawableView, ShadowLegendView
from PySide2.QtWidgets import QWidget
from axipy.da import DataObject
from axipy.render import Map, Layer, VectorLayer, Report
from axipy.cs import CoordSystem, LinearUnit
from PySide2.QtCore import QRectF, Signal
from PySide2.QtGui import QTransform
from typing import Optional
from axipy.render.legend import Legend
from axipy.utl import Rect, Pnt


class View:
    """Базовый класс для отображения данных в окне."""

    def __init__(self):
        raise NotImplementedError

    @classmethod
    def _wrap_typed(cls, shadow):
        result = cls.__new__(cls)
        result.shadow = shadow
        return result

    @classmethod
    def _wrap(cls, obj: ShadowView):
        if isinstance(obj, ShadowTableView):
            return TableView._wrap_typed(obj)
        elif isinstance(obj, ShadowMapView):
            return MapView._wrap_typed(obj)
        elif isinstance(obj, ShadowReportView):
            return ReportView._wrap_typed(obj)
        elif isinstance(obj, ShadowLegendView):
            return LegendView._wrap_typed(obj)
        return None

    @property
    def widget(self) -> QWidget:
        """Виджет, соответствующий содержимому окна.

        Return:
            Qt5 виджет содержимого.
        """
        return self.shadow.widget()

    @property
    def title(self) -> str:
        """Заголовок окна просмотра."""
        return self.shadow.widget().windowTitle()

    @title.setter
    def title(self, v: str):
        self.shadow.widget().setWindowTitle(v)

    def __str__(self):
        return str(type(self))


class TableView(View):
    """Таблица просмотра атрибутивной информации.
    Для создания экземпляра необходимо использовать :meth:`axipy.gui.ViewManager.create_tableview` через экземпляр `view_manager`
    """

    @property
    def data_object(self) -> DataObject:
        """Таблица, на основании которой создается данное окно просмотра.

        Return:
            Таблица.
        """
        return DataObject._wrap(self.shadow.dataobject())


class DrawableView(View):
    """"Базовый класс для визуализации геометрических данных."""

    def __init__(self):
        raise NotImplementedError

    @property
    def snap_mode(self) -> bool:
        """Включает режим привязки координат при редактировании геометрии в окне карты или отчета."""
        return self.shadow.isSnapMode()
    
    @snap_mode.setter
    def snap_mode(self, v):
        self.shadow.setSnapMode(v)

    def scale_with_center(self, zoom: float, center: Pnt):
        """Установка нового центра с заданным масштабированием.
        
        Args:
            zoom: Коэффициент масштабирования по отношению к текущему.
            center: Устанавливаемый центр.
        """
        self.shadow.scaleWithCenter(zoom, Pnt._point_value_to_qt(center))

    @property
    def scene_changed(self) -> Signal:
        """"Сигнал об изменении контента окна."""
        return self.shadow.sceneChanged


    @property
    def is_modified(self) -> bool:
        """Есть ли изменения в окне."""
        return self.shadow.hasModified()

    @property
    def can_undo(self) -> bool:
        """Возможен ли откат на один шаг назад."""
        return self.shadow.canRollBack()
    
    def undo(self):
        """Производит откат на один шаг назад."""
        if self.can_undo:
            self.shadow.rollBack()

    @property
    def can_redo(self) -> bool:
        """Возможен ли откат на один шаг вперед."""
        return self.shadow.canRollForward()
    
    def redo(self):
        """Производит откат на один шаг вперед. При этом возвращается состояние до последней отмены."""
        if self.can_redo:
            self.shadow.rollForward()

class Guidelines:

    def __init__(self):
        raise NotImplementedError

    @classmethod
    def _wrap(cls, shadow):
        result = cls.__new__(cls)
        result.shadow = shadow
        return result


class XGuidelines(Guidelines):

    def append(self, v):
        self.shadow.addXGuideline(v)

    def __getitem__(self, idx):
        return self.shadow.xGuidelines()[idx]

    def __setitem__(self, idx, v):
        self.shadow.setXGuideline(idx, v)
    
    def __len__(self):
        return len(self.shadow.xGuidelines())


class YGuidelines(Guidelines):

    def append(self, v):
        self.shadow.addYGuideline(v)

    def __getitem__(self, idx):
        return self.shadow.yGuidelines()[idx]

    def __setitem__(self, idx, v):
        self.shadow.setYGuideline(idx, v)
    
    def __len__(self):
        return len(self.shadow.yGuidelines())


class ReportView(DrawableView):
    """Окно с планом отчета. 
    Для создания экземпляра необходимо использовать :meth:`axipy.gui.ViewManager.create_reportview` через экземпляр `view_manager`.
    До параметров самого отчета :class:`axipy.render.Report` можно доступиться через свойство :meth:`ReportView.report`

    Пример создания отчета::

        reportview = view_manager.create_reportview()

        # Добавим полигон
        geomItem = GeometryReportItem()
        geomItem.geometry = Polygon((10,10), (10, 100), (100, 100), (10, 10))
        geomItem.style = PolygonStyle(45, Qt.red)
        reportview.report.items.add(geomItem)

        # Установим текущий масштаб
        reportview.scale = 33
    """

    def __init__(self):
        super().__init__()
        self.__xguidelines = None
        self.__yguidelines = None

    @classmethod
    def _wrap_typed(cls, shadow):
        result = cls.__new__(cls)
        result.shadow = shadow
        result.mouse_moved = result.shadow.mouseMoved
        result.__xguidelines = None
        result.__yguidelines = None
        return result

    @property
    def report(self) -> Report:
        """Объект отчета.

        Return:
            Отчет.
        """
        return Report._wrap(ShadowReportView.report(self.shadow))

    @property
    def show_mesh(self) -> bool:
        """Показывать сетку привязки."""
        return self.shadow.showMesh()
    
    @show_mesh.setter
    def show_mesh(self, v):
        self.shadow.setShowMesh(v)

    @property
    def snap_to_mesh(self) -> bool:
        """Включение режима притяжения элементов отчета к узлам сетки."""
        return self.shadow.snapToMesh()
    
    @snap_to_mesh.setter
    def snap_to_mesh(self, v):
        self.shadow.setSnapToMesh(v)

    @property
    def snap_to_guidelines(self) -> bool:
        """Включение режима притяжения элементов отчета к направляющим."""
        return self.shadow.snapToGuidelines()
    
    @snap_to_guidelines.setter
    def snap_to_guidelines(self, v):
        self.shadow.setSnapToGuidelines(v)

    @property
    def x_guidelines(self):
        """Вертикальные направляющие. Значения содержатся в единицах измерения отчета. 
        
        Рассмотрим на примере::

            # Добавление вертикальной направляющей
            reportview.x_guidelines.append(20)
            # Изменение значения направляющей по индексу
            reportview.x_guidelines[0] = 80
            # Удаление всех направляющих.            
            reportview.clear_guidelines()
        """
        if self.__xguidelines is None:
            self.__xguidelines = XGuidelines._wrap(self.shadow)
        return self.__xguidelines

    @property
    def y_guidelines(self):
        """Горизонтальные направляющие. Работа с ними производится по аналогии с вертикальными направляющими."""
        if self.__yguidelines is None:
            self.__yguidelines = YGuidelines._wrap(self.shadow)
        return self.__yguidelines

    def clear_guidelines(self):
        """Удаляет все направляющие."""
        self.shadow.clearGuidelines()

    def clear_selected_guidelines(self):
        """Очищает выбранные направляющие."""
        self.shadow.removeSelectedGuideline()

    def fill_on_pages(self):
        """Наиболее эффективно заполняет пространство отчета масштабированием его элементов."""
        self.shadow.fillOnPages()

    @property
    def scale(self) -> float:
        """Текущий масштаб."""
        return self.shadow.scale()

    @scale.setter
    def scale(self, v):
        self.shadow.setScale(v)

    @property
    def show_borders(self) -> float:
        """Показывать границы страниц."""
        return self.shadow.showLayoutBorders()

    @show_borders.setter
    def show_borders(self, v):
        self.shadow.setShowLayoutBorders(v)

    @property
    def show_ruler(self) -> float:
        """Показывать линейку по краям."""
        return self.shadow.showLayoutRuler()

    @show_ruler.setter
    def show_ruler(self, v):
        self.shadow.setShowLayoutRuler(v)

    @property
    def mesh_size(self) -> float:
        """Размер ячейки сетки."""
        return self.shadow.meshSize()

    @mesh_size.setter
    def mesh_size(self, v):
        self.shadow.setMeshSize(v)

    def mouse_moved(self, x: float, y: float)-> Signal:
        """Сигнал при смещении курсора мыши. Возвращает значения в координатах отчета.
        
        Args:
            x: X координата
            y: Y координата

        Пример::

            reportview.mouse_moved.connect(lambda x,y: print('Coords: {} {}'.format(x, y)))
        """
        return None



class MapView(DrawableView):
    """Окно просмотра карты. Используется для проведения различных манипуляций с картой.
    Для создание экземпляра необходимо использовать :meth:`axipy.gui.ViewManager.create_mapview` через экземпляр `view_manager` (пример см. ниже).

    Note:
        При создании 'MapView' посредством :meth:`axipy.gui.ViewManager.create_mapview` производится клонирование экземпляра карты :class:`axipy.render.Map` 
        и для последующей работы при доступе к данному объекту необходимо использовать свойство :attr:`map`.

    Свойство :attr:`device_rect` определяет размер самого окна карты, а свойство :attr:`scene_rect`
    - прямоугольную область, которая умещается в этом окне в СК карты.

    Преобразование между этими двумя прямоугольниками производится с помощью матриц трансформации 
    :attr:`scene_to_device_transform` и :attr:`device_to_scene_transform`. 

    До параметров самой карты можно доступиться через свойство :attr:`map`.

    Рассмотрим пример создания карты с последующим помещением ее в окно ее просмотра. Далее, попробуем преобразовать
    объект типа полигон из координат окна экрана в координаты СК слоя посредством :meth:`axipy.da.Geometry.affine_transform`.

    ::

        # Откроем таблицу, создадим на ее базе слой и добавим в карту
        table_world = provider_manager.openfile('world.tab')
        world = Layer.create(table_world)
        map = Map([ world ])
        # Для полученной карты создадим окно просмотра
        mapview = view_manager.create_mapview(map)
        # Выведем полученные параметры отображения
        print('Прямоугольник экрана:', mapview.device_rect)
        print('Прямоугольник карты:', mapview.scene_rect)
        # Установим ширину карты и ее центр
        mapview.center = (1000000, 1000000)
        mapview.set_zoom(10e6 )

        >>> Прямоугольник экрана: (0.0 0.0) (300.0 200.0)
        >>> Прямоугольник карты: (-16194966.287183324 -8621185.324024437) (16789976.633236416 8326222.646170927)

    ::

        #Создадим геометрический объект полигон в координатах экрана и преобразуем его в СК карты
        poly_device = Polygon([(100,100), (100,150), (150, 150), (150,100)])
        # Используя матрицу трансформации, преобразуем его в координаты карты.
        poly_scene = poly_device.affine_transform(mapview.device_to_scene_transform)
        # Для контроля выведем полученный полигон в виде WKT
        print('WKT:', poly_scene.wkt)

        >>> WKT: POLYGON ((-5199985.31371008 -147481.338926755, -5199985.31371008 -4384333.3314756, 297505.173026545 -4384333.3314756, 297505.173026545 -147481.338926755, -5199985.31371008 -147481.338926755))
    """

    @property
    def map(self) -> Map:
        """Объект карты.

        Return:
            Карта.
        """
        return Map._wrap(ShadowMapView.map(self.shadow))

    @property
    def coordsystem(self) -> CoordSystem:
        """Система координат карты."""
        return CoordSystem._wrap(self.shadow.coordsystem())

    @coordsystem.setter
    def coordsystem(self, new_coordsystem):
        self.shadow.setCoordSystem(new_coordsystem.shadow)

    @property
    def editable_layer(self) -> Optional[VectorLayer]:
        """Редактируемый слой на карте.

        Return:
            Редактируемый слой. Если не определен, возвращает None.
        """
        return Layer._wrap(self.shadow.editableLayer())

    @editable_layer.setter
    def editable_layer(self, layer):
        self.map.editable_layer = layer

    @property
    def scene_rect(self) -> Rect:
        """Видимая область в координатах карты (в единицах измерения СК).

        Return:
            Прямоугольник в координатах карты.
        """
        return Rect.from_qt(self.shadow.sceneRect())

    @scene_rect.setter
    def scene_rect(self, rect: Rect):
        self.shadow.setSceneRect(rect.to_qt())

    @property
    def device_rect(self) -> Rect:
        """Видимая область в координатах окна (пиксели).

        Return:
            Прямоугольник в координатах окна.
        """
        return Rect.from_qt(self.shadow.deviceRect())

    @property
    def scene_to_device_transform(self) -> QTransform:
        """Объект трансформации из координат карты в координаты окна.

        Return:
            Объект трансформации.
        """
        return self.shadow.sceneToDeviceTransform()

    @property
    def device_to_scene_transform(self) -> QTransform:
        """Объект трансформации из координат окна в координаты карты.

        Return:
            Объект трансформации.
        """
        return self.shadow.deviceToSceneTransform()

    @property
    def editable_layer_changed(self) -> Signal:
        """Сигнал о том, что редактируемый слой сменился."""
        return self.shadow.editableLayerChanged

    def zoom(self, unit: LinearUnit = None) -> float:
        """Ширина окна карты.
        
        Args:
            unit: Единицы измерения. Если не заданы, берутся текущие для карты.
        """
        return self.shadow.zoom(unit.shadow if unit is not None else None)

    def set_zoom(self, v:float, unit: LinearUnit = None):
        """"Задание ширины окна карты.
        
        Args:
            unit: Единицы измерения. Если не заданы, берутся текущие для карты.
        """
        self.shadow.setZoom(v, unit.shadow if unit is not None else None)

    @property
    def center(self) -> Pnt:
        """Центр окна карты."""
        return Pnt.from_qt(self.shadow.center())
    
    @center.setter
    def center(self, p: Pnt):
        self.shadow.setCenter(Pnt._point_value_to_qt(p))

    def show_all(self):
        """Полностью показывает все слои карты."""
        self.shadow.showAll()

    @property
    def coordsystem_changed(self) -> Signal:
        """Сигнал о том, что система координат изменилась.
        
        Пример::

            layer_world = Layer.create(table_world)
            map = Map([ layer_world ])
            mapview = view_manager.create_mapview(map)
            mapview.coordsystem_changed.connect(lambda: print('СК была изменена'))
            csLL = CoordSystem.from_prj("1, 104")
            mapview.coordsystem = csLL

            >>> СК была изменена
        """
        return self.shadow.coordSystemChanged

    @classmethod
    def _wrap_typed(cls, shadow):
        result = cls.__new__(cls)
        result.shadow = shadow
        result.mouse_moved = result.shadow.mouseMoved
        return result

    def mouse_moved(self, x: float, y: float)-> Signal:
        """Сигнал при смещении курсора мыши. Возвращает значения в СК карты.
        
        Args:
            x: X координата
            y: Y координата

        Пример::

            mapview.mouse_moved.connect(lambda x,y: print('Coords: {} {}'.format(x, y)))

            >> Coords: -5732500.0 958500.0
            >> Coords: -5621900.0 847900.0
        """
        return None


class ListLegend:
    """Список добавленных в окно легенд."""

    def __init__(self):
        raise NotImplementedError

    @classmethod
    def _wrap(cls, _shadow: ShadowLegendView):
        result = ListLegend.__new__(ListLegend)
        result.shadow = _shadow
        return result

    def append(self, legend: Legend):
        self.shadow.append(legend.shadow)

    def __getitem__(self, idx: int):
        return Legend._wrap_typed(ShadowLegendView.at(self.shadow, idx))

    def __len__(self):
       return self.shadow.count()

    def remove(self, idx: int):
        self.shadow.remove(idx)

    def __iter__(self):
       return (self.__getitem__(idx) for idx in range(self.__len__()))


class LegendView(View):
    """Легенда для карты.
    Для создания экземпляра необходимо использовать :meth:`axipy.gui.ViewManager.create_legendview` через экземпляр `view_manager`.
    В качестве параметра передается открытое ранее окно с картой::

        legendView = view_manager.create_legendview(map_view)

    Список легенд доступен через свойство :attr:`legends`::

        for legend in legendView.legends:
           print(legend.caption)

    Состав может меняться посредством вызова соответствующих методов свойства :attr:`legends`.

    Добавление легенды для слоя карты::

        legend = Legend(map_view.map.layers[0])
        legend.caption = 'Легенда слоя'
        legendView.legends.append(legend)
        legendView.arrange()

    Доступ к элементу по индексу. Поменяем описание четвертого оформления у первой легенды :class:`axipy.render.Legend` окна::

        legend = legendView.legends[1]
        item = legend.items[3]
        item.title = 'Описание'
        legend.items[3] = item

    Удаление первой легенды из окна::

        legendView.legends.remove(0)


    """

    @classmethod
    def _wrap_typed(cls, _shadow):
        result = cls.__new__(cls)
        result.shadow = _shadow
        result._legends = None
        return result

    @property
    def legends(self) -> ListLegend:
        """Перечень добавленных в окно легенд.
        """
        if self._legends is None:
            self._legends = ListLegend._wrap(self.shadow)
        return self._legends

    def arrange(self):
        """Упорядочивает легенды с целью устранения наложений легенд друг на друга."""
        self.shadow.arrange()