import sys
import inspect, os
import PyQt5.QtGui
from PyQt5.QtGui import QIcon
from PyQt5.QtCore import QPointF
import axioma.gui.extension
from axioma.cs import *
from PyQt5.QtPositioning import *
from axioma.core.geometry import *
from axioma.core.dp import *
from axioma.mapinfo import *
from PyQt5.QtWidgets import QMessageBox

'''
Пример работы с текущими координатами, предоставленными службой ilocation.
Должно присутствовать соответствующее расширение для Qt.
'''

# Базовый класс расширения работы с картой
class MapViewActionExtension(axioma.gui.NativeActionExtension):
    def __init__(self, action_id, ribbon_action_info, parent=None):
        super().__init__(action_id, ribbon_action_info, parent)
        self._mapManager = self.createDependencyGetter(axioma.gui.MapViewManager)
        self._action = None

    def isReady(self):
        if self._mapManager() is not None:
            self._mapManager().mapCountChanged.connect(self.mapCountChanged)
        return True

    def mapCountChanged(self):
        if self._action is not None:
            self._action.setEnabled(self._mapManager().mapViewCount())

    def getIconByName(self, name):
        cwd = os.path.dirname(os.path.abspath(inspect.getfile(inspect.currentframe())))
        return QIcon(os.path.join(cwd, name))

    def mapView(self):
        return self._mapManager().currentMapView()

# Получение текущей позиции (базовый класс)
class PositionActionExtension(MapViewActionExtension):
    def __init__(self, action_id, ribbon_action_info, parent=None):
        super().__init__(action_id, ribbon_action_info, parent)
        self._source = QGeoPositionInfoSource.createSource("ilocation", self);
#          print(QGeoPositionInfoSource.availableSources())
        if self._source is not None:
            self._source.positionUpdated.connect(self.coordUpdated)
            self._source.error.connect(self.errorCoordUpdated)
        else:
           mess = 'Информация по координатам недоступна. {}'.format(QGeoPositionInfoSource.availableSources())
           print(mess)
           QMessageBox.information(axioma.app.mainWindow, "Информация", mess)

    def retrieveCoordinate(self):
        if self._source is not None:
          self._source.requestUpdate(60*1000) # Таймаут 60 секунд

#    def coordUpdated_impl(self, pos):
#      pass

    def coordUpdated(self, pos):
      self.coordUpdated_impl(pos)

    def errorCoordUpdated(self, err):
        mess = 'Ошибка при получении координаты: {}'.format(err)
        print(mess)
        self._action.setIcon(self._icon_position_off)
        QMessageBox.information(axioma.app.mainWindow, "Информация", mess)


# Переход на текущую позицию
class GotoPositionActionExtension(PositionActionExtension):
    def __init__(self, action_id, ribbon_action_info, parent=None):
        super().__init__(action_id, ribbon_action_info, parent)
        self._icon_position_on = self.getIconByName("position_on.png")
        self._icon_position_off = self.getIconByName("position_off.png")

    def createAction_impl(self, parent):
        if self._action is None:
          self._action = PyQt5.QtWidgets.QAction(parent)
          self._action.setText("Поиск на карте")
          self._action.setIcon(self._icon_position_off)
          self._action.setEnabled(False)
          self._action.triggered.connect(self.slot)
        return self._action

    def slot(self):
      self.retrieveCoordinate()

    def coordUpdated_impl(self, pos):
        self._action.setIcon(self._icon_position_on)
        #print(pos.coordinate().toString())
        try:
          mv = self.mapView()
          if mv is not None:
            vp = mv.viewport()
            ct = CoordTransform(CoordSysFactory.defaultCoordSysFactory().LatLongCoordSystem(), vp.coordSystem())
            p_in = QPointF(pos.coordinate().longitude(), pos.coordinate().latitude())
            p_out = ct.forward(p_in)
            vp.setCenter(p_out)
            vp.setZoom(1, CoordSysFactory.defaultCoordSysFactory().unitById(LinearUnit.MIUnit_Kilometer))
            mv.setViewport(vp)
        except Exception as ex:
          print(ex)

# Изменение масштаба
class ZoomActionExtension(MapViewActionExtension):

    def createAction_impl(self, parent):
        if self._action is None:
          self._action = PyQt5.QtWidgets.QAction(parent)
          self._action.setText(self.text())
          self._action.setIcon(self.icon())
          self._action.setEnabled(False)
          self._action.triggered.connect(self.slot)
        return self._action

    def slot(self):
        mv = self.mapView()
        if mv is not None:
          vp = mv.viewport()
          vp.setZoom(vp.zoom() * self.coeff())
          mv.setViewport(vp)

# Увеличить масштаб
class ZoomInActionExtension(ZoomActionExtension):

    def coeff(self):
      return 2/3

    def text(self):
      return "Увеличить масштаб"

    def icon(self):
      return self.getIconByName("zoomIn.png")

# Уменьшить масштаб
class ZoomOutActionExtension(ZoomActionExtension):

    def coeff(self):
      return 3/2

    def text(self):
      return "Уменьшить масштаб"

    def icon(self):
      return self.getIconByName("zoomOut.png")


# Поставить точку. Если есть выделение одного объекта, то неометрия подменяется на новое значение
class AddPointToCurrentPositionExtension(PositionActionExtension):

    def createAction_impl(self, parent):
        if self._action is None:
          self._action = PyQt5.QtWidgets.QAction(parent)
          self._action.setText("Поставить текущую точку")
          self._action.setIcon(self.getIconByName("addPoint.png"))
          self._action.setEnabled(False)
          self._action.triggered.connect(self.slot)
        return self._action

    def slot(self):
      self.retrieveCoordinate()

    def coordUpdated_impl(self, pos):
        mv = self.mapView()
        if mv is not None:
          try:
            geometry = Point(CoordSysFactory.defaultCoordSysFactory().LatLongCoordSystem(), pos.coordinate().longitude(), pos.coordinate().latitude())
            if mv.hasSelection():
              if len(mv.selectionModel().selectedItems()) == 1:
                it = mv.selectionModel().selectedItems()[0]
                it.updateOriginalGeometry(geometry)
                mv.updateItems([it])
                return
            if mv.editableLayer() is not None:
              layer = mv.editableLayer()
            else:
              layer = mv.cosmeticLayer()
            geometry_index = layer.tableSchema().geometryIndex()
            feature = Feature.createFeature(layer.tableSchema())  # Добавляем новую точку
            rendition = MapBasicStyle().styleFromString('Symbol (33,255,14)')
            rendition_index = layer.tableSchema().styleIndex()
            feature.setAttribute(geometry_index, geometry)
            feature.setModified(geometry_index, True)
            feature.setAttribute(rendition_index, rendition)
            feature.setModified(rendition_index, True)
            layer.insertFeatures([feature], pos.timestamp().toString())
          except Exception as ex:
            print(ex)


# Создаем расшширение
def createGeoPoisitionExtensions():
    ribbonExt = axioma.gui.extension.RibbonExtension()
    ribbonExt.addTab("PositionTab", "Текущая позиция")
    ribbonExt.addAction("PositionActionId", "PositionTab", "")
    ribbonExt.addAction("AddPointToCurrentPositionId", "PositionTab", "")
    ribbonExt.addAction("ZoomInActionId", "PositionTab", "")
    ribbonExt.addAction("ZoomOutActionId", "PositionTab", "")
    actionExtPos = GotoPositionActionExtension("PositionActionId", axioma.gui.RibbonActionInfo("", "", ""))
    actionAddPointToCurrentPosition = AddPointToCurrentPositionExtension("AddPointToCurrentPositionId", axioma.gui.RibbonActionInfo("", "", ""))
    actionExtZoomIn = ZoomInActionExtension("ZoomInActionId", axioma.gui.RibbonActionInfo("", "", ""))
    actionExtZoomOut = ZoomOutActionExtension("ZoomOutActionId", axioma.gui.RibbonActionInfo("", "", ""))
    return [ribbonExt, actionExtPos, actionExtZoomIn, actionExtZoomOut, actionAddPointToCurrentPosition]
