
import sys
from PyQt5.QtWidgets import QWidget, QApplication, QDialog, \
    QProgressBar, QLabel, QPushButton, QVBoxLayout
from PyQt5.QtCore import pyqtSignal, QEvent
from PyQt5.QtGui import QCloseEvent
from enum import Enum



class ConcurrentProgressDialog(QDialog):
    """
    Пример пользовательского диалога отображающего прогресс. Спроектирован 
    таким образом, чтобы отображать состояние задачи выполняемой в другом 
    фоновом потоке.

    Например после нажатия Esc или отмены диалог не закрывается, а только
    показывается что началась обработка отменты. Т.к отменяемая задача 
    может завершиться не сразу. 
    """

    class InternalState(Enum):
        Init = 0
        Canceled = 1
        Finished = 2

    canceled = pyqtSignal()

    def __init__(self, parent: QWidget):
        super(ConcurrentProgressDialog, self).__init__(parent)
        self.progressBar = QProgressBar(self)
        self.progressBar.setRange(0, 1)
        self.mainMessage = QLabel("")
        self.cancelButton = QPushButton(self.tr("Отмена"))
        self.layout = QVBoxLayout(self)
        self.layout.addWidget(self.mainMessage)
        self.layout.addWidget(self.progressBar)
        self.layout.addWidget(self.cancelButton)
        self.setLayout(self.layout)
        self.__state = self.InternalState.Init
        self.cancelButton.clicked.connect(self.cancel_impl)

    def set_progress_range(self, min, max):
        self.progressBar.setRange(min, max)

    def set_value(self, value):
        self.progressBar.setValue(value)

    def set_main_message(self, text: str):
        self.mainMessage.setText(text)

    def forceClose(self):
        # После выставления этого флага диалог может закрыться
        self.__state = self.InternalState.Finished
        self.close()
 
    def reject(self):
        """
        Обрабатываем возможное завершение по нажатию  Esc
        """
        if self.__state == self.InternalState.Init:
            self.cancel_impl()
        if self.__state == self.InternalState.Finished:
            super().reject()

    def closeEvent(self, event: QCloseEvent):
        if self.__state == self.InternalState.Finished:
            super().closeEvent(event)
            return
        if self.__state == self.InternalState.Init:
            self.cancel_impl()
            event.ignore()
            return
        event.ignore()


    def cancel_impl(self):
        """
        Закрываем не сразу. Вместо этого ждём пока завершиться операция
        и просто уведомляем о том что пользователь хотел отменить
        """
        self.__state = self.InternalState.Canceled
        self.progressBar.setRange(0,0)
        self.cancelButton.setEnabled(False)
        self.mainMessage.setText(self.tr("Отменяется..."))
        self.canceled.emit()

