from axipy.cpp_cs import ShadowLinearUnit, ShadowAreaUnit, ShadowUnit
from typing import Union, List


__all__ = [
    'Unit',
    'LinearUnit',
    'AreaUnit',
]


class classproperty(property):
    pass


class classinstanceproperty(property):
    pass


class StaticProperty(type):
    def __new__(self, name, bases, props):
        class_properties = {}
        to_remove = {}
        for key, value in props.items():
            if isinstance(value, (classproperty, classinstanceproperty)):
                class_properties[key] = value
                if isinstance(value, classproperty):
                    to_remove[key] = value

        for key in to_remove:
            props.pop(key)

        HoistMeta = type('HoistMeta', (type,), class_properties)
        return HoistMeta(name, bases, props)


class UnitBase(metaclass=StaticProperty):
    def __init__(self):
        raise NotImplementedError
    
    @classproperty
    def all_linear(self) -> 'List[LinearUnit]':
        """Перечень всех линейных единиц измерения."""
        return [self.km, self.m, self.mm, self.cm, self.mi, self.nmi, self.inch, self.ft,
                self.yd, self.survey_ft, self.li, self.ch, self.rd]

    @classproperty
    def km(self) -> 'LinearUnit':
        return LinearUnit._wrap(ShadowLinearUnit.kilometer())

    @classproperty
    def m(self) -> 'LinearUnit':
        return LinearUnit._wrap(ShadowLinearUnit.meter())

    @classproperty
    def mm(self) -> 'LinearUnit':
        return LinearUnit._wrap(ShadowLinearUnit.millimeter())

    @classproperty
    def cm(self) -> 'LinearUnit':
        return LinearUnit._wrap(ShadowLinearUnit.centimeter())

    @classproperty
    def mi(self) -> 'LinearUnit':
        return LinearUnit._wrap(ShadowLinearUnit.mile())

    @classproperty
    def nmi(self) -> 'LinearUnit':
        return LinearUnit._wrap(ShadowLinearUnit.nautical_mile())

    @classproperty
    def inch(self) -> 'LinearUnit':
        return LinearUnit._wrap(ShadowLinearUnit.inch())

    @classproperty
    def ft(self) -> 'LinearUnit':
        return LinearUnit._wrap(ShadowLinearUnit.foot())

    @classproperty
    def yd(self) -> 'LinearUnit':
        return LinearUnit._wrap(ShadowLinearUnit.yard())

    @classproperty
    def survey_ft(self) -> 'LinearUnit':
        return LinearUnit._wrap(ShadowLinearUnit.usFoot())

    @classproperty
    def li(self) -> 'LinearUnit':
        return LinearUnit._wrap(ShadowLinearUnit.link())

    @classproperty
    def ch(self) -> 'LinearUnit':
        return LinearUnit._wrap(ShadowLinearUnit.chain())

    @classproperty
    def rd(self) -> 'LinearUnit':
        return LinearUnit._wrap(ShadowLinearUnit.rod())

    @classproperty
    def all_area(self) -> 'List[AreaUnit]':
        """Перечень всех площадных единиц измерения."""
        return [self.sq_km, self.sq_m, self.sq_mm, self.sq_cm, self.sq_mi, self.sq_nmi, self.sq_inch, self.sq_ft,
                self.sq_yd, self.sq_survey_ft, self.sq_li, self.sq_ch, self.sq_rd]

    @classproperty
    def sq_mm(self) -> 'AreaUnit':
        return AreaUnit._wrap(ShadowAreaUnit.sq_millimeter())

    @classproperty
    def sq_cm(self) -> 'AreaUnit':
        return AreaUnit._wrap(ShadowAreaUnit.sq_centimeter())

    @classproperty
    def sq_m(self) -> 'AreaUnit':
        return AreaUnit._wrap(ShadowAreaUnit.sq_meter())

    @classproperty
    def sq_km(self) -> 'AreaUnit':
        return AreaUnit._wrap(ShadowAreaUnit.sq_kilometer())

    @classproperty
    def sq_mi(self) -> 'AreaUnit':
        return AreaUnit._wrap(ShadowAreaUnit.sq_mile())

    @classproperty
    def sq_nmi(self) -> 'AreaUnit':
        return AreaUnit._wrap(ShadowAreaUnit.sq_nautical_mile())

    @classproperty
    def sq_inch(self) -> 'AreaUnit':
        return AreaUnit._wrap(ShadowAreaUnit.sq_inch())

    @classproperty
    def sq_ft(self) -> 'AreaUnit':
        return AreaUnit._wrap(ShadowAreaUnit.sq_foot())

    @classproperty
    def sq_yd(self) -> 'AreaUnit':
        return AreaUnit._wrap(ShadowAreaUnit.sq_yard())

    @classproperty
    def sq_survey_ft(self) -> 'AreaUnit':
        return AreaUnit._wrap(ShadowAreaUnit.sq_usFoot())

    @classproperty
    def sq_li(self) -> 'AreaUnit':
        return AreaUnit._wrap(ShadowAreaUnit.sq_link())

    @classproperty
    def sq_ch(self) -> 'AreaUnit':
        return AreaUnit._wrap(ShadowAreaUnit.sq_chain())

    @classproperty
    def sq_rd(self) -> 'AreaUnit':
        return AreaUnit._wrap(ShadowAreaUnit.sq_rod())


class Unit(UnitBase):
    """Класс единиц измерения.

    Получение экземпляра единиц измерения осуществляется по атрибуту.

    .. literalinclude:: /../../tests/doc_examples/test_example_unit.py
        :caption: Пример создания
        :pyobject: test_run_example_unit_create
        :lines: 2-
        :dedent: 4

    .. csv-table:: Доступные единицы расстояний
        :header: Атрибут, Тип, Наименование

        km, :class:`LinearUnit`, Километры
        m, :class:`LinearUnit`, Метры
        mm, :class:`LinearUnit`, Миллиметры
        cm, :class:`LinearUnit`, Сантиметры
        mi, :class:`LinearUnit`, Мили
        nmi, :class:`LinearUnit`, Морские мили.
        inch, :class:`LinearUnit`, Дюймы
        ft, :class:`LinearUnit`, Футы
        yd, :class:`LinearUnit`, Ярды
        survey_ft, :class:`LinearUnit`, Топографические футы.
        li, :class:`LinearUnit`, Линки
        ch, :class:`LinearUnit`, Чейны
        rd, :class:`LinearUnit`, Роды


    .. csv-table:: Доступные единицы площадей
        :header: Атрибут, Тип, Наименование

        sq_km, :class:`AreaUnit`, Квадратные километры
        sq_m, :class:`AreaUnit`, Квадратные метры
        sq_mm, :class:`AreaUnit`, Квадратные миллиметры
        sq_cm, :class:`AreaUnit`, Квадратные сантиметры
        sq_mi, :class:`AreaUnit`, Квадратные мили
        sq_nmi, :class:`AreaUnit`, Квадратные морские мили.
        sq_inch, :class:`AreaUnit`, Квадратные дюймы
        sq_ft, :class:`AreaUnit`, Квадратные футы
        sq_yd, :class:`AreaUnit`, Квадратные ярды
        sq_survey_ft, :class:`AreaUnit`, Квадратные топографические футы.
        sq_li, :class:`AreaUnit`, Квадратные линки
        sq_ch, :class:`AreaUnit`, Квадратные чейны
        sq_rd, :class:`AreaUnit`, Квадратные роды
    """

    def __init__(self):
        raise NotImplementedError

    @classmethod
    def _wrap(cls, shadow: ShadowUnit):
        result = cls.__new__(cls)
        result.shadow = shadow
        return result

    @property
    def name(self) -> str:
        """Краткое наименование единиц измерения."""
        return self.shadow.name()

    @property
    def localized_name(self) -> str:
        """Локализованное краткое наименование единиц измерения."""
        return self.shadow.localized_name()

    @property
    def description(self) -> str:
        """Краткое описание."""
        return self.shadow.description()

    @property
    def conversion(self) -> float:
        """Коэффициент преобразования в метры."""
        return self.shadow.conversion()

    def to_unit(self, unit: Union['LinearUnit', 'AreaUnit'], value: float = 1) -> float:
        """Перевод значения в другие единицы измерения.

        Args:
            unit: Единицы измерения, в которые необходимо перевести значение.
            value: Значение для перевода.

        Пример::

            from axipy import *

            print("Linear:", unit.km.to_unit(unit.m, 2))
            print("Area:", unit.sq_km.to_unit(unit.sq_m, 2))

            >>> Linear: 2000.0
            >>> Area: 2000000.0
        """
        if type(self) == type(unit):
            return self.shadow.conversion(unit.shadow) * value
        raise ValueError('Wrong parameter')

    @staticmethod
    def _get_area_unit(u):
        userv = UnitBase
        mapping = {}
        mapping[userv.km.name] = userv.sq_km
        mapping[userv.m.name] = userv.sq_m
        mapping[userv.mm.name] = userv.sq_mm
        mapping[userv.cm.name] = userv.sq_cm
        mapping[userv.mi.name] = userv.sq_mi
        mapping[userv.nmi.name] = userv.sq_nmi
        mapping[userv.inch.name] = userv.sq_inch
        mapping[userv.ft.name] = userv.sq_ft
        mapping[userv.yd.name] = userv.sq_yd
        mapping[userv.survey_ft.name] = userv.sq_survey_ft
        mapping[userv.li.name] = userv.sq_li
        mapping[userv.ch.name] = userv.sq_ch
        mapping[userv.rd.name] = userv.sq_rd
        if isinstance(u, LinearUnit):
            if u.name in mapping:
                return mapping[u.name]
            else:
                raise KeyError
        return u

    @staticmethod
    def _to_unit_fixed(unit_from, unit_to, value=1):
        if isinstance(unit_from, LinearUnit) and isinstance(unit_to, AreaUnit):
            return Unit._get_area_unit(unit_from).to_unit(unit_to, value)
        return unit_from.to_unit(unit_to, value)


class LinearUnit(Unit):
    """Линейные единицы измерения.
    
    Используются для работы с координатами объектов или расстояний.

    Note:
        Получить экземпляр можно через базовый класс :class:`axipy.cs.Unit` по соответствующему атрибуту.

    .. literalinclude:: /../../tests/doc_examples/test_example_unit.py
        :caption: Пример создания
        :pyobject: test_run_example_unit_create
        :lines: 2-
        :dedent: 4
    """

    def __init__(self):
        raise NotImplementedError

    def __eq__(self, other):
        if isinstance(other, LinearUnit):
            return self.name == other.name
        return False

    @classmethod
    def list_all(cls) -> 'List[LinearUnit]':
        """Возвращает перечень всех линейных единиц измерения."""
        return cls.all_linear

class AreaUnit(Unit):
    """Единицы измерения площадей. 

    Note:
        Получить экземпляр можно через базовый класс :class:`axipy.cs.Unit` по соответствующему атрибуту.

    .. literalinclude:: /../../tests/doc_examples/test_example_unit.py
        :caption: Пример создания
        :pyobject: test_run_example_unit_create
        :lines: 2-
        :dedent: 4
    """

    def __init__(self):
        raise NotImplementedError

    def __eq__(self, other):
        if isinstance(other, AreaUnit):
            return self.name == other.name
        return False

    @classmethod
    def list_all(cls) -> 'List[AreaUnit]':
        """Возвращает перечень всех площадных единиц измерения."""
        return cls.all_area
