
from typing import List
from axipy.interface import AxiomaInterface
from axipy.menubar import ToolButton, ActionButton
from .extension import UnloadActionInterface, UnloadQtAction
from axipy.da import DefaultKeys, state_manager
from axipy.menubar import Position

from .convexhull import ConvexHullAction
from .copy_paste_geometry import (
    GeometryStorage,
    register_paste_geometry_observer,
    paste_geometry_observer_key,
    PasteGeometry,
    CopyGeometry)
from .copy_paste_style import (
    StyleStorage,
    CopyStyle,
    PasteStyle,
    paste_style_observer_key,
    register_paste_style_observer)
from .create_outline import CreateOutline
from .mirror import HorizontalMirror, VerticalMirror
from .redirect import RedirectLine
from .voronoi import PolygonsVoronogo


class Plugin:
    def __init__(self, iface: AxiomaInterface):
        self.iface = iface
        self.__unload_actions = []  # type: List[UnloadActionInterface]
        self.load_acitons()

    def load_acitons(self):
        menubar = self.iface.menubar
        self.position = menubar.get_position(
            'Дополнительно', 'Инструменты')  # type: Position

        self.init_convexhull()
        self.init_copy_paste_geometry()
        self.init_copy_paste_style()
        self.init_outline()
        self.init_mirror()
        self.init_redirect()
        self.init_voronoi()

    def unload(self):
        for action in self.__unload_actions:
            action.unload()

    def init_convexhull(self):
        title = self.iface.tr('Оконтурить')
        convexhull = ConvexHullAction(title, self.iface)
        button = ActionButton(
            title=title,
            on_click=lambda: convexhull.on_triggered(),
            enable_on=DefaultKeys.SelectionEditable,
            icon=self.iface.local_file(
                'images',
                '24px',
                'convexhull.png'))
        self.__unload_actions.append(UnloadQtAction(button.action, self.iface))
        self.position.add(button, size=1)

    def init_copy_paste_geometry(self):
        storage = GeometryStorage()
        copy_title = self.iface.tr("Копировать геометрию")
        cad_action = CopyGeometry(copy_title, self.iface, storage)
        copy_button = ActionButton(
            title=copy_title,
            on_click=lambda: cad_action.on_triggered(),
            enable_on=state_manager.Selection,
            icon=self.iface.local_file('images', '24px', 'copy_geometry.png'))
        self.__unload_actions.append(
            UnloadQtAction(
                copy_button.action,
                self.iface))
        self.position.add(copy_button, size=1)

        paste_title = self.iface.tr("Вставить геометрию")
        def tool_factory(): return PasteGeometry(paste_title, self.iface, storage)
        # Регистрируем наблюдатель
        register_paste_geometry_observer(storage)
        paste_button = ToolButton(
            title=paste_title,
            on_click=tool_factory,
            enable_on=paste_geometry_observer_key,
            icon=self.iface.local_file('images', '24px', 'paste_geometry.png'))
        self.__unload_actions.append(
            UnloadQtAction(
                paste_button.action,
                self.iface))
        self.position.add(paste_button, size=1)

    def init_copy_paste_style(self):
        storage = StyleStorage()
        copy_title = self.iface.tr("Копировать стиль")
        copy_action = CopyStyle(copy_title, self.iface, storage)
        copy_button = ActionButton(
            title=copy_title,
            on_click=lambda: copy_action.on_triggered(),
            enable_on=DefaultKeys.Selection,
            icon=self.iface.local_file('images', '24px', 'copy_style.png'))
        self.__unload_actions.append(
            UnloadQtAction(
                copy_button.action,
                self.iface))
        self.position.add(copy_button, size=1)

        paste_title = self.iface.tr("Вставить стиль")
        paste_action = PasteStyle(paste_title, self.iface, storage)
        # Регистрируем наблюдателя
        register_paste_style_observer(storage)
        paste_button = ActionButton(
            title=paste_title,
            on_click=lambda: paste_action.on_triggered(),
            enable_on=paste_style_observer_key,
            icon=self.iface.local_file('images', '24px', 'paste_style.png'))
        self.__unload_actions.append(
            UnloadQtAction(
                paste_button.action,
                self.iface))
        self.position.add(paste_button, size=1)

    def init_outline(self):
        title = self.iface.tr('Создание контура')
        observer_id = DefaultKeys.SelectionEditable
        create_ounline = CreateOutline(title, self.iface, observer_id)
        button = ActionButton(
            title=title,
            on_click=lambda: create_ounline.on_triggered(),
            enable_on=observer_id,
            icon=self.iface.local_file('images', '24px', 'create_outline.png'))
        self.__unload_actions.append(UnloadQtAction(button.action, self.iface))
        self.position.add(button, size=1)

    def init_mirror(self):
        vertical_title = self.iface.tr("Отразить по вертикали")
        vertical_mirror = VerticalMirror(vertical_title, self.iface)
        vertical_button = ActionButton(
            title=vertical_title,
            on_click=lambda: vertical_mirror.on_triggered(),
            enable_on=DefaultKeys.SelectionEditableIsSame,
            icon=self.iface.local_file(
                'images',
                '24px',
                'vertical_mirror.png'))
        self.__unload_actions.append(
            UnloadQtAction(
                vertical_button.action,
                self.iface))
        self.position.add(vertical_button, size=1)

        horizontal_title = self.iface.tr("Отразить по горизонтали")
        horizontal_mirror = HorizontalMirror(horizontal_title, self.iface)
        horizontal_button = ActionButton(
            title=horizontal_title,
            on_click=lambda: horizontal_mirror.on_triggered(),
            enable_on=DefaultKeys.SelectionEditableIsSame,
            icon=self.iface.local_file(
                'images',
                '24px',
                'horizontal_mirror.png'))
        self.__unload_actions.append(
            UnloadQtAction(
                horizontal_button.action,
                self.iface))
        self.position.add(horizontal_button, size=1)

    def init_redirect(self):
        title = self.iface.tr("Изменить направление линии")
        redirect_line = RedirectLine(title, self.iface)
        button = ActionButton(
            title=title,
            on_click=lambda: redirect_line.on_triggered(),
            enable_on=DefaultKeys.SelectionEditableIsSame,
            icon=self.iface.local_file(
                'images',
                '24px',
                'redirect.png'))
        self.__unload_actions.append(UnloadQtAction(button.action, self.iface))
        self.position.add(button, size=1)

    def init_voronoi(self):
        title = self.iface.tr("Полигоны Вороного")
        observer_id = DefaultKeys.SelectionEditable
        polygons_voronogo = PolygonsVoronogo(title, self.iface, observer_id)
        button = ActionButton(
            title=title,
            on_click=lambda: polygons_voronogo.on_triggered(),
            enable_on=observer_id,
            icon=self.iface.local_file('images', '24px', 'voronoi.png'))
        self.__unload_actions.append(UnloadQtAction(button.action, self.iface))
        self.position.add(button, size=1)
