from axipy import *


class CoordinateProcess:
    # True - Координаты представлены в виде списка точек
    # False - Первая координата рассматривается как точка, остальные длина + направление (азимут)
    SimpleCoordinates = True
    IdFeatureField = 'id_feature'
    IdGeometryField = 'id_geometry'
    IdPointField = 'id_point'
    ValueFirstField = 'v1'
    ValueSecondField = 'v2'


class CoordinateReader (CoordinateProcess):
    # Чтение координат из таблицы

    def __get_typed_geometry(self, points, coordsystem: CoordSystem = None):
        if points[0] == points[len(points)-1]:
            return Polygon(points, cs = coordsystem)
        elif len(points) == 2:
            return Line(points[0], points[1], cs = coordsystem)
        return LineString(points, cs = coordsystem)

    def __get_geometry(self, points, coordsystem: CoordSystem = None):
        if len(points) == 1 and len(points[0]):
            yield self.__get_typed_geometry(points[0], coordsystem)
        elif len(points) > 1:
            coll = GeometryCollection( cs = coordsystem )
            for a in points:
                coll.append(self.__get_typed_geometry(a))
            yield coll

    def read(self, features, coordsystem: CoordSystem = None):
        # Предполагается, что записи предварительно отсортированы запросом.
        # Возращает генератор с зачитанными геометриями
        id = -1
        id_g = -1
        points = []
        for f in features:
            if id != f[0]:
                if len(points):
                    yield from self.__get_geometry(points, coordsystem)
                    points = []
                id = f[0]
                id_g = -1
            if id_g != f[1]:
                points.append([])
                id_g = f[1]
            if self.SimpleCoordinates or len(points[f[1]]) == 0:
                points[f[1]].append((f[3], f[4]))
            elif not self.SimpleCoordinates:
                p = Geometry.point_by_azimuth(points[f[1]][ len(points[f[1]]) - 1 ], f[4], f[3])
                points[f[1]].append( p )

        if len(points):
            yield from self.__get_geometry(points, coordsystem)


class CoordinateWriter (CoordinateProcess):
    # Запись геометрий в таблицу

    def __process_geometry(self, g, idx = 0):
        if isinstance(g, Polygon) or isinstance(g, LineString):
            yield from self.__process_points(g.points, idx)
        elif isinstance(g, Line):
            yield from self.__process_points([g.begin, g.end], idx)
        elif isinstance(g, GeometryCollection):
            for idx, gg in enumerate(g):
                yield from self.__process_geometry(gg, idx)
        else:
            print('{} не поддерживается.'.format(g))

    def __process_simple_points(self, points, idx_geom):
        for idx, pnt in enumerate(points):
            yield idx_geom, idx, pnt.x, pnt.y

    def __process_distance_and_azimuth(self, points, idx_geom):
        l = len(points)
        for idx in range(0, l-1):
            if idx == 0:
                yield idx_geom, idx, points[idx].x, points[idx].y
            v = Geometry.distance_by_points(points[idx], points[idx + 1])
            yield idx_geom, idx + 1, v[0], v[1]

    def __process_points(self, points, idx_geom):
        if self.SimpleCoordinates:
            yield from self.__process_simple_points(points, idx_geom)
        else:
            yield from self.__process_distance_and_azimuth(points, idx_geom)

    def write(self, features):
        # Возвращает итератор двух видов в зависимости от установленного SimpleCoordinates:
        # (Номер записи, Номер геометрии, номер точки, координата X, координата Y)
        # (Номер записи, Номер геометрии, номер точки, дистанция, азимут)
        for feature in features:
            if feature.geometry is not None:
                for v in self.__process_geometry(feature.geometry):
                    yield feature.id, v[0], v[1], v[2], v[3]
