from typing import Union
from axipy.da import Style


def tuple_to_int(red: int, green: int, blue: int):
    return (red << 16) + (green << 8) + blue


class MapBasicStyleHelper:
    """Класс для более удобного формирования строки MapBasic или создания объекта стиля."""
    @classmethod
    def create_pen(cls, width: int = 1, pattern: int = 2, color: Union[tuple, int] = 0) -> Style:
        """Создание стиля линии."""
        return cls.create(cls.pen(width, pattern, color))

    @classmethod
    def create_symbol_compat(cls, pattern: int = 2, color: Union[tuple, int] = 0, size: int = 12) -> Style:
        """Создание стиля точки."""
        return cls.create(cls.symbol_compat(pattern, color, size))

    @classmethod
    def create_symbol_font(cls, pattern: int = 2, color: Union[tuple, int] = 0, size: int = 12, font: str = 'Map Symbols') -> Style:
        """Создание стиля точки."""
        return cls.create(cls.symbol_font(pattern, color, size, font))

    @classmethod
    def create_font(cls, fontname: str = "Sans Serif", style: int = 0, size: int = 24, color: Union[tuple, int] = 0, bgcolor: Union[tuple, int, None] = None) -> Style:
        """Создание стиля текста."""
        return cls.create(cls.font(fontname, style, size, color, bgcolor))

    @classmethod
    def create_symbol_pic(cls, bitmap: str, color: Union[tuple, int] = 0, size: int = 12) -> Style:
        """Создание стиля точки."""
        return cls.create(cls.symbol_pic(bitmap, color, size))

    @classmethod
    def create_areal(cls, outline: str, fill) -> Style:
        """Создание стиля площадного объекта."""
        return cls.create(cls.areal(outline, fill))

    @classmethod
    def pen(cls, width: int = 1, pattern: int = 2, color: Union[tuple, int] = 0) -> str:
        """Формирование строки стиля линии."""
        if isinstance(color, tuple):
            color = tuple_to_int(*color)
        return f'Pen ({width}, {pattern}, {color})'

    @classmethod
    def brush(cls, pattern: int = 2, color: Union[tuple, int] = 0, bgcolor: Union[tuple, int] = (255, 255, 255)) -> str:
        """Формирование строки стиля заливки."""
        if isinstance(color, tuple):
            color = tuple_to_int(*color)
        if isinstance(bgcolor, tuple):
            bgcolor = tuple_to_int(*bgcolor)
        return f'Brush ({pattern}, {color}, {bgcolor})'

    @classmethod
    def font(cls, fontname: str = "Sans Serif", style: int = 0, size: int = 24, color: Union[tuple, int] = 0, bgcolor: Union[tuple, int, None] = None) -> str:
        """Формирование строки стиля текстового объекта."""
        if isinstance(color, tuple):
            color = tuple_to_int(*color)
        if isinstance(bgcolor, tuple):
            bgcolor = tuple_to_int(*bgcolor)
        bgcolor = '' if bgcolor is None else f', {bgcolor}'
        return f'Font ("{fontname}", {style}, {size}, {color}{bgcolor})'

    @classmethod
    def symbol_compat(cls, pattern: int = 2, color: Union[tuple, int] = 0, size: int = 12) -> str:
        """Формирование строки символа."""
        if isinstance(color, tuple):
            color = tuple_to_int(*color)
        return f'Symbol ({pattern}, {color}, {size})'

    @classmethod
    def symbol_font(cls, pattern: int = 2, color: Union[tuple, int] = 0, size: int = 12, font: str = 'Map Symbols') -> str:
        """Формирование строки True Type символа."""
        if isinstance(color, tuple):
            color = tuple_to_int(*color)
        return f'Symbol ({pattern}, {color}, {size}, "{font}")'

    @classmethod
    def symbol_pic(cls, bitmap: str, color: Union[tuple, int] = 0, size: int = 12) -> str:
        """Формирование строки растрового символа."""
        if isinstance(color, tuple):
            color = tuple_to_int(*color)
        return f'Symbol ("{bitmap}", {color}, {size})'

    @classmethod
    def areal(cls, outline: str = None, fill: str = None) -> str:
        """Формирование площадного объекта."""
        if outline is None:
            outline = cls.pen()
        if fill is None:
            fill = cls.brush()
        return f'{outline} {fill}'

    @classmethod
    def collection(cls, symbol: str = None, pen: str = None, areal: str = None, font: str = None) -> str:
        """Формирование строки для коллекции."""
        if symbol is None:
            symbol = cls.symbol_compat()
        if pen is None:
            pen = cls.pen()
        if areal is None:
            areal = cls.areal(cls.pen(), cls.brush())
        if font is None:
            font = cls.font()
        pen = pen.replace('Pen', 'Line')
        return f'{symbol} {pen} {areal} {font}'

    @classmethod
    def create(cls, mapbasic_string: str) -> Style:
        return Style.from_mapinfo(mapbasic_string)


#mapbasic = MapBasicStyleHelper()
