"""Модуль приложения.

Данный модуль является основным модулем приложения.

Attributes:
    mainwindow (MainWindow): Готовый экземпляр главного окна ГИС "Аксиома".
"""

from axipy._util import fixup_module_metadata
import types
from axipy.cpp_app import *
from axipy.cpp_core_core import ShadowInformation
from axipy.da import DataManager, DataObject
from axipy.gui import View, gui_instance
from PySide2.QtCore import QObject, QVersionNumber
from PySide2.QtWidgets import QMainWindow, QMdiSubWindow


class Version:
    """Информация о версии ГИС "Аксиома"."""

    @staticmethod
    def number() -> int:
        """Возвращает версию в виде одного числа, в котором каждый сегмент
        располагается в отдельном байте."""
        return ShadowInformation.currentVersion()

    @staticmethod
    def qtFormat() -> QVersionNumber:
        """Возвращает версию в Qt формате."""
        return QVersionNumber(*Version.segments())
        
    @staticmethod
    def segments() -> tuple:
        """Возвращает кортеж чисел - сегменты версии: (major, minor, patch)."""
        v = Version.number()
        return (v & 0xFF0000) >> 16, (v & 0xFF00) >> 8, v & 0xFF
        
    @staticmethod
    def string() -> str:
        """Возвращает версию в виде строки."""
        return ShadowInformation.currentVersionString()


class Notifications:
    """Отправление уведомлений в виде всплывающего окна с его последующей регистрацией в окне уведомлений.
    """

    Information = 0
    Warning = 1
    Critical = 2
    Success = 3

    @staticmethod
    def push(title: str, text: str, type_mesage: int = 0):
        """Отправляет уведомление.

        Args:
            title: Заголовок
            text: Текст сообщения.
            type_mesage: Тип сообщения. В зависимости от типа сообщения в окне уведомлений оно помечается соответствующим цветом.

        Допустимые значения для типа сообщения:
            Information: 
                Информационное сообщение. Устанавливается по умолчанию.
            Warning:
                Предупреждение.
            Critical:
                Критическая ошибка.
            Success:
                Успешное выполнение процесса.

        Пример::

            Notifications.push('Предупреждение', 'Сообщение', Notifications.Warning)
        """
        from axipy.app import ShadowNotifications
        return ShadowNotifications.push(title, text, type_mesage)


class MainWindow(QObject):
    """Главное окно ГИС "Аксиома".

    Note:
        Используйте готовый объект :attr:`axipy.app.mainwindow`.
    """

    def __init__(self):
        super().__init__()
        self.shadow = None
        self._catalog = None

    def service(self):
        if self.shadow is None:
            from axipy.app import mainwindow_instance_hidden
            self.shadow = mainwindow_instance_hidden
        return self.shadow

    @property
    def is_valid(self) -> bool:
        """Корректность состояния главного окна."""
        try:
            self.service()
            return True
        except ImportError:
            return False

    def qt_object(self) -> QMainWindow:
        """Возвращает Qt5 объект окна."""
        return self.service().qt_object()

    @property
    def catalog(self) -> DataManager:
        """Хранилище объектов приложения.

        Это то же хранилище, которое отображается в панели "Открытые данные".

        Note:
            При открытии объектов данных :meth:`axipy.da.ProviderManager.openfile`
            они автоматически попадают в каталог.
        """
        if self._catalog is None:
            self._catalog = DataManager._wrap(gui_instance.shadow.catalog())
        return self._catalog

    def add(self, view: View) -> QMdiSubWindow:
        """Добавляет окно просмотра данных.

        Args:
            view: окно просмотра данных.

        Note:
            При создании окон просмотра данных :meth:`axipy.gui.ViewManager.create_mapview` или :meth:`axipy.gui.ViewManager.create_tableview`
            они автоматически добавляются в главное окно программы.
        """
        return self.service().add(view.shadow)

    def load_workspace(self, fileName: str):
        """Читает рабочее пространство из файла.
        
        Args:
            fileName: Наименование входного файла.
        """
        self.service().open_workspace(fileName)

    def save_workspace(self, fileName: str):
        """Сохраняет рабочее пространство в файл.

        Args:
            fileName: Наименование выходного файла.
        """
        self.service().save_workspace(fileName)


mainwindow = MainWindow()


def _decorate_opener():
    from axipy.da.providers import opener_instance as op
    op.shadow.setDefaultCatalog(mainwindow.catalog.shadow)


def _apply_post(original_method, post_func):
    """Takes method and returns another  method that calls it while calling
    post_func at the end.

    post_func takes the instance itself and result as parameters.
    """
    def _wrapper_method(self, *args, **kwargs):
        results = original_method(*args, **kwargs)
        post_func(self, results)
        return results
    return _wrapper_method


def _decorate_view_manager():
    from axipy.gui import view_manager as service
    def _show_view(instance, view):
        if isinstance(view, View):
            window = mainwindow.add(view)
            window.show()
    service.create_mapview = types.MethodType(_apply_post(service.create_mapview, _show_view), service)
    service.create_tableview = types.MethodType(_apply_post(service.create_tableview, _show_view), service)
    service.create_reportview = types.MethodType(_apply_post(service.create_reportview, _show_view), service)
    service.create_legendview = types.MethodType(_apply_post(service.create_legendview, _show_view), service)


def _hook_core_registered():
    from axipy.da import provider_manager, data_manager
    provider_manager._init_instance()
    data_manager._init_instance(gui_instance._cpp_catalog())


def _hook_mainwindow_registered():
    """Вызывается из С++ после регистрации главного окна."""
    _decorate_opener()
    _decorate_view_manager()


# fix sphinx inheritance and typehints
#fixup_module_metadata(__name__, globals())
#del fixup_module_metadata
