"""Модуль операций с растрами.
"""

from axipy._util import fixup_module_metadata
from PySide2.QtGui import QTransform
from PySide2.QtCore import QPointF
from axipy.cpp_core_dp import RasterFacade as CppRasterFacade
from axipy.cpp_core_dp import PointBinding as CppPointBinding
from axipy.cpp_core_dp import PointBinding
from axipy.cs import CoordSystem
from typing import List, Union, Tuple
from enum import Enum
from typing import NamedTuple


class GCP(NamedTuple):
    """Точка привязки (Ground Control Point).

    Attributes:
        device: Точка на изображении (пиксели).
        scene: Точка на карте.
        label: Идентификатор.
    """
    device: Union[Tuple[float, float], QPointF]
    scene: Union[Tuple[float, float], QPointF]
    label: str = ''

    @staticmethod
    def _wrap(p: CppPointBinding) -> 'GCP':
        return GCP(
            (p.layerPoint.x(), p.layerPoint.y()),
            (p.worldPoint.x(), p.worldPoint.y()),
            p.description
        )

    def _to_cpp(self) -> CppPointBinding:
        def _to_qpointf(p):
            if not isinstance(p, QPointF):
                p = QPointF(*p)
            return p
        return CppPointBinding(
            _to_qpointf(self.device),
            _to_qpointf(self.scene),
            self.label
        )


class Algorithm(Enum):
    """Алгоритм трансформации.

    .. csv-table:: Значения
        :header: Значение, Наименование
        
        POLYNOM1, Аффинитет
        POLYNOM2, Полиномиальный второго порядка
        POLYNOM4, Полиномиальный третьего порядка
        SPLINE, Сплайновый
    """
    POLYNOM1 = 1
    POLYNOM2 = 2
    POLYNOM4 = 3
    SPLINE = 4

class Compression(Enum):
    """Сжатие.
    
    Note:
        Сжатие можно использовать только для файлов формата GeoTIFF.

    .. csv-table:: Значения
        :header: Значение
        
        NONE
        PACKBITS
        LZW
        DEFLATE
    """
    NONE = 1
    PACKBITS = 2
    LZW = 3
    DEFLATE = 4


class Format(Enum):
    """Формат изображения.
    
    .. csv-table:: Значения
        :header: Значение
        
        JPEG
        PNG
        BMP
        GTiff
    """
    JPEG = 1
    PNG = 2
    BMP = 3
    GTiff = 4

    @property
    def ext(self) -> str:
        return ''


class Resample(Enum):
    """Метод интерполяции.

    .. csv-table:: Значения
        :header: Значение, Наименование

        NearestNeighbour, Ближайший
        Bilinear, Билинейный
        Cubic, Кубический
        CubicSpline, Кубический сплайн
        Lanczos, Ланцоша
        Average, Средний
        Mode, Самый встречающийся
        Max, Максимальный
        Min, Минимальный
        Med, Медианный
        Q1, Первый квартиль
        Q3, Третий квартиль
        Sum, Сумма
    """
    NearestNeighbour = 0
    Bilinear = 1
    Cubic = 2
    CubicSpline = 3
    Lanczos = 4
    Average = 5
    Mode = 6
    Max = 8
    Min = 9
    Med = 10
    Q1 = 11
    Q3 = 12
    Sum = 13


def register(filepath: str, bindings: Union[List[GCP], QTransform], coordsystem: CoordSystem):
    """Регистрирует растр.
    
    Добавляет изображению пространственную привязку.

    Args:
        filepath: Файл с изображением.
        bindings: Привязка в виде точек или матрицы преобразования.
        coordsystem: Координатная система.

    .. literalinclude:: /../../tests/doc_examples/test_example_raster_register.py
        :caption: Пример использования
        :pyobject: example_raster_register
        :lines: 2-
        :dedent: 4
    """
    if isinstance(bindings, QTransform):
        CppRasterFacade.registerAffine(filepath, bindings, coordsystem.shadow)
        return
    CppRasterFacade.registerPoints(filepath, list(
        map(lambda p: p._to_cpp(), bindings)), coordsystem.shadow)


def transform(inputfile: str, outputfile: str, points: List[GCP], coordsystem: CoordSystem,
              algorithm: Algorithm = Algorithm.SPLINE, resample: Resample = Resample.NearestNeighbour,
              output_format: Format = Format.GTiff, compression: Compression = Compression.NONE):
    """Трансформирует растр.
    
    Растру, имеющему пространственную привязку, задает новую привязку. На выходе
    получается новый растр.

    Args:
        inputfile: Входной файл с растром.
        outputfile: Выходной файл с растром.
        points: Точки привязки.
        coordsystem: Координатная система.
        algorithm: Алгоритм трансформации.
        resample: Метод интерполяции.
        output_format: Выходной формат.
        compression: Метод сжатия.

    .. literalinclude:: /../../tests/doc_examples/test_example_raster_register.py
        :caption: Пример использования
        :pyobject: example_raster_transform
        :lines: 2-
        :dedent: 4
    """
    CppRasterFacade.transform(inputfile, outputfile, list(map(lambda p: p._to_cpp(), points)),
                              coordsystem.shadow, algorithm.value, resample.value,
                              output_format.name, compression.name)


# fix sphinx inheritance and typehints
fixup_module_metadata(__name__, globals())
del fixup_module_metadata
