
from axipy.app import Notifications
from axipy.cs.CoordSystem import CoordSystem
from axipy.da import Feature, Geometry, state_manager, Style, Table, Point, \
    GeometryType, Schema
from axipy.interface import AxiomaInterface
from PySide2.QtCore import QObject, Signal, Qt, QRect, QPoint, QSize, QPointF
from PySide2.QtGui import QMouseEvent, QCursor, QPen, QKeyEvent, QTransform
from axipy.gui import MapTool, selection_manager, view_manager, MapView
from typing import List, Optional
from axipy.concurrent import task_manager, ProgressSpecification, ProgressGuiFlags, AxipyProgressHandler
from axipy.utl import Rect, Pnt

from .helper import ensure_editable_table, quick_envelope

paste_geometry_observer_key = "PasteGeometryStateManager"


def copy_feature(feature: Feature) -> Feature:
    copy_feature = Feature(dict(feature.items()))
    copy_feature.geometry = feature.geometry.clone()
    copy_feature.style = feature.style
    return copy_feature


class GeometryStorage(QObject):
    """
    Хранилище геометрии для инструментов Копировать/Вставить геометрию
    """

    state_changed = Signal(bool)

    def __init__(self) -> None:
        super().__init__()
        self.__features = None  # type: List[Feature]
        self.__schema = None  # type: Schema

    def is_empty(self) -> bool:
        if isinstance(self.__features, list):
            return len(self.__features) == 0
        return self.__features is None

    @property
    def shalow_features(self) -> List[Feature]:
        """
        Возвращает список из ссылок на реальные объекты Feature.
        Следует использовать без модификаци. Т.к. нет копирования
        работа с этой функции происходит быстрее чем с features().
        """
        return self.__features

    @property
    def features(self) -> List[Feature]:
        """
        Возвращает копию объектов из кеша. Таким образом можно их
        изменять не беспокоясь о сохранности кеша.
        """
        return [copy_feature(f) for f in self.__features]

    @features.setter
    def features(self, features: List[Feature]):
        self.__features = features
        self.state_changed.emit(True)

    @property
    def schema(self) -> Schema:
        """Таблица, являющаяся источником текущего выделения."""
        return self.__schema

    @schema.setter
    def schema(self, schema: Schema):
        assert(issubclass(type(schema), Schema))
        self.__schema = schema


def register_paste_geometry_observer(storage: GeometryStorage):
    """
    Создаём и регистрируем наблюдателя для инструментов Копировать/Вставить геометрию
    """

    editable_observer = state_manager.find(state_manager.Editable)

    def is_enabled():
        return not storage.is_empty() and editable_observer.value()

    paste_geometry_observer = state_manager.find(paste_geometry_observer_key)
    if paste_geometry_observer is None:
        paste_geometry_observer = state_manager.create(
            paste_geometry_observer_key, is_enabled())

    def set_value():
        paste_geometry_observer.setValue(is_enabled())

    editable_observer.changed.connect(set_value)
    storage.state_changed.connect(set_value)
    set_value()
    return paste_geometry_observer


class CopyGeometry:

    def __init__(
            self,
            title: str,
            iface: AxiomaInterface,
            storage: GeometryStorage) -> None:
        self.__storage = storage
        self.title = title
        self.iface = iface

    def __copy_operation(self, handler: AxipyProgressHandler):
        self.__storage.schema = selection_manager.table.schema
        cursor = selection_manager.get_as_cursor()
        features = []  # type: List[Feature]
        handler.set_max_progress(selection_manager.count)
        for f in cursor:
            handler.raise_if_canceled()
            features.append(f)
            handler.add_progress(1)
        handler.prepare_to_write_changes()
        self.__storage.features = features
        self.__notify_about_copied(len(self.__storage.features))

    def on_triggered(self):
        spec = ProgressSpecification(
            description=self.title,
            flags=ProgressGuiFlags.CANCELABLE)
        task_manager.run_and_get(spec, self.__copy_operation)

    def __notify_about_copied(self, size: int):
        type = Notifications.Success
        if size == 0:
            type = Notifications.Warning
        self.iface.notifications.push(self.title, self.iface.tr(
            f"Было скопированно геометрий: {size}"), type)


class PasteGeometry(MapTool):

    def __init__(
            self,
            title: str,
            iface: AxiomaInterface,
            storage: GeometryStorage) -> None:
        super().__init__()
        self.__title = title
        self.__iface = iface
        self.__storage = storage  # type: GeometryStorage
        if (self.__storage.is_empty()):
            raise RuntimeError(
                self.__iface.tr("В буфере нет скопированных геометрий"))
        view_manager.active_changed.connect(self.__active_map_view_changed)
        self.__current_map_view = None  # type: MapView
        self.__envelope = None  # type: Rect
        self.__active_map_view_changed()

    def __active_map_view_changed(self):
        # TODO: use self.view
        map_view = view_manager.active
        if not isinstance(map_view, MapView):
            return
        if self.__current_map_view is not None:
            self.__current_map_view.coordsystem_changed.disconnect(
                self.__update_envelope)
        self.__current_map_view = map_view
        self.__current_map_view.coordsystem_changed.connect(
            self.__update_envelope)
        self.__update_envelope()

    def __update_envelope(self):
        target_cs = self.__current_map_view.coordsystem
        geometries = [f.geometry.reproject(target_cs)
                      for f in self.__storage.shalow_features]
        self.__envelope = quick_envelope(geometries)

    def paintEvent(self, event, painter):
        painter.setPen(QPen(Qt.DashLine))
        device_pnt = self.view.widget.mapFromGlobal(QCursor.pos())
        device_pnt = self.snap_device(device_pnt)
        rect = self.to_device(self.__envelope)  # type: QRect
        rect.moveCenter(device_pnt)
        # Если геометрия одна вставляем её по центроиду
        features = self.__storage.shalow_features
        if len(
                features) == 1 and features[0].geometry.type != GeometryType.Text:
            centroid = features[0].geometry.centroid()
            if centroid.coordsystem != self.__current_map_view.coordsystem:
                centroid = centroid.reproject(
                    self.__current_map_view.coordsystem)
            p2 = self.to_device(Pnt(centroid.x, centroid.y))
            p1 = self.to_device(self.__envelope).center()
            dp = p1 - p2
            rect.translate(dp.x(), dp.y())
        painter.drawRect(rect)

    def mouseMoveEvent(self, event: QMouseEvent) -> Optional[bool]:
        self.redraw()
        return super().mouseMoveEvent(event)

    def mouseReleaseEvent(self, event: QMouseEvent):
        if event.button() == Qt.RightButton:
            return self.BlockEvent
        if event.button() != Qt.LeftButton:
            return self.PassEvent
        spec = ProgressSpecification(
            description=self.__title,
            flags=ProgressGuiFlags.CANCELABLE)
        task_manager.run_and_get(
            spec,
            self.__paste_operation,
            event,
            ensure_editable_table())
        return self.BlockEvent

    def keyPressEvent(self, event: QKeyEvent) -> Optional[bool]:
        if event.key() == Qt.Key_Escape:
            self.reset()
            return
        return super().keyPressEvent(event)

    def deactivate(self):
        # Перерисовываем карту перед отключением инструмента. Т.к. на ней
        # могут остаться графические артефакты от границы вставляемой геометрии
        self.redraw()

    def __paste_operation(
            self,
            handler: AxipyProgressHandler,
        event: QMouseEvent,
            editable_table: Table):
        if editable_table is None:
            Notifications.push(
                self.__title,
                self.__iface.tr("Отсутствует редактируемая таблица"),
                Notifications.Critical)
            return
        # Получаем вектор смещения в текущих координатах карты
        p1 = self.__envelope.center.to_qt()  # type: QPointF
        p2 = self.to_scene(event.pos())
        p2 = self.snap(p2).to_qt()  # type: QPointF
        features = self.__storage.features
        if len(
                features) == 1 and features[0].geometry.type != GeometryType.Text:
            geom = features[0].geometry
            centroid = geom.centroid()  # type: Point
            # Мы хотим получить смещение в координатах карты, поэтому сентроид
            # нужно тоже привести к координатам карты
            if centroid.coordsystem != self.__current_map_view.coordsystem:
                centroid = centroid.reproject(
                    self.__current_map_view.coordsystem)
            dp_to_centroid = p1 - QPointF(centroid.x, centroid.y)
            transform = QTransform.fromTranslate(
                dp_to_centroid.x(), dp_to_centroid.y())
            p2 = transform.map(p2)
        dp = p2 - p1
        handler.set_max_progress(len(features))
        result = []
        for feature in features:
            handler.raise_if_canceled()
            geom = feature.geometry
            # Т.к. координатная система (КС) слоя с которого была скопирована геометрия
            # может отличать от КС карты то для правильного смещения её нужно
            # перепроицировать
            if geom.coordsystem != self.__current_map_view.coordsystem:
                geom = geom.reproject(self.__current_map_view.coordsystem)
            # Копируем геометрию на новое место
            geom = geom.shift(dp.x(), dp.y())
            style = feature.style
            if style is None:
                style = Style.for_geometry(geom)
            f = Feature(geometry=geom, style=style)
            if editable_table.schema == self.__storage.schema:
                # Если вставляем в ту же самую таблицу, то можно сохранить
                # список аттрибутов
                feature.geometry = geom
                feature.style = style
                f = feature
            result.append(f)
            handler.add_progress(1)
        handler.prepare_to_write_changes()
        editable_table.insert(result)
