from axipy import *

from .CoordinateProcess import *
from .LoadPointsDialog import *
from .SavePointsDialog import *

from PySide2.QtGui import QGuiApplication, QCursor, QIcon
from PySide2.QtCore import Qt

from PySide2.QtWidgets import QWidget, QDialog, QMessageBox


class Plugin:

    def __init__(self, iface):
        self.__iface = iface
        self.__tr = self.__iface.tr
        menubar = iface.menubar
        position = menubar.get_position('Дополнительно', 'Инструменты')
        self.loadTool = menubar.create_button(
            self.__tr('Загрузить данные из точек'), icon=self.__get_icon('open.png'), on_click=self.loadPoints,
            enable_on=state_manager.Editable)
        position.add(self.loadTool)
        self.saveTool = menubar.create_button(
            self.__tr('Сохранить в виде точек'), icon=self.__get_icon('save.png'), on_click=self.savePoints,
            enable_on=state_manager.HasTables)
        position.add(self.saveTool)

    def __get_icon(self, filename):
        return QIcon(os.path.join(os.path.dirname(__file__), filename))

    def unload(self):
        self.__iface.menubar.remove(self.saveTool)
        self.__iface.menubar.remove(self.loadTool)

    def savePoints(self):
        # Сохранить геометрию в виде табличных данных
        fd = SavePointsDialog(self.__iface)
        if fd.exec() == QDialog.Accepted:
            definition = {
                'src': fd.result_file_name(),
                'schema': attr.schema(
                    attr.integer(CoordinateProcess.IdFeatureField),
                    attr.integer(CoordinateProcess.IdGeometryField),
                    attr.integer(CoordinateProcess.IdPointField),
                    attr.float(CoordinateProcess.ValueFirstField),
                    attr.float(CoordinateProcess.ValueSecondField)
                )
            }
            table_out = provider_manager.create(definition)
            cw = CoordinateWriter()
            cw.SimpleCoordinates = fd.is_simple_coordinates()
            try:
                QGuiApplication.instance().setOverrideCursor(QCursor(Qt.WaitCursor))
                table_out.insert((Feature({CoordinateProcess.IdFeatureField: v[0],
                                            CoordinateProcess.IdGeometryField: v[1],
                                            CoordinateProcess.IdPointField: v[2],
                                            CoordinateProcess.ValueFirstField: v[3],
                                            CoordinateProcess.ValueSecondField: v[4]}) 
                        for v in cw.write(fd.result_table().items())
                    ))
                table_out.commit()
            finally:
                data_manager.remove(table_out)
                QGuiApplication.instance().restoreOverrideCursor()
                self.__iface.notifications.push(self.__tr('Информация'), self.__tr('Данные сохранены.'))

    def __load_to_edited(self, reader, query):
        edit_layer = None
        if isinstance(view_manager.active, MapView):
            edit_layer = view_manager.active.map.editable_layer
            if edit_layer is not None:
                generator = ( Feature(geometry = g, style = Style.for_geometry(g)) 
                              for g in reader.read(query.items(), view_manager.active.coordsystem) )
                try:
                    QGuiApplication.instance().setOverrideCursor(QCursor(Qt.WaitCursor))
                    edit_layer.data_object.insert(generator)
                finally:
                    QGuiApplication.instance().restoreOverrideCursor()
                    self.__iface.notifications.push(self.__tr('Информация'), self.__tr('Данные загружены.'))
        if edit_layer == None:
            QMessageBox.information(self.__iface.window(), self.__tr('Информация'),  self.__tr('Не определен редактируемый слой'))

        
    def loadPoints(self):
        # Зачитать геометрию из файла
        cr = CoordinateReader()
        fd = LoadPointsDialog(self.__iface)
        if fd.exec() == QDialog.Accepted:
            cr.SimpleCoordinates = fd.is_simple_coordinates()
            query_table_in = provider_manager.query(fd.result_sql_text(), fd.result_table())
            try:
                self.__load_to_edited(cr, query_table_in)
            finally:
                data_manager.remove(query_table_in)

